/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kportal.extension.module;

import java.sql.Date;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Observable;
import java.util.Observer;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.kportal.cache.CacheLoaderManager;
import com.kportal.core.autorisation.Permission;
import com.kportal.core.context.BeanUtil;
import com.kportal.core.context.OverridedContextBean;
import com.kportal.extension.ExtensionManager;
import com.kportal.extension.IExtension;
import com.kportal.extension.om.Module;
import com.univ.utils.ContexteDao;

/**
 * The Class ModuleManager.
 */
public class ModuleManager extends Observable implements Observer {

	public static final String ID_BEAN = "moduleManager";

	private static final Logger LOGGER = LoggerFactory.getLogger(ModuleManager.class);

	private ExtensionManager extensionManager;

	private CacheLoaderManager cacheLoaderManager;

	/** The modules. */
	private Map<String, IModule> modules;

	public void init() {
		cacheLoaderManager.addObserver(this);
	}

	public void refresh() {
		modules = new HashMap<>();
		for (final IExtension extension : extensionManager.getExtensions().values()) {
			final ApplicationContext appCtx = ApplicationContextManager.getApplicationContext(extension.getId());
			// on récupère le nom du bean spring
			for (final Object name : appCtx.getBeansOfType(IModule.class).keySet()) {
				// et on recupere le module dans le contexte spring
				final IModule module = (IModule) appCtx.getBean((String) name);
				module.setId((String) name);
				module.setIdExtension(extension.getId());
				module.setLibelleExtension(extension.getLibelle());
				// set id module et extension sur les permissions
				for (final Permission permission : module.getPermissions()) {
					if (StringUtils.isEmpty(permission.getId())) {
						permission.setId(module.getId());
					}
					permission.setIdModule(module.getId());
					permission.setIdExtension(module.getIdExtension());
				}
				try {
					loadModuleDB(module);
					// on charge temporairement l'ensemble des modules
					modules.put(BeanUtil.getBeanKey((String) name, extension.getId()), module);
				} catch (final Exception e) {
					LOGGER.warn("Erreur de chargement du module id=" + module.getId() + ", extension=" + extension.getId() + " : " + e.getMessage());
				}
			}
		}
		// nettoyage des beans obsolètes
		cleanDB();
		// chargement définitif des modules
		loadModules();
		// notification des observers
		setChanged();
		notifyObservers();
		LOGGER.info("Chargement des modules OK");
	}

	private void loadModules() {
		// liste des modules surchargés à désactiver
		final List<String> aDecharger = new ArrayList<>();
		// liste des modules inactifs à désactiver
		final List<String> aDesactiver = new ArrayList<>();
		// boucle sur l'ensemble des modules chargés
		for (final IModule module : modules.values()) {
			// module actif
			if (module.getEtat() == IModule.ETAT_ACTIF) {
				// si c'est un bean qui surcharge un autre
				if (module instanceof OverridedContextBean) {
					// cle du bean a surcharger
					final String idBeanOver = BeanUtil.getBeanKey(((OverridedContextBean) module).getIdBeanToOverride(), ((OverridedContextBean) module).getIdExtensionToOverride());
					// si le bean a surcharger existe
					if (modules.get(idBeanOver) != null) {
						/*
						ContexteDao ctx = new ContexteDao();
						ModuleOver moduleOver = new ModuleOver();
						try {
							moduleOver.setCtx(ctx);
							// l'id en base est le hashcode de l'id du bean + extension
							moduleOver.setIdBean(new Long(BeanUtil.getBeanKey(module.getId(), module.getIdExtension()).hashCode()));
							moduleOver.retrieve();
						} catch (Exception e) {
							moduleOver.setIdBeanOver(new Long(idBeanOver.hashCode()));
							moduleOver.add();
						} finally {
							ctx.close();
						}*/
						// on garde une trace du module à décharger
						aDecharger.add(idBeanOver);
					}
				}
			}
			// non actif à désactiver
			else {
				aDesactiver.add(BeanUtil.getBeanKey(module.getId(), module.getIdExtension()));
			}
		}
		// mise à jour des modules surchargés en base : actif mais non paramètrable
		try (ContexteDao ctx = new ContexteDao()) {
			for (final String key : aDecharger) {
				final Module moduleDB = new Module();
				try {
					moduleDB.setCtx(ctx);
					moduleDB.setIdModule((long) key.hashCode());
					moduleDB.retrieve();
					moduleDB.setEtat(IModule.ETAT_ACTIF);
					moduleDB.setType(IModule.TYPE_NON_PARAMETRABLE_AFFICHABLE);
					moduleDB.update();
				} catch (final Exception e) {
					LOGGER.error("erreur lors des requetes SQL sur la table module", e);
				}
				// déchargement des modules surchargés
				modules.remove(key);
			}
		}
		// désactivation des modules inactifs
		for (final String key : aDesactiver) {
			modules.remove(key);
		}
	}

	private void cleanDB() {
		final Module moduleDB = new Module();
		try (ContexteDao ctx = new ContexteDao()) {
			moduleDB.setCtx(ctx);
			// on supprime tous les modules qui n'ont pas été restaurés donc obsolètes
			moduleDB.select("", null, "", null, null, IModule.ETAT_A_RESTAURER, null, "");
			while (moduleDB.nextItem()) {
				moduleDB.delete();
			}
		} catch (final Exception e) {
			LOGGER.error("Erreur de nettoyage de la table MODULE", e);
		}
	}

	private void loadModuleDB(final IModule module) throws Exception {
		final Module moduleDB = new Module();
		try (ContexteDao ctx = new ContexteDao()) {
			// l'id en base est le hashcode de l'id du bean + extension
			moduleDB.setIdModule((long) BeanUtil.getBeanKey(module.getId(), module.getIdExtension()).hashCode());
			moduleDB.setCtx(ctx);
			try {
				moduleDB.retrieve();
			} catch (final Exception e) {
				setModuleDB(module, moduleDB);
				moduleDB.setDateCreation(new Date(System.currentTimeMillis()));
				moduleDB.add();
			}
			// restauration ou mise à jour du type (c'est la seule valeur qui est forcee de la conf vers la base)
			if (moduleDB.getEtat() == IModule.ETAT_A_RESTAURER || moduleDB.getType() != module.getType()) {
				if (moduleDB.getEtat() == IModule.ETAT_A_RESTAURER) {
					setModuleDB(module, moduleDB);
				}
				moduleDB.setType(module.getType());
				moduleDB.update();
			}
		}
		// mise à jour de l'état et du libellé par rapport à la conf en base
		module.setEtat(moduleDB.getEtat());
		module.setLibelle(moduleDB.getLibelle());
	}

	private void setModuleDB(final IModule module, final Module moduleDB) {
		moduleDB.setEtat(module.getEtat());
		moduleDB.setType(module.getType());
		moduleDB.setIdBean(module.getId());
		moduleDB.setLibelle(StringUtils.isNotEmpty(module.getLibelle()) ? module.getLibelle() : module.getId());
		moduleDB.setIdExtension((long) module.getIdExtension().hashCode());
		moduleDB.setDateModification(new Date(System.currentTimeMillis()));
	}

	/**
	 * Retourne la liste des modules actifs d'un certain type
	 *
	 * @param type
	 *            type de module souhaité
	 * @return la liste des modules
	 */
	@SuppressWarnings("unchecked")
	public <T extends IModule> Collection<T> getModules(final Class<T> type) {
		final ArrayList<T> res = new ArrayList<>();
		for (final IModule module : modules.values()) {
			if (type.isAssignableFrom(module.getClass())) {
				res.add((T) module);
			}
		}
		return res;
	}

	/**
	 * Retourne la liste des modules actifs
	 *
	 * @return la liste
	 */
	public Map<String, IModule> getModules() {
		return modules;
	}

	public ExtensionManager getExtensionManager() {
		return extensionManager;
	}

	public void setExtensionManager(final ExtensionManager extensionManager) {
		this.extensionManager = extensionManager;
	}

	public void setCacheLoaderManager(final CacheLoaderManager cacheLoaderManager) {
		this.cacheLoaderManager = cacheLoaderManager;
	}

	@Override
	public void update(final Observable o, final Object arg) {
		refresh();
	}
}
