/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kportal.extension.module.plugin.rubrique;

import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.kportal.core.config.MessageHelper;
import com.kportal.core.webapp.WebAppUtil;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.Metatag;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.utils.ContexteUtil;
import com.univ.utils.FicheUnivMgr;
import com.univ.utils.UnivWebFmt;

public class FichePageAccueilRubrique extends DefaultPageAccueilRubrique {

	/**
	 *
	 */
	private static final long serialVersionUID = 6420883169765691077L;

	public static class BeanFichePageAccueil implements BeanPageAccueil {

		private String code = "";

		public void setCode(final String code) {
			this.code = code;
		}

		public void setLangue(final String langue) {
			this.langue = langue;
		}

		public void setObjet(final String objet) {
			this.objet = objet;
		}

		private String langue = "";

		private String objet = "";

		@Override
		public String getUrlRubrique(final String codeRubrique, final String codeLangue, final boolean ampersands) throws Exception {
			return UnivWebFmt.determinerUrlFiche(ContexteUtil.getContexteUniv(), objet, code, langue, ampersands, codeRubrique);
		}

		@Override
		public String getUrlModification(final String codeRubrique, final String langue, final boolean ampersands) throws Exception {
			if (ContexteUtil.getContexteUniv() != null) {
				final AutorisationBean autorisation = ContexteUtil.getContexteUniv().getAutorisation();
				final FicheUniv fiche = getFichePageTete(true);
				if (fiche != null && autorisation.estAutoriseAModifierLaFiche(fiche)) {
					final String esperluette = (ampersands ? "&amp;" : "&");
					return WebAppUtil.SG_PATH + "?EXT=" + ReferentielObjets.getExtension(fiche) + esperluette + "PROC=SAISIE_" + objet.toUpperCase() + esperluette + "ECRAN_LOGIQUE=LISTE" + esperluette + "ACTION=MODIFIER" + esperluette + "ID_FICHE=" + fiche.getIdFiche();
				}
			}
			return "";
		}

		private String getCodePageTete() {
			return (StringUtils.isNotEmpty(code) ? code + ",LANGUE=" + langue + ",TYPE=" + objet : "");
		}

		/**
		 * Retourne une instance de ficheUniv correspondant à la page de tete de la rubrique le parametre all permet de savoir si on doit retrouver juste la fiche si elle est en
		 * ligne, ou bien si on peut rechercher dans tous les etats de fiche.
		 *
		 * @param _ctx
		 *            the _ctx
		 * @param codePageTete
		 *            the code page tete
		 * @param all
		 *            the all
		 *
		 * @return the fiche page tete
		 *
		 * @throws Exception
		 *             the exception
		 */
		public FicheUniv getFichePageTete(final boolean all) throws Exception {
			FicheUniv ficheUniv = null;
			if (StringUtils.isNotEmpty(objet)) {
				ficheUniv = ReferentielObjets.instancierFiche(objet);
				if(ficheUniv != null) {
					ficheUniv.init();
					ficheUniv.setCtx(ContexteUtil.getContexteUniv());
					if (ficheUniv.selectCodeLangueEtat(code, langue, "0003") > 0) {
						ficheUniv.nextItem();
					} else if (all) {
						if (ficheUniv.selectCodeLangueEtat(code, langue, "") > 0) {
							ficheUniv.nextItem();
						}
					}
				}
			}
			return ficheUniv;
		}

		public void parseCodePageTete(final String codePageTete) {
			code = codePageTete;
			langue = "0";
			objet = "pagelibre";
			final int indiceLangue = codePageTete.indexOf(",LANGUE=");
			final int indiceType = codePageTete.indexOf(",TYPE=");
			if (indiceLangue != -1) {
				if (indiceType != -1) {
					langue = codePageTete.substring(indiceLangue + 8, indiceType);
				} else {
					langue = codePageTete.substring(indiceLangue + 8);
				}
				code = codePageTete.substring(0, indiceLangue);
			}
			if (indiceType != -1) {
				objet = codePageTete.substring(indiceType + 6);
			}
		}

		@Override
		public String getLibelleAffichable() {
			String res = ReferentielObjets.getLibelleObjet(objet);
			try {
				final FicheUniv fiche = getFichePageTete(false);
				if (fiche != null) {
					res += " - " + fiche.getLibelleAffichable();
				}
			} catch (final Exception e) {
				// nothing
			}
			return res;
		}

		public String getCode() {
			return code;
		}

		public String getLangue() {
			return langue;
		}

		public String getObjet() {
			return objet;
		}
	}

	@Override
	public void preparerPRINCIPAL(final Map<String, Object> infoBean, final BeanPageAccueil bean) throws Exception {
		final BeanFichePageAccueil b = (BeanFichePageAccueil) bean;
		final FicheUniv ficheUniv = b.getFichePageTete(true);
		final String typeFiche = ReferentielObjets.getLibelleObjet(b.objet);
		if (ficheUniv != null) {
			infoBean.put("LIBELLE_CODE_PAGE_TETE", typeFiche + " : " + ficheUniv.getLibelleAffichable());
		} else {
			infoBean.put("LIBELLE_CODE_PAGE_TETE", "");
		}
		infoBean.put("CODE_PAGE_TETE", b.getCodePageTete());
	}

	@Override
	public void traiterPRINCIPAL(final Map<String, Object> infoBean, final BeanPageAccueil bean) throws Exception {
		final BeanFichePageAccueil b = (BeanFichePageAccueil) bean;
		if (StringUtils.isEmpty(((String) infoBean.get("CODE_PAGE_TETE")))) {
			throw new ErreurApplicative(MessageHelper.getCoreMessage("RUBRIQUE.PAGE_TETE.ERREUR.PAGE_TETE_OBLIGATOIRE"));
		}
		b.parseCodePageTete(((String) infoBean.get("CODE_PAGE_TETE")));
		//Le rattachement rubrique de la page de tête est mis à jour automatique dans la fiche si il est vide
		final FicheUniv ficheUniv = b.getFichePageTete(true);
		if (ficheUniv != null) {
			// ticket 7493 : ficheUniv.code_rubrique et meta.code_rubrique doivent rester solidaires
			if (ficheUniv.getCodeRubrique().length() == 0) {
				ficheUniv.setCodeRubrique((String) infoBean.get("CODE"));
				ficheUniv.update();
				final Metatag meta = FicheUnivMgr.lireMeta(ficheUniv);
				meta.setMetaCodeRubrique((String) infoBean.get("CODE"));
				FicheUnivMgr.enregistrerMeta(meta);
			}
		}
	}
}
