/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kportal.extension.module.plugin.toolbox;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.kportal.tag.interpreteur.InterpreteurTag;
import com.kportal.tag.util.ContexteTagUtil;

/**
 * Classe gérant l'interpretation des tags. Il récupère l'ensemble des beans implémentants l'interface {@link InterpreteurTag} et se charge de remplacer le code du tag par
 * l'interpretation.
 * 
 * @author olivier.camon
 * 
 */
public class PluginTagHelper {

	public static PluginTagManager getTagManager() {
		return ((PluginTagManager) ApplicationContextManager.getCoreContextBean(PluginTagManager.ID_BEAN));
	}

	/**
	 * Interprete l'ensemble des tags défini dans l'application dans le contexte par défaut.
	 * 
	 * @param texte
	 *            le texte devant être parser
	 * @return le texte interpreté
	 * @throws Exception
	 *             il est possible que les tags lèvent des exceptions
	 */
	public static String interpreterTags(String texte) throws Exception {
		return interpreterTags(texte, ContexteTagUtil.DEFAUT);
	}

	/**
	 * 
	 * Parse le texte fourni en paramètre pour voir si il contient un tag Si un ou plusieurs tags sont présent, on remplace le code du tag par son interpertation exple :
	 * "lorem ipsum [foo]bar[/foo]" tag -> foo present, devient "lorem ipsum foo bar"
	 * 
	 * @param texte
	 *            le texte à parser
	 * @param typeInterpretation
	 *            : le contexte d'interpretation : indexation, defaut, mobile...
	 * @return le texte avec les tags interprété
	 * @throws Exception
	 *             lors de l'interpretation, on peut lever des exceptions
	 */
	public static String interpreterTags(String texte, String typeInterpretation) throws Exception {
		Collection<IPluginTag> tagDuType = getTagManager().getTagsParType().get(typeInterpretation);
		if (tagDuType != null && !tagDuType.isEmpty()) {
			for (IPluginTag tagCourant : tagDuType) {
				String texteAInterpreter = tagCourant.extraitTagDuContenu(texte);
				while (StringUtils.isNotEmpty(texteAInterpreter)) {
					String texteInterpreter = tagCourant.interpreteTag(texteAInterpreter);
					texteInterpreter = interpreterTags(texteInterpreter, typeInterpretation);
					texte = StringUtils.replace(texte, texteAInterpreter, texteInterpreter);
					texteAInterpreter = tagCourant.extraitTagDuContenu(texte);
				}
			}
		}
		return texte;
	}

	public static String getReferencesTags(String texte) throws Exception {
		Set<String> referencesParTag = new HashSet<>();
		Collection<IPluginTag> tagsReferences = getTagManager().getTagsParType().get(ContexteTagUtil.DEFAUT);
		for (IPluginTag tagRef : tagsReferences) {
			String texteAInterpreter = tagRef.extraitTagDuContenu(texte);
			while (StringUtils.isNotEmpty(texteAInterpreter)) {
				String texteInterpreter = tagRef.getReferenceTag(texteAInterpreter);
				texteInterpreter = getReferencesTags(texteInterpreter);
				referencesParTag.add(texteInterpreter);
				texte = StringUtils.replace(texte, texteAInterpreter, texteInterpreter);
				texteAInterpreter = tagRef.extraitTagDuContenu(texte);
			}
		}
		StringBuilder references = new StringBuilder("");
		if (!referencesParTag.isEmpty()) {
			for (String ref : referencesParTag) {
				references.append(ref);
			}
		}
		return references.toString();
	}

}
