/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kportal.extension.module.service.impl.beanexport;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.apache.commons.lang3.StringUtils;

import com.jsbsoft.jtf.database.OMContext;
import com.kosmos.usinesite.utils.UASOmHelper;
import com.kosmos.usinesite.utils.UASServicesHelper;
import com.kportal.extension.module.bean.BeanExportMap;
import com.kportal.extension.module.bean.FicheBeanExport;
import com.kportal.extension.module.service.ServiceBeanExport;
import com.univ.objetspartages.bean.MediaBean;
import com.univ.objetspartages.bean.MetatagBean;
import com.univ.objetspartages.bean.RubriquepublicationBean;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.Metatag;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.objetspartages.om.Ressource;
import com.univ.objetspartages.om.Rubriquepublication;
import com.univ.utils.ContexteDao;
import com.univ.utils.sql.RequeteSQL;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;

public class ServiceBeanExportFiche implements ServiceBeanExport<FicheBeanExport<?>> {

	@Override
	public BeanExportMap<FicheBeanExport<?>> getBeansByRubrique(final List<String> codesRubrique, final String idModule, final String pathExport, final String... params)
		throws Exception {
		final BeanExportMap<FicheBeanExport<?>> fichesBeans = new BeanExportMap<>();
		try (ContexteDao ctx = new ContexteDao()) {
			final Metatag metatag = new Metatag();
			metatag.init();
			metatag.setCtx(ctx);
			final RequeteSQL requete = new RequeteSQL();
			final ClauseWhere where = new ClauseWhere();
			where.setPremiereCondition(ConditionHelper.in("T1.META_CODE_RUBRIQUE", codesRubrique));
			if (StringUtils.isNoneEmpty(getParam(params, 0))) {
				where.and(ConditionHelper.egalVarchar("T1.META_CODE_OBJET", params[0]));
			}
			if (StringUtils.isNoneEmpty(getParam(params, 1))) {
				where.and(ConditionHelper.egalVarchar("T1.META_ETAT_OBJET", params[1]));
			}
			requete.where(where);
			metatag.select(requete.formaterRequete());
			while (metatag.nextItem()) {
				processContent(ctx, idModule, metatag, codesRubrique, fichesBeans, pathExport);
			}
		}
		return fichesBeans;
	}

	private String getParam(final String[] params, final int index) {
		if (params.length > index) {
			return params[index];
		}
		return StringUtils.EMPTY;
	}

	/**
	 * export du contenu et d es ressources liées
	 * @param ctx le contexte pour l'accès au données...
	 * @param idModule l'id du module à exporter
	 * @param metatag le meta de la fiche à exporter
	 * @param codesRubriquesExportees l'ensemble des codes de rubriques à exporter
	 * @param fichesBeans les fiches déjà exporter
	 * @param pathExport le chemin de l'export
	 * @throws Exception
	 */
	public void processContent(final OMContext ctx, final String idModule, final Metatag metatag, final List<String> codesRubriquesExportees,
		final Map<String, FicheBeanExport<?>> fichesBeans, final String pathExport) throws Exception {
		final String typeObjet = ReferentielObjets.getNomObjet(metatag.getMetaCodeObjet());
		final String codeFiche = metatag.getMetaCode();
		final String langue = metatag.getMetaLangue();
		if (!fichesBeans.containsKey(typeObjet + "," + codeFiche + "," + langue)) {
			final FicheUniv ficheUniv = ReferentielObjets.instancierFiche(typeObjet);
			if (ficheUniv != null) {
				ficheUniv.setCtx(ctx);
				ficheUniv.init();
				if (ficheUniv.selectCodeLangueEtat(codeFiche, langue, metatag.getMetaEtatObjet()) > 0) {
					ficheUniv.nextItem();
					final FicheBeanExport<?> beanExport = UASServicesHelper.getFicheBeanExport(ficheUniv, UASServicesHelper.getBeanClass(ficheUniv));
					beanExport.setTypeObjet(typeObjet);
					beanExport.setIdModule(idModule);
					//export du metatag pour ne pas perdre certaines propriétés de la fiche : structure visible, forum actif...
					final MetatagBean metatagBean = new MetatagBean();
					UASOmHelper.copyProperties(metatagBean, metatag);
					beanExport.setMetatag(metatagBean);
					// export des rubriques de publication
					final Rubriquepublication rubpub = new Rubriquepublication();
					rubpub.setCtx(ctx);
					rubpub.init();
					final int nbRubpub = rubpub.select(ReferentielObjets.getCodeObjet(typeObjet), ficheUniv.getCode(), ficheUniv.getLangue());
					final ArrayList<RubriquepublicationBean> listeRubpub = new ArrayList<>(nbRubpub);
					while (rubpub.nextItem()) {
						if (codesRubriquesExportees.contains(rubpub.getRubriqueDest())) {
							final RubriquepublicationBean rubriquePublicationBean = new RubriquepublicationBean();
							UASOmHelper.copyProperties(rubriquePublicationBean, rubpub);
							listeRubpub.add(rubriquePublicationBean);
						}
					}
					beanExport.setRubriquespublication(listeRubpub);
					// gestion des fichiers
					final Vector<Ressource> vRessources = Ressource.getListeTotale(ficheUniv);
					// gestion des médias
					final Map<String, MediaBean> medias = UASServicesHelper.getMediasByRessources(vRessources, pathExport);
					medias.putAll(UASServicesHelper.getMedias(ctx, ficheUniv, pathExport, Boolean.FALSE));
					beanExport.setRessources(UASServicesHelper.getRessources(vRessources));
					beanExport.setMedias(medias);
					//ajout du contenu de la fiche dans le tableau des contenus à exporter
					fichesBeans.put(typeObjet + "," + codeFiche + "," + langue, beanExport);
				}
			}
		}
	}
}
