/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kportal.extension.om;

import java.sql.Date;

import org.apache.commons.lang3.StringUtils;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.core.Formateur;
import com.kportal.extension.ExtensionHelper;
import com.kportal.extension.IExtension;
import com.kportal.extension.exception.NoSuchExtensionException;
import com.kportal.extension.sgbd.ExtensionDB;
import com.univ.utils.sql.RequeteSQL;
import com.univ.utils.sql.clause.ClauseLimit;
import com.univ.utils.sql.clause.ClauseOrderBy;
import com.univ.utils.sql.clause.ClauseOrderBy.SensDeTri;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.condition.ConditionList;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.utils.sql.criterespecifique.LimitHelper;
import com.univ.utils.sql.criterespecifique.OrderByHelper;
import com.univ.utils.sql.operande.TypeOperande;

/**
 * Classe representant un objet extension.
 */
public class Extension extends ExtensionDB implements Cloneable {

	/**
	 *
	 */
	private static final long serialVersionUID = -996480676318427303L;

	/**
	 * Initialise l'objet metier.
	 */
	public void init() {
		setIdExtension((long) 0);
		setIdBean("");
		setLibelle("");
		setDateCreation(new Date(0));
		setDateModification(new Date(0));
		setEtat(IExtension.ETAT_ACTIF);
		setType(IExtension.TYPE_PARAMETRABLE);
		setTables("");
	}

	public String getLibelleAffichable(final String key) {
		return ExtensionHelper.getMessage(getIdBean(), key);
	}

	/**
	 * Selection d'une instance de l'objet Extension a partir de l'ensemble des criteres.
	 */
	public int select(final String idBean, final String libelle, final Date dateCreation, final Date dateModification, final Integer etat, final Integer[] type,
		final String nombre, final String order) throws Exception {
		final RequeteSQL requeteSelect = new RequeteSQL();
		final ClauseWhere where = new ClauseWhere();
		// Traitement du parametre idBean
		if (StringUtils.isNotEmpty(idBean)) {
			where.and(ConditionHelper.egalVarchar("T1.ID_BEAN", idBean));
		}
		// Traitement du parametre libelle
		if (StringUtils.isNotEmpty(libelle)) {
			where.and(ConditionHelper.likePourValeursMultiple("T1.LIBELLE", libelle));
		}
		// Traitement du parametre dateCreation
		if (Formateur.estSaisie(dateCreation)) {
			where.and(ConditionHelper.critereDateDebut("T1.DATE_CREATION", dateCreation));
		}
		// Traitement du parametre dateModification
		if (Formateur.estSaisie(dateModification)) {
			where.and(ConditionHelper.critereDateDebut("T1.DATE_MODIFICATION", dateModification));
		}
		// Traitement du parametre etat
		if (etat != null) {
			where.and(ConditionHelper.egal("T1.ETAT", etat, TypeOperande.INTEGER));
		}
		// Traitement du parametre type
		if (type != null) {
			final ConditionList conditionsSurTypes = new ConditionList();
			for (final Integer integer : type) {
				conditionsSurTypes.or(ConditionHelper.egal("T1.TYPE", integer, TypeOperande.INTEGER));
			}
			where.and(conditionsSurTypes);
		}
		requeteSelect.where(where);
		ClauseOrderBy orderBy = new ClauseOrderBy();
		if (StringUtils.isNotEmpty(order)) {
			orderBy = OrderByHelper.reconstruireClauseOrderBy(order);
		} else {
			orderBy.orderBy("T1.LIBELLE", SensDeTri.ASC);
		}
		requeteSelect.orderBy(orderBy);
		final ClauseLimit limite = LimitHelper.ajouterCriteresLimitesEtOptimisation(ctx, nombre);
		requeteSelect.limit(limite);
		return select(requeteSelect.formaterRequete());
	}

	@Override
	public Extension clone() throws CloneNotSupportedException {
		return (Extension) super.clone();
	}

	public Extension setConfig() throws Exception {
		final IExtension extension = getExtension();
		if (extension == null) {
			throw new NoSuchExtensionException(getIdBean());
		}
		setAuteur(extension.getAuteur());
		setLibelle(extension.getLibelle());
		setDescription(extension.getDescription());
		if (StringUtils.isNotBlank(extension.getLogo())) {
			setLogo(extension.getRelativePath() + extension.getLogo());
		}
		setUrl(extension.getUrl());
		setCoreVersion(extension.getCoreVersion());
		return this;
	}

	public IExtension getExtension() {
		return (IExtension) ApplicationContextManager.getBean(getIdBean(), getIdBean());
	}
}
