/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kportal.extension.processus;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.core.InfoBean;
import com.jsbsoft.jtf.database.ProcessusBean;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.jsbsoft.jtf.exception.ErreurAsyncException;
import com.jsbsoft.jtf.session.SessionUtilisateur;
import com.kportal.core.cluster.ClusterHelper;
import com.kportal.core.config.MessageHelper;
import com.kportal.core.webapp.WebAppUtil;
import com.kportal.extension.ExtensionHelper;
import com.kportal.extension.IExtension;
import com.kportal.extension.exception.NoSuchExtensionException;
import com.kportal.extension.module.IModule;
import com.kportal.extension.om.Extension;
import com.kportal.extension.om.Module;
import com.kportal.extension.service.EtatModuleHelper;
import com.kportal.extension.service.ServiceExtension;
import com.univ.objetspartages.om.AutorisationBean;

public class GestionExtension extends ProcessusBean {

	private static final Logger LOGGER = LoggerFactory.getLogger(GestionExtension.class);

	/** The Constant ECRAN_LISTE. */
	private static final String ECRAN_LISTE = "LISTE";

	/** The autorisations. */
	private AutorisationBean autorisations = null;

	public GestionExtension(final InfoBean infoBean) {
		super(infoBean);
	}

	@Override
	protected boolean traiterAction() throws Exception {
		autorisations = (AutorisationBean) getGp().getSessionUtilisateur().getInfos().get(SessionUtilisateur.AUTORISATIONS);
		if (autorisations == null) {
			infoBean.setEcranRedirection(WebAppUtil.CONNEXION_BO);
			infoBean.setEcranLogique("LOGIN");
		} else {
			try {
				ecranLogique = infoBean.getEcranLogique();
				action = infoBean.getActionUtilisateur();
				if (action.equals("LISTE")) {
					preparerListe();
				} else if (action.equals("ACTIVER_EXTENSION")) {
					traiterActivation();
				} else if (action.equals("SUPPRIMER_EXTENSION")) {
					traiterSuppression();
				} else if (action.equals("ACTIVER_MODULE")) {
					traiterActivation();
				} else if (action.equals("RECHARGER")) {
					traiterRechargement(false);
				} else if (action.equals("RESTAURER")) {
					traiterRechargement(true);
				}
				//placer l'état dans le composant d'infoBean
				infoBean.setEcranLogique(ecranLogique);
			} catch (final ErreurAsyncException e) {
				throw e;
			} catch (final Exception e) {
				LOGGER.error(e.getMessage(), e);
				infoBean.addMessageErreur(e.toString());
			}
		}
		// On continue si on n'est pas à la FIN !!!
		return (etat == FIN);
	}

	/*
	 * gere la supression directe d'une extension
	 */
	private void traiterSuppression() throws Exception {
		try {
			final String idExtension = infoBean.getString("ID_EXTENSION");
			if (StringUtils.isNotEmpty(idExtension)) {
				final Extension extension = new Extension();
				extension.init();
				extension.setCtx(this);
				extension.setIdExtension(new Long(idExtension));
				extension.retrieve();
				if (extension.getType() != IExtension.TYPE_PARAMETRABLE) {
					throw new ErreurApplicative(MessageHelper.getCoreMessage("BO_OPERATION_INTERDITE"));
				}
				ExtensionHelper.getExtensionManager().removeExtension(extension.getIdBean());
				reset();
			}
			preparerListe();
		} catch (final Exception e) {
			throw new ErreurAsyncException("Une erreur s'est produite", e);
		}
	}

	/*
	 * gere l'activation ou la desactivation directe d'un module ou d'une extension
	 */
	private void traiterActivation() throws Exception {
		try {
			final String idExtension = infoBean.getString("ID_EXTENSION");
			final String idModule = infoBean.getString("ID_MODULE");
			// traitement de l'extension
			if (StringUtils.isNotEmpty(idExtension)) {
				final Extension extension = new Extension();
				extension.init();
				extension.setCtx(this);
				extension.setIdExtension(new Long(idExtension));
				extension.retrieve();
				if (extension.getType() != IExtension.TYPE_PARAMETRABLE) {
					throw new ErreurApplicative(MessageHelper.getCoreMessage("BO_OPERATION_INTERDITE"));
				}
				if (extension.getEtat() == IExtension.ETAT_ACTIF) {
					extension.setEtat(IExtension.ETAT_NON_ACTIF);
				} else if (extension.getEtat() == IExtension.ETAT_NON_ACTIF) {
					extension.setEtat(IExtension.ETAT_ACTIF);
				}
				extension.update();
				EtatModuleHelper.setEtatDB(extension.getEtat(), extension.getIdExtension());
				refresh();
			}
			// traitement du module
			else if (StringUtils.isNotEmpty(idModule)) {
				final Module module = new Module();
				module.init();
				module.setCtx(this);
				module.setIdModule(new Long(idModule));
				module.retrieve();
				if (module.getType() != IModule.TYPE_PARAMETRABLE) {
					throw new ErreurApplicative(MessageHelper.getCoreMessage("BO_OPERATION_INTERDITE"));
				}
				if (module.getEtat() == IModule.ETAT_ACTIF) {
					module.setEtat(IModule.ETAT_NON_ACTIF);
				} else if (module.getEtat() == IModule.ETAT_NON_ACTIF) {
					module.setEtat(IModule.ETAT_ACTIF);
				}
				module.update();
				refresh();
			}
			preparerListe();
		} catch (final Exception e) {
			throw new ErreurAsyncException("Une erreur s'est produite", e);
		}
	}

	private void traiterRechargement(final boolean reset) throws ErreurAsyncException {
		// mise à jour de l'état des modules et extensions
		if (reset) {
			ServiceExtension.setEtatDB(IExtension.ETAT_A_RESTAURER);
			EtatModuleHelper.setEtatDB(IModule.ETAT_A_RESTAURER, null);
		}
		reset();
		try {
			preparerListe();
		} catch (final Exception e) {
			throw new ErreurAsyncException("Une erreur s'est produite", e);
		}
	}

	private void preparerListe() throws Exception {
		final Extension extension = new Extension();
		extension.init();
		extension.setCtx(this);
		final Map<Extension, List<Module>> lstExtension = new LinkedHashMap<>();
		extension.select("", "", null, null, null, new Integer[] { IExtension.TYPE_NON_PARAMETRABLE_AFFICHABLE, IExtension.TYPE_PARAMETRABLE }, "", "T1.DATE_CREATION,T1.LIBELLE");
		while (extension.nextItem()) {
			final List<Module> lstModule = new ArrayList<>();
			final Extension cloneExtention = extension.clone();
			try {
				cloneExtention.setConfig();
				if (cloneExtention.getEtat() == IExtension.ETAT_ACTIF) {
					final Module module = new Module();
					module.init();
					module.setCtx(this);
					module.selectByExtension(cloneExtention.getIdExtension(), null, new Integer[] { IModule.TYPE_NON_PARAMETRABLE_AFFICHABLE, IModule.TYPE_PARAMETRABLE });
					while (module.nextItem()) {
						try {
							lstModule.add(module.clone().setConfig(cloneExtention));
						} catch (final NoSuchBeanDefinitionException e) {
							// on supprime le bean si il n'existe plus
							module.delete();
						}
					}
				}
				lstExtension.put(cloneExtention, lstModule);
			} catch (final NoSuchExtensionException e) {
				// on supprime le bean si il n'existe plus
				extension.delete();
			}
		}
		infoBean.set("LISTE_EXTENSIONS", lstExtension);
		ecranLogique = ECRAN_LISTE;
	}

	private void reset() {
		// rechargement du singleton ApplicationContextManager
		ClusterHelper.refresh(ApplicationContextManager.getInstance(), null);
	}

	private void refresh() {
		// rechargement du singleton ExtensionManager
		ClusterHelper.refresh(ExtensionHelper.getExtensionManager(), null);
	}
}
