/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kportal.ihm.utils;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.InfoBean;
import com.jsbsoft.jtf.exception.ErreurDonneeNonTrouve;
import com.kosmos.usinesite.utils.InfosSiteHelper;
import com.kportal.core.config.MessageHelper;
import com.kportal.extension.module.composant.Menu;
import com.kportal.extension.module.plugin.rubrique.BeanPageAccueil;
import com.kportal.extension.module.plugin.rubrique.FichePageAccueilRubrique.BeanFichePageAccueil;
import com.kportal.extension.module.plugin.rubrique.PageAccueilRubriqueManager;
import com.kportal.ihm.service.MenuFrontFactory;
import com.univ.multisites.InfosSite;
import com.univ.multisites.Site;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.InfosRubriques;
import com.univ.objetspartages.om.LigneContenu;
import com.univ.objetspartages.om.Media;
import com.univ.objetspartages.om.ParagrapheBean;
import com.univ.objetspartages.om.ParagrapheContenuHelper;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.objetspartages.om.Ressource;
import com.univ.objetspartages.om.Rubrique;
import com.univ.utils.ContexteUniv;
import com.univ.utils.ContexteUtil;
import com.univ.utils.URLResolver;
import com.univ.utils.UnivWebFmt;

public class FrontUtil {

	private static final Logger LOG = LoggerFactory.getLogger(FrontUtil.class);

	public static final String NOM_PROPRIETE_LOGO = "LOGO";

	public static final String NOM_PROPRIETE_BASELINE = "BASELINE";

	public static final String NOM_PROPRIETE_COULEUR_PRINCIPALE = "COULEUR_PRINCIPAL";

	public static final String NOM_PROPRIETE_COULEUR_SECONDAIRE = "COULEUR_SECONDAIRE";

	public static final String NOM_PROPRIETE_MENU_PRINCIPAL = "MENU_PRINCIPAL";

	public static final String NOM_PROPRIETE_MENU_PIED_PAGE = "MENU_PIED_PAGE";

	public static final String NOM_PROPRIETE_NIVEAU_MENU_SECONDAIRE = "NIVEAU_MENU_SECONDAIRE";

	public static final String CATEGORIE_NAVIGATION = "NAV";

	public static final String CATEGORIE_LANGUE = "LANGUE";

	public static final String CATEGORIE_ACCES_DIRECT = "ACCES";

	public static final String CATEGORIE_PIED_PAGE = "PIEDPAGE";

	private static final String CATEGORIE_RESEAUX_SOCIAUX = "RX_SOC";

	private static final int NIVEAU_MENU_SECONDAIRE = 2;

	public static final String RGBA_TEMPLATE = "rgba(%s,%s,%s,%s)";

	public static final String COLOR_PREFIX = "#";

	private static final String HSL_TEMPLATE = "hsl(%s,%s%%,%s%%)";

	/**
	 * On gère 2 cas particuliers d'affichage : Cas de la page d'accueil du site : la fiche est page de tête de rubriqueSite Cas de la page d'accueil de rubrique : Si la fiche est
	 * page de tête de rubrique de type langue alors c'est une page d'accueil de site
	 *
	 * @return true si c'est l'accueil de site...
	 */
	public static boolean isAccueilSite(final FicheUniv ficheUniv) {
		boolean isAccueil = Boolean.FALSE;
		if (ficheUniv != null && StringUtils.isNotBlank(ficheUniv.getCodeRubrique())) {
			final String codeRubriquePageCourante = ficheUniv.getCodeRubrique();
			final InfosSite siteCourant = getSiteCodeRubrique(codeRubriquePageCourante);
			if (siteCourant != null) {
				isAccueil = codeRubriquePageCourante.equals(siteCourant.getCodeRubrique());
				if (!isAccueil) {
					isAccueil = isAccueilRubriqueLangue(codeRubriquePageCourante, siteCourant.getCodeRubrique());
				}
			}
		}
		return isAccueil;
	}

	private static boolean isAccueilRubriqueLangue(final String codeRubriquePageCourante, final String codeRubriqueSiteCourant) {
		Boolean isAccueil = Boolean.FALSE;
		final List<InfosRubriques> rubriquesLangues = getListeSousRubriquesDepuisCategorie(CATEGORIE_LANGUE, codeRubriqueSiteCourant);
		int i = 0;
		while (i < rubriquesLangues.size() && !isAccueil) {
			final InfosRubriques rubriqueLangue = rubriquesLangues.get(i);
			isAccueil = codeRubriquePageCourante.equals(rubriqueLangue.getCode());
			i++;
		}
		return isAccueil;
	}

	private static InfosSite getSiteCodeRubrique(final String codeRubriquePageCourante) {
		InfosSite siteCourant = Site.determinerSiteRubrique(codeRubriquePageCourante);
		if (siteCourant == null) {
			siteCourant = Site.getSitePrincipal();
		}
		return siteCourant;
	}

	/**
	 * Cette fonction determine si la fiche en court de consultation est la page d'accueil d'une rubrique.
	 *
	 * @param ficheUniv
	 *            la fiche à vérifier
	 * @return TRUE s'il sagit d'une page d'accueil dans le cas inverse vous pouvez devinez la valeur qui sera retourné:p.
	 */
	public static boolean isFicheAccueilRubrique(final FicheUniv ficheUniv) {
		if (ficheUniv != null && !isAccueilSite(ficheUniv)) {
			return isFicheAccueilRubrique(ficheUniv, Rubrique.renvoyerItemRubrique(ficheUniv.getCodeRubrique()));
		}
		return Boolean.FALSE;
	}

	/**
	 * Cette fonction determine si la fiche en court de consultation est la page d'accueil d'une rubrique.
	 *
	 * @param ficheUniv
	 *            Fiche en court de consultation
	 * @param rubrique
	 *            La rubrique pour laquelle on veut savoir si la fiche est la page de tête.
	 * @return TRUE s'il sagit d'une page d'accueil sinon FALSE.
	 */
	public static boolean isFicheAccueilRubrique(final FicheUniv ficheUniv, final InfosRubriques rubrique) {
		Boolean isAccueilRubrique = Boolean.FALSE;
		if (ficheUniv != null && rubrique != null) {
			final BeanPageAccueil beanAccueil = PageAccueilRubriqueManager.getInstance().getBeanPageAccueil(rubrique);
			if (beanAccueil != null && beanAccueil instanceof BeanFichePageAccueil) {
				isAccueilRubrique = ficheUniv.getCode().equals(((BeanFichePageAccueil) beanAccueil).getCode()) && ficheUniv.getLangue().equals(
					((BeanFichePageAccueil) beanAccueil).getLangue()) && ReferentielObjets.getNomObjet(ficheUniv).equals(((BeanFichePageAccueil) beanAccueil).getObjet());
			}
		}
		return isAccueilRubrique;
	}

	/**
	 * Permet de savoir si on est en saisie front ou pas. A l'heure actuelle, on doit vérifier dans les paramètres SAISIE_FRONT & PROC de la requête
	 *
	 * @param paramSaisieFront
	 *            si il a pour valeur "true" on est en saisie front
	 * @param paramProcessus
	 *            si le nom du processus fini par _FRONT on est en saisie fron
	 * @return vrai ssi SAISIE_FRONT=true et que le nom du processus termine par _FRONT...
	 */
	public static boolean isSaisieFront(final String paramSaisieFront, final String paramProcessus) {
		return "true".equalsIgnoreCase(paramSaisieFront) || StringUtils.endsWith(paramProcessus, "_FRONT");
	}

	/**
	 * Calcule le menu principal de l'application
	 *
	 * @return un bean contenant le menu ou null si il n'est pas retrouvé.
	 * @deprecated utiliser {@link FrontUtil#getMenuPrincipalParCategorie()}
	 */
	@Deprecated
	public static Menu getMenuPrincipal() {
		return MenuFrontFactory.getServiceMenuFront().getMenuSiteCourant(NOM_PROPRIETE_MENU_PRINCIPAL);
	}

	/**
	 * Calcule le menu principal de l'application
	 *
	 * @return un bean contenant le menu ou null si il n'est pas retrouvé.
	 * @deprecated utiliser {@link FrontUtil#getMenuPiedDePageParCategorie()}
	 */
	@Deprecated
	public static Menu getMenuPiedDePage() {
		return MenuFrontFactory.getServiceMenuFront().getMenuSiteCourant(NOM_PROPRIETE_MENU_PIED_PAGE);
	}

	/**
	 * Calcule le menu de langue du site
	 *
	 * @return un bean contenant le menu ou null si il n'est pas retrouvé.
	 * @deprecated utiliser {@link com.kportal.ihm.utils.FrontUtil#getMenuLangueParCategorie()}
	 */
	@Deprecated
	public static Menu getMenuLangue() {
		return MenuFrontFactory.getServiceMenuFront().getMenuLangue(ContexteUtil.getContexteUniv().getInfosSite(), NOM_PROPRIETE_MENU_PRINCIPAL);
	}

	/**
	 * Calcule le menu secondaire de la page courante de l'application
	 *
	 * @return un bean contenant le menu ou null si il n'est pas retrouvé.
	 * @deprecated utiliser {@link FrontUtil#getMenuSecondairePageCourante()}
	 */
	@Deprecated
	public static Menu getMenuSecondaire() {
		final ContexteUniv ctx = ContexteUtil.getContexteUniv();
		final String codeRubriquePageCourante = ctx.getCodeRubriquePageCourante();
		InfosRubriques rubriqueCourante = Rubrique.renvoyerItemRubrique(codeRubriquePageCourante);
		final String proprieteNiveauMenuSecondaire = ctx.getInfosSite().getProprieteComplementaireString(NOM_PROPRIETE_NIVEAU_MENU_SECONDAIRE);
		String codeRubriqueSecondaire = StringUtils.EMPTY;
		if (StringUtils.isNotBlank(proprieteNiveauMenuSecondaire)) {
			final int niveauMenuSecondaire = Integer.valueOf(proprieteNiveauMenuSecondaire);
			while (rubriqueCourante.getNiveau() > niveauMenuSecondaire) {
				rubriqueCourante = Rubrique.renvoyerItemRubrique(rubriqueCourante.getCodeRubriqueMere());
			}
			if (rubriqueCourante.getNiveau() == niveauMenuSecondaire) {
				codeRubriqueSecondaire = rubriqueCourante.getCode();
			}
		}
		return MenuFrontFactory.getServiceMenuFront().getMenuParCodeRubrique(codeRubriqueSecondaire);
	}

	/**
	 * Vérifie si le menu fourni en paramètre vient de la rubrique courante ou non.
	 *
	 * @param menuAVerifier menu dont on souhaite savoir si il est bien le menu courant
	 * @param codeRubriqueCourante le code de la rubrique de la page
	 * @return True si la rubrique du menu contient la rubrique de codeRubriqueCourante .
	 */
	public static boolean isMenuCourant(final Menu menuAVerifier, final String codeRubriqueCourante) {
		boolean isSousRubrique = Boolean.FALSE;
		if (StringUtils.isNotBlank(menuAVerifier.getCodeRubriqueOrigine()) && StringUtils.isNotBlank(codeRubriqueCourante)) {
			final InfosRubriques rubriqueParente = Rubrique.renvoyerItemRubrique(menuAVerifier.getCodeRubriqueOrigine());
			final InfosRubriques sousRubrique = Rubrique.renvoyerItemRubrique(codeRubriqueCourante);
			isSousRubrique = rubriqueParente.contains(sousRubrique);
		}
		return isSousRubrique;
	}

	/**
	 * Vérifie si le menu fourni en paramètre contient un enfant actif ou non (si un des enfants est le menu courant).
	 *
	 * @param menuAVerifier le menu dont on souhaite savoir si il est le menu courant. Ne peut être null
	 * @return True si le menu contient un enfant actif.
	 */
	public static boolean hasActiveChild(final Menu menuAVerifier) {
		for (final Menu elementSousMenu : menuAVerifier.getSousMenu()) {
			if (elementSousMenu.isMenuCourant()) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Récupère à partir d'une ficheUniv sur le champ "contenu" les lignes & paragraphe défini dans ce champ. Si la fiche ne contient pas de valeur, on renvoie une liste vide
	 *
	 * @param fiche
	 *            la fiche à analyser. Si elle est null on retourne une liste vide
	 * @return la liste de {@link LigneContenu} ou rien si rien n'est défini
	 */
	public static Collection<LigneContenu> getLignesFicheCourante(final FicheUniv fiche) {
		Collection<LigneContenu> ligneFicheCourante = new ArrayList<>();
		String contenu = StringUtils.EMPTY;
		try {
			if (fiche != null) {
				contenu = (String) PropertyUtils.getProperty(fiche, "contenu");
			}
		} catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
			LOG.debug("pas de propriété contenu sur cette fiche");
		}
		if (StringUtils.isNotBlank(contenu) && StringUtils.contains(contenu, "[paragraphe")) {
			ligneFicheCourante = ParagrapheContenuHelper.getLignes(contenu);
		}
		return ligneFicheCourante;
	}

	/**
	 * Calcule la liste de paragraphe contenu dans la fiche fourni en paramètre.
	 *
	 * @param fiche la fiche dont on souhaite calculer les paragraphes
	 * @return la liste de paragraphe contenu de la fiche ou une liste vide
	 */
	public static Collection<ParagrapheBean> getParagrapheDepuisFicheCourante(final FicheUniv fiche) {
		final Collection<ParagrapheBean> paragraphe = new ArrayList<>();
		final Collection<LigneContenu> lignes = getLignesFicheCourante(fiche);
		for (final LigneContenu ligne : lignes) {
			paragraphe.addAll(ligne.getParagraphes());
		}
		return paragraphe;
	}



	/**
	 * Méthode là uniquement pour la retro-compat...
	 * @param ctx le contexte de la page courante
	 * @param ficheUniv pour rien...
	 * @param sizeListe le nivveau de profondeur que l'on souhaite atteindre au maximum
	 * @return la liste de l'arbo des rubriques de la page courante
	 * @deprecated méthode inutile vu qu'on ne passe pas l'attribut ficheUniv. Utiliser {@link FrontUtil#calculerRubriquesPageCourante(ContexteUniv, int)}
	 */
	@Deprecated
	public static ArrayList<InfosRubriques> calculerRubriquesPageCourante(final ContexteUniv ctx,final FicheUniv ficheUniv, final int sizeListe) {
		return calculerRubriquesPageCourante(ctx, sizeListe);
	}
	/**
	 * Recopie de fonctions.jsp méthode valorisant les différentes rubriques utilisées pour gérer la navigation on lui passe en paramètre le contexte et la ficheUniv courantes
	 * (pour trouver la rubrique courante), et le nombre de rubriques à récupérer (=la taille de la liste)
	 *
	 * @param ctx le contexte de la page courante
	 * @param sizeListe le nivveau de profondeur que l'on souhaite atteindre au maximum
	 * @return la liste de l'arbo des rubriques de la page courante
	 */
	public static ArrayList<InfosRubriques> calculerRubriquesPageCourante(final ContexteUniv ctx, final int sizeListe) {
		final ArrayList<InfosRubriques> listeRubriques = new ArrayList<>(sizeListe);
		final String codeRubriqueCourante = ctx.getCodeRubriquePageCourante();
		InfosRubriques rubriqueCourante;
		rubriqueCourante = Rubrique.renvoyerItemRubrique(codeRubriqueCourante);
		InfosRubriques rubriqueTemp = rubriqueCourante;
		int niveauRubriqueCourante = rubriqueCourante.getNiveau();
		/* On initialise la liste avec des null : d'une part pour obtenir une liste dont la taille vaut sizeListe,
		   d'autre part pour donner tout de suite la bonne valeur aux rubriques des niveaux inférieurs au niveau courant, car elles sont non-significatives */
		for (int i = 0; i < sizeListe; i++) {
			listeRubriques.add(null);
		}
		if (niveauRubriqueCourante > 0) {
			/* Si la rubrique courante est trop basse dans l'arborescence par rapport au niveau significatif le plus bas, on remonte dans cette arborescence
			   jusqu'au niveau le plus bas */
			while (niveauRubriqueCourante > sizeListe) {
				rubriqueTemp = rubriqueTemp != null ? rubriqueTemp.getRubriqueMere() : null;
				if (rubriqueTemp != null) {
					niveauRubriqueCourante = rubriqueTemp.getNiveau();
				}
			}
			/* Ensuite, on insère les rubriques renseignées dans la liste, depuis son début jusqu'au niveau-1 de la rubrique courante */
			for (int i = niveauRubriqueCourante - 2; i >= 0; i--) {
				rubriqueTemp = rubriqueTemp != null ? rubriqueTemp.getRubriqueMere() : null;
				listeRubriques.set(i, rubriqueTemp);
			}
			/* puis on ajoute la rubrique courante à sa place dans la liste */
			listeRubriques.set(niveauRubriqueCourante - 1, rubriqueCourante);
		}
		return listeRubriques;
	}

	public static String getUrlBandeauCourant(final ContexteUniv ctx) {
		String urlBandeau = StringUtils.EMPTY;
		InfosRubriques rubriqueCourante;
		rubriqueCourante = Rubrique.renvoyerItemRubrique(ctx.getCodeRubriquePageCourante());
		while (rubriqueCourante != null && StringUtils.isNotBlank(rubriqueCourante.getCode())) {
			if (StringUtils.isNotBlank(rubriqueCourante.getUrlBandeau())) {
				urlBandeau = rubriqueCourante.getUrlBandeau();
				break;
			}
			rubriqueCourante = rubriqueCourante.getRubriqueMere();
		}
		return urlBandeau;
	}

	/**
	 * Recopie de fonctions.jsp. Méthode formatant les fichiers joints d'une fiche sous forme de liste à puces, englobée d'un UL déterminé
	 *
	 * @param ctx le contexte de la page courante
	 * @param ficheUniv la fiche dont on souhaite connaitre la liste des fichiers joints
	 * @return la liste ul / li des fichiers joints...
	 */
	public static String calculerListeFichiersJoints(final ContexteUniv ctx, final FicheUniv ficheUniv) {
		String res = calculerContenuListeFichiersJoints(ctx, ficheUniv);
		if (res.length() > 0) {
			res = "<ul id=\"telecharger\">" + res + "</ul>";
		}
		return res;
	}

	/**
	 * Recopie de fonctions.jsp. Méthode formatant les fichiers joints d'une fiche sous forme de liste à puces
	 *
	 * @param ctx le contexte de la page courante
	 * @param ficheUniv la fiche dont on souhaite connaitre la liste des fichiers joints
	 * @return la liste ul / li des fichiers joints...
	 */
	public static String calculerContenuListeFichiersJoints(final ContexteUniv ctx, final FicheUniv ficheUniv) {
		final StringBuilder res = new StringBuilder();
		Collection<Ressource> fichiers;
		try {
			fichiers = Ressource.getListeFichier(ficheUniv);
		} catch (final Exception e) {
			LOG.error("impossible de récupérer la liste des fichiers", e);
			fichiers = Collections.emptyList();
		}
		final String classeFichierDefaut = "defaut";
		String classeFichier;
		for (final Ressource fichier : fichiers) {
			if (ClasseFichier.getInstance().containsKey(fichier.getFormat())) {
				classeFichier = ClasseFichier.getInstance().get(fichier.getFormat());
			} else {
				classeFichier = classeFichierDefaut;
			}
			res.append("<li class=\"").append(classeFichier).append("\">");
			res.append("<a href=\"").append(URLResolver.getRessourceUrl(fichier.getUrl(), ctx)).append("\">");
			if (fichier.getTitre().length() > 0) {
				res.append(fichier.getTitre());
			} else {
				res.append(fichier.getSource());
			}
			res.append("</a> ");
			/* extension du fichier + poids en Ko */
			res.append("<span class='extension-poids-fichiers'> (");
			if (fichier.isLocal() && fichier.getExtension().length() > 0) {
				res.append(fichier.getExtension().toUpperCase());
			} else if (classeFichier.equals(classeFichierDefaut)) {
				res.append(MessageHelper.getCoreMessage("ST_EXTENSION_AUTRE"));
			} else {
				res.append(classeFichier.toUpperCase());
			}
			if (fichier.getPoids() > 0) {
				res.append(", ").append(fichier.getPoids()).append(" ").append(MessageHelper.getCoreMessage("ST_KILO_OCTETS"));
			}
			res.append(")</span>");
			res.append("</li>");
		}
		return res.toString();
	}

	/**
	 * Calcule le titre de la page courante à partir du composant sur lequel on est exple : pagelibre -> Saisie Page libre ...
	 *
	 * @param infoBean l'infobean courant
	 * @return le titre de l'écran si il est défini, le titre du composant sinon
	 */
	public static String getTitrePageCourante(final InfoBean infoBean) {
		return AdminsiteUtils.getTitrePageCourante(infoBean);
	}

	/**
	 * On affiche le visuel de rubrique uniquement si : - le visuel est renseigné - on est pas sur l'accueil d'un site - on est pas sur l'accueil d'une rubrique de niveau 1.
	 *
	 * @param ficheUniv
	 *            la fiche courante.
	 * @param visuelRubrique
	 *            le visuel à afficher
	 * @return vrai si les trois conditions ci-dessus sont vérifiées.
	 */
	public static boolean isAffichageVisuelRubrique(final FicheUniv ficheUniv, final String visuelRubrique) {
		final List<InfosRubriques> rubriquesPageCourante = FrontUtil.calculerRubriquesPageCourante(ContexteUtil.getContexteUniv(), 2);
		final InfosRubriques rubriqueNiveau1 = rubriquesPageCourante.get(1);
		boolean affichageVisuel = ficheUniv != null;
		if (affichageVisuel) {
			affichageVisuel = !FrontUtil.isAccueilSite(ficheUniv) && StringUtils.isNotBlank(visuelRubrique) && (rubriqueNiveau1 != null && ficheUniv.getCode().equals(
				rubriqueNiveau1.getCode()));
		}
		return affichageVisuel;
	}

	/**
	 * Calcule le menu de langue de l'application :
	 * <ul>
	 * <li>Si la rubrique de site est aussi la rubrique de langue, recherche des rubrique de langue depuis son parent</li>
	 * <li>Si la rubrique de site n'est pas la rubrique de langue, recherche à partir de celle-ci les rubriques de langues</li>
	 * </ul>
	 *
	 * @return Une liste de bean Menu ou une liste vide si non trouvé
	 */
	public static List<Menu> getMenuLangueParCategorie() {
		final InfosRubriques rubriqueSite = Rubrique.renvoyerItemRubrique(ContexteUtil.getContexteUniv().getInfosSite().getCodeRubrique());
		if (CATEGORIE_LANGUE.equals(rubriqueSite.getCategorie())) {
			return getMenuParRubriqueParente(rubriqueSite.getRubriqueMere());
		} else {
			return getMenuParRubriqueParente(rubriqueSite);
		}
	}

	/**
	 * Calcule le menu des réseaux sociaux par catégorie de rubrique
	 *
	 * @return Une liste de bean Menu ou une liste vide si non trouvé
	 */
	public static List<Menu> getMenuReseauxSociauxParCategorie() {
		return FrontUtil.getMenuParRubriqueParente(FrontUtil.getRubriqueDepuisCategorie(CATEGORIE_RESEAUX_SOCIAUX));
	}

	/**
	 * Calcule le menu principal de l'application
	 *
	 * @return Une liste de bean Menu ou une liste vide si non trouvé
	 */
	public static List<Menu> getMenuPrincipalParCategorie() {
		return getMenu(getRubriqueDepuisCategorie(CATEGORIE_NAVIGATION), 2);
	}

	/**
	 * Calcule le menu du pied de page de l'application
	 *
	 * @return Une liste de bean Menu ou une liste vide si non trouvé
	 */
	public static List<Menu> getMenuPiedDePageParCategorie() {
		return getMenu(getRubriqueDepuisCategorie(CATEGORIE_PIED_PAGE), 2);
	}

	/**
	 * Calcule le menu d'accès direct par catégorie de rubrique
	 *
	 * @return Une liste de bean Menu ou une liste vide si non trouvé
	 */
	public static List<Menu> getMenuAccesDirectParCategorie() {
		return getMenuParRubriqueParente(getRubriqueDepuisCategorie(CATEGORIE_ACCES_DIRECT));
	}

	public static List<Menu> getMenuPlanSite() {
		return getMenu(getRubriqueDepuisCategorie(CATEGORIE_NAVIGATION), 2);
	}

	/**
	 * calcule le menu secondaire de la page courante
	 * @return la liste de menu qui
	 */
	public static List<Menu> getMenuSecondairePageCourante() {
		final ContexteUniv ctx = ContexteUtil.getContexteUniv();
		final String codeRubriquePageCourante = ctx.getCodeRubriquePageCourante();
		final InfosRubriques rubriqueCourante = Rubrique.renvoyerItemRubrique(codeRubriquePageCourante);
		final Collection<Menu> menusPrincipal = getMenuPrincipalParCategorie();
		InfosRubriques rubriqueParente = null;
		for (final Menu itemMenuPrincipal : menusPrincipal) {
			final InfosRubriques rubriqueMenuPrincipal = Rubrique.renvoyerItemRubrique(itemMenuPrincipal.getCodeRubriqueOrigine());
			if (rubriqueMenuPrincipal.getCode().equals(rubriqueCourante.getCode()) || rubriqueMenuPrincipal.contains(rubriqueCourante)) {
				rubriqueParente = rubriqueMenuPrincipal;
				break;
			}
		}
		final List<Menu> menuSecondaire = new ArrayList<>();
		if (rubriqueParente != null) {
			menuSecondaire.addAll(getMenu(rubriqueParente, NIVEAU_MENU_SECONDAIRE));
		}
		return menuSecondaire;
	}

	/**
	 * Calcule un objet menu et ses enfants jusqu'à une certaine profondeur
	 *
	 * @param rubriqueParente
	 *            Rubrique d'origine
	 * @return Objet Menu contenant tous les menus de premier niveau
	 */
	public static List<Menu> getMenuParRubriqueParente(final InfosRubriques rubriqueParente) {
		final List<Menu> menusParRubriqueParente = new ArrayList<>();
		final ContexteUniv ctx = ContexteUtil.getContexteUniv();
		if (rubriqueParente != null && StringUtils.isNotBlank(rubriqueParente.getCode())) {
			for (final InfosRubriques rubrique : rubriqueParente.getListeSousRubriquesFront(ctx)) {
				menusParRubriqueParente.add(getMenuParRubrique(rubrique));
			}
		}
		return menusParRubriqueParente;
	}

	/**
	 * Calcule un menu à partir de la rubrique parente fourni en paramètre & le niveau de profondeur choisi.
	 *
	 * @param rubriqueParente
	 *            la rubrique à partir de laquelle on calcule le menu
	 * @param profondeur
	 *            le niveau de profondeur des rubriques a aller chercher
	 * @return une collection de {@link Menu} correspondant au sous rubrique de la rubrique fourni en paramètre ou une liste vide si non trouvé
	 */
	public static List<Menu> getMenu(final InfosRubriques rubriqueParente, final int profondeur) {
		final List<Menu> menuCourant = new ArrayList<>();
		if (rubriqueParente != null && StringUtils.isNotBlank(rubriqueParente.getCode())) {
			for (final InfosRubriques rubrique : rubriqueParente.getListeSousRubriquesFront(ContexteUtil.getContexteUniv())) {
				final Menu itemMenuCourant = getMenuParRubrique(rubrique);
				if (profondeur > 0) {
					itemMenuCourant.addAllSousMenu(getMenu(rubrique, profondeur - 1));
				}
				menuCourant.add(itemMenuCourant);
			}
		}
		return menuCourant;
	}

	/**
	 * Calcul un objet menu en indiquant son titre, son url, …
	 *
	 * @param rubrique
	 *            Rubrique de référence
	 * @return Objet Menu valorisé en fonction de la rubrique.
	 */
	public static Menu getMenuParRubrique(final InfosRubriques rubrique) {
		final Menu menuRubriqueCourante = new Menu();
		if (rubrique != null && StringUtils.isNotBlank(rubrique.getCode())) {
			menuRubriqueCourante.setCode(rubrique.getCode());
			menuRubriqueCourante.setCodeRubriqueOrigine(rubrique.getCode());
			menuRubriqueCourante.setLibelle(rubrique.getIntitule());
			menuRubriqueCourante.setLangue(rubrique.getLangue());
			menuRubriqueCourante.setType(rubrique.getTypeRubrique());
			menuRubriqueCourante.setVisuel(rubrique.getUrlBandeau());
			menuRubriqueCourante.setPicto(rubrique.getUrlPicto());
			menuRubriqueCourante.setAccroche(rubrique.getAccroche());
			try {
				final ContexteUniv ctx = ContexteUtil.getContexteUniv();
				menuRubriqueCourante.setUrl(UnivWebFmt.renvoyerUrlAccueilRubrique(ctx, rubrique.getCode()));
				menuRubriqueCourante.setMenuCourant(isMenuCourant(menuRubriqueCourante, ctx.getCodeRubriquePageCourante()));
			} catch (final Exception e) {
				LOG.error("Pas de page d'accueil pour la rubrique « " + rubrique.getCode() + " ».");
				menuRubriqueCourante.setUrl(StringUtils.EMPTY);
			}
		}
		return menuRubriqueCourante;
	}

	/**
	 * Renvoie la rubrique correspondant à la catégorie en fonction du site du contexte courant. Renvoie null si aucune rubrique de cette catégorie n'est trouvée.
	 *
	 * @param categorie la catégorie à rechercher
	 * @return la première rubrique de la catégorie ou null si non trouvé.
	 */
	public static InfosRubriques getRubriqueDepuisCategorie(final String categorie) {
		final ContexteUniv ctx = ContexteUtil.getContexteUniv();
		String codeRubriqueSiteLangue = ctx.getInfosSite().getCodeRubrique();
		final InfosRubriques rubrique = Rubrique.renvoyerItemRubrique(codeRubriqueSiteLangue);
		if (!CATEGORIE_LANGUE.equals(rubrique.getCategorie())) {
			final String langueCourante = ctx.getLangue();
			codeRubriqueSiteLangue = getCodeRubriqueLangueParCodeLangue(codeRubriqueSiteLangue, langueCourante);
		}
		final List<InfosRubriques> listeRubriques = getListeSousRubriquesDepuisCategorie(categorie, codeRubriqueSiteLangue);
		if (listeRubriques.size() > 0) {
			return listeRubriques.get(0);
		} else {
			return null;
		}
	}

	/**
	 * Récupère le code de rubrique de catégorie {@link FrontUtil#CATEGORIE_LANGUE} ayant pour langue la langue fourni en paramètre.
	 *
	 * @param codeRubriqueSiteLangue
	 *            le code de rubrique de site ou de langue
	 * @param langueCourante
	 *            la langue de la rubrique que l'on souhaite récupérer
	 * @return le code rubrique de la rubrique de langue ou {@link StringUtils#EMPTY} si non trouvé
	 */
	public static String getCodeRubriqueLangueParCodeLangue(final String codeRubriqueSiteLangue, final String langueCourante) {
		final List<InfosRubriques> rubriques = getListeSousRubriquesDepuisCategorie(CATEGORIE_LANGUE, codeRubriqueSiteLangue);
		String codeRubrique = StringUtils.EMPTY;
		for (final InfosRubriques rubrique : rubriques) {
			if (langueCourante.equals(rubrique.getLangue())) {
				codeRubrique = rubrique.getCode();
			}
		}
		return codeRubrique;
	}

	/**
	 * Récupère les {@link InfosRubriques} de la catégorie et de la rubrique parente fourni en paramètre
	 *
	 * @param categorie
	 *            la catégorie de rubrique à rechercher
	 * @param codeRubriqueParente
	 *            la rubrique à partir de laquelle on recherche les {@link InfosRubriques} à retourner
	 * @return La liste des {@link InfosRubriques} correspondant à la catégorie ou une liste vide sinon.
	 */
	public static List<InfosRubriques> getListeSousRubriquesDepuisCategorie(final String categorie, final String codeRubriqueParente) {
		final List<InfosRubriques> rubriques = new ArrayList<>();
		final InfosRubriques rubriqueParente = Rubrique.renvoyerItemRubrique(codeRubriqueParente);
		for (final InfosRubriques rubrique : rubriqueParente.getListeSousRubriques()) {
			if (categorie.equals(rubrique.getCategorie())) {
				rubriques.add(rubrique);
			} else {
				rubriques.addAll(getListeSousRubriquesDepuisCategorie(categorie, rubrique.getCode()));
			}
		}
		return rubriques;
	}

	/**
	 * Renvoie la rubrique en fonction de la catégorie et du code site. Renvoie null si aucune sous-rubrique de cette catégorie n'est trouvée.
	 *
	 * @param categorie la catégorie de la rubrique que l'on souhaite récupérer
	 * @param codeRubriqueSiteLangue le code de la rubrique mère
	 * @return L {@link InfosRubriques} correspondant ou null si non trouvé.
	 */
	public static InfosRubriques getRubriqueDepuisCategorieEtRubriqueDeSite(final String categorie, final String codeRubriqueSiteLangue) {
		InfosRubriques resultat = null;
		if (StringUtils.isNotEmpty(categorie)) {
			final InfosRubriques rubriqueSiteLangue = Rubrique.renvoyerItemRubrique(codeRubriqueSiteLangue);
			for (final InfosRubriques rubrique : rubriqueSiteLangue.getListeSousRubriques()) {
				if (categorie.equals(rubrique.getCategorie())) {
					resultat = rubrique;
					break;
				}
			}
		}
		return resultat;
	}

	/**
	 * Calcule le logo du site courant
	 * @return l'url du logo ou null si non trouvé
	 */
	public static String getLogoUrl() {
		try {
			return InfosSiteHelper.getURLRelativeFichierPropertyTemplate(ContexteUtil.getContexteUniv().getInfosSite(), NOM_PROPRIETE_LOGO);
		} catch (final ErreurDonneeNonTrouve e) {
			// TODO : placeholder ?
			return StringUtils.EMPTY;
		}
	}

	/**
	 * Renvoie la baseline défini sur la déclaration du site
	 * @return la baseline ou null si non trouvé.
	 */
	public static String getBaseline() {
		return ContexteUtil.getContexteUniv().getInfosSite().getProprieteComplementaireString(NOM_PROPRIETE_BASELINE);
	}

	/**
	 * Calcule la couleur principale spécifiée sur le site courant
	 * @return la couleur ou null si non trovué
	 */
	public static String getCouleurPrincipale() {
		final ContexteUniv ctx = ContexteUtil.getContexteUniv();
		final InfosSite siteCourant = ctx.getInfosSite();
		return siteCourant.getProprieteComplementaireString(NOM_PROPRIETE_COULEUR_PRINCIPALE);
	}

	/**
	 * Calcule la couleur secondaire spécifiée sur le site courant
	 * @return la couleur ou null si non trovué
	 */
	public static String getCouleurSecondaire() {
		final ContexteUniv ctx = ContexteUtil.getContexteUniv();
		final InfosSite siteCourant = ctx.getInfosSite();
		return siteCourant.getProprieteComplementaireString(NOM_PROPRIETE_COULEUR_SECONDAIRE);
	}

	/**
	 * Retourne une valeur rgba() à partir d'un code hexadécimal de couleur et d'un taux d'opacité
	 *
	 * @param colorToTransform
	 *            Le code de la couleur à rendre translucide. Il doit être de la forme suivante : ^#?[0-9a-fA-F]{6}$
	 * @param alpha
	 *            Le taux d'opacité, chaîne de caractères dont la valeur décimale doit être comprise entre 0 et 1 inclus.
	 * @return Retourne une chaîne formatée rgba() pour le CSS. Si le code couleur n'est pas saisi dans le bon format, une chaîne vide est retournée
     * @deprecated utiliser {@link ColorUtils#getRGBaFromHexa(String, double)}
     */
    @Deprecated
	public static String getRGBaFromHexa(final String colorToTransform, final double alpha) {
		if (!isValidHtmlHexaValue(colorToTransform) || alpha > 1) {
			return colorToTransform;
		}
		final String hex = StringUtils.removeStart(colorToTransform, COLOR_PREFIX);
		final int r = Integer.valueOf(hex.substring(0, 2), 16);
		final int g = Integer.valueOf(hex.substring(2, 4), 16);
		final int b = Integer.valueOf(hex.substring(4, 6), 16);
		return String.format(RGBA_TEMPLATE, r, g, b, alpha);
	}

	/**
	 * Permet de vérifier si la valeur fourni en paramètre correspond à une valeur Hexa valide pour l'html
	 * @param colorValue la chaine à tester pour vérifier que c'est bien une couleur
	 * @return vrai si la chaine est de la forme 090909 #090909
     * @deprecated utiliser {@link ColorUtils#isValidHtmlHexaValue(String)}
     */
    @Deprecated
	public static boolean isValidHtmlHexaValue(String colorValue) {
		return StringUtils.isNotBlank(colorValue) && colorValue.matches("^(?:#|)[0-9a-fA-F]{6}$");
	}

	/**
	 * Retourne une couleur plus clair du taux de clarté passé en paramêtre.
	 *
	 * @param colorToTransform
	 *            la valeur heaxdécimale de la couleur
	 * @param lumi
	 *            la clarté à apporter (%)
	 * @return la chaine css hsl
     * @deprecated utiliser {@link ColorUtils#getHslFromHexa(String, double)}
	 */
    @Deprecated
	public static String getHslFromHexa(final String colorToTransform, final double lumi) {
		if (!isValidHtmlHexaValue(colorToTransform)) {
			return colorToTransform;
		}
		final String hex = StringUtils.removeStart(colorToTransform, COLOR_PREFIX);
		final int r = Integer.valueOf(hex.substring(0, 2), 16);
		final int g = Integer.valueOf(hex.substring(2, 4), 16);
		final int b = Integer.valueOf(hex.substring(4, 6), 16);
		//    Minimum and Maximum RGB values are used in the HSL calculations
		final float rp = r / 255f;
		final float gp = g / 255f;
		final float bp = b / 255f;
		final float min = Math.min(rp, Math.min(gp, bp));
		final float max = Math.max(rp, Math.max(gp, bp));
		//  Calculate the Hue
		final float delta = max - min;
		float h = 0f, s = 0f;
		float l = 0f;
		l = (max + min) / 2;
		if (max == rp) {
			h = (60 * (((gp - bp) / delta) % 6));
		} else if (max == gp) {
			h = (60 * (((bp - rp) / delta) + 2));
		} else if (max == bp) {
			h = (60 * (((rp - gp) / delta) + 4));
		}
		if (delta == 0) {
			s = 0;
		} else {
			s = delta / (1 - Math.abs((2 * l - 1)));
		}
		return String.format(HSL_TEMPLATE, Math.round(h), s * 100, l * 100 + lumi);
	}

	/**
	 * Permet de calculer si au moins une des sequences fourni en paramètre est non vide
	 *
	 * @param sequences les chaines de caractères à vérifier
	 * @return vrai ssi une des chaines fourni en paramètre n'est pas vide ou rempli d'espace.
	 */
	public static boolean isAnyNotBlank(final CharSequence... sequences) {
		boolean isAnyNotBlank = Boolean.FALSE;
		if (ArrayUtils.isNotEmpty(sequences)) {
			int i = 0;
			while (!isAnyNotBlank && i < sequences.length) {
				isAnyNotBlank = StringUtils.isNotBlank(sequences[i]);
				i++;
			}
		}
		return isAnyNotBlank;
	}
}
