/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kportal.scheduling.monitoring.impl.quartz;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.quartz.JobKey;
import org.quartz.listeners.JobListenerSupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Listener quartz permettant d'observer si un job a été éxécuté par le scheduler.
 * Le statut du job peut être observé via la méthode isJobEnded qui renvoie vrai si le job est terminé
 * 
 * @author alexandre.garbe
 * 
 */
public class BatchExecutionEndHelper extends JobListenerSupport {

	/**
	 * Le nom du listener
	 */
	private static final String NAME = "BatchExecutionEndHelper";

	/**
	 * Le logger de la classe
	 */
	Logger LOGGER = LoggerFactory.getLogger(BatchExecutionEndHelper.class);

	/**
	 * Map contenant les statuts des différents jobs observés par le listener
	 */
	private final Set<JobKey> jobNamesAndStatus = Collections.synchronizedSet(new HashSet<JobKey>(1));

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.quartz.JobListener#getName()
	 */
	@Override
	public String getName() {
		return NAME;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seeorg.quartz.listeners.JobListenerSupport#jobWasExecuted(org.quartz.
	 * JobExecutionContext, org.quartz.JobExecutionException)
	 */
	@Override
	public void jobWasExecuted(JobExecutionContext context, JobExecutionException jobException) {
		if(jobNamesAndStatus.contains(context.getJobDetail().getKey())){
			jobNamesAndStatus.remove(context.getJobDetail().getKey());
			LOGGER.debug("Fin du job : " + context.getJobDetail().getKey().getName());
		}
		super.jobWasExecuted(context, jobException);
	}

	/** Permet de nettoyer les jobs **/
	public void jobWasExecuted(JobKey key) {
		if(jobNamesAndStatus.contains(key)){
			jobNamesAndStatus.remove(key);
			LOGGER.debug("Fin du job : " + key);
		}
	}

	/**
	 * Retourne vrai si le job s'est terminé depuis la dernière fois
	 * 
	 * @param jobName
	 *            Le nom du job
	 * @param groupName
	 *            Le groupe du job
	 * @return Un booleen
	 */
	public boolean isJobEnded(String jobName, String groupName) {
		return !(jobNamesAndStatus.contains(new JobKey(jobName, groupName)));
	}

	public void jobToBeExecuted(String jobName, String groupName) {
		jobNamesAndStatus.add(new JobKey(jobName, groupName));
		LOGGER.debug("Execution du job : " + jobName);
	}

	public boolean isJobStarted(String jobName, String groupName) {
		return (jobNamesAndStatus.contains(new JobKey(jobName, groupName)));
	}

	public boolean isJobRunning(){
		return (jobNamesAndStatus.size()>0);
	}

	public JobKey getJobKeyRunning(){
		for (JobKey key :jobNamesAndStatus){
			return key;
		}
		return null;
	}
}
