/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kportal.scheduling.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.Hashtable;

import javax.activation.MimetypesFileTypeMap;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.quartz.JobDataMap;
import org.quartz.JobDetail;
import org.quartz.JobKey;
import org.quartz.SchedulerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.session.SessionUtilisateur;
import com.kportal.core.config.MessageHelper;
import com.kportal.scheduling.module.SchedulerManagerHelper;
import com.kportal.scheduling.spring.quartz.LogReportJob;
import com.kportal.servlet.ExtensionServlet;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.utils.SessionUtil;

public class DownloaderServlet extends ExtensionServlet {

	private static final long serialVersionUID = -6211231526718980078L;

	private static final String DROITS_INSUFFISANTS = "BO_DROITS_INSUFFISANTS";
	
	private static final String BO_ERREUR_TELECHARGEMENT = "BO_ERREUR_TELECHARGEMENT_LOG";

	private static final String BO_OPERATION_INTERDITE = "BO_OPERATION_INTERDITE";

	private static final String TYPE_PARAM = "TYPE";

	private static final String OCTETSTREAM_MIMETYPE = "application/octet-stream";

	private static final String CONTENT_DISPOSITION_FIELD = "Content-Disposition";

	private static final String LOG_OUT = "LOG";

	private static final Logger LOGGER = LoggerFactory.getLogger(DownloaderServlet.class);

	public void process(final HttpServletRequest req, final HttpServletResponse resp) throws ServletException, IOException {

		final Hashtable<String, Object> infosSessions = SessionUtil.getInfosSession(req);
		final AutorisationBean autorisation = (AutorisationBean) infosSessions.get(SessionUtilisateur.AUTORISATIONS);
		final String type = req.getParameter(TYPE_PARAM);

		final boolean permission = LOG_OUT.equals(type) && autorisation!=null && autorisation.possedePermission(ScriptsAutomatisesUtil.getPermissionTelechargement());
		if (infosSessions.get(SessionUtilisateur.CODE) == null || autorisation == null || !permission) {
			LOGGER.error(MessageHelper.getCoreMessage(DROITS_INSUFFISANTS));
			resp.setStatus(HttpServletResponse.SC_FORBIDDEN);
			resp.getWriter().write(MessageHelper.getCoreMessage(DROITS_INSUFFISANTS));
			return;
		}

		// Récupération des paramètres obligatoires de requête
		final String jobName = req.getParameter("JOB_NAME");
		
		JobDetail jobDetail;
		try {
			jobDetail = SchedulerManagerHelper.getSchedulerManager().getScheduler().getJobDetail(new JobKey(jobName));
			if (jobDetail==null ) {
				LOGGER.error("Impossible de récupérer le JobDetail du job [{}]", jobName);
				resp.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
				return;
			}
			JobDataMap jobData = jobDetail.getJobDataMap();
			if ( jobData==null ) {
				LOGGER.error("Impossible de récupérer le JobData du job [{0}]", jobName);
				resp.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
				return;
			}
			String filename = jobData.getString(LogReportJob.LOGNAME_KEY);
			if ( filename!=null ) {
				File logfile = new File(filename);
				definirEntetes(logfile, resp);
				try (
					final FileInputStream in = new FileInputStream(logfile); 
					final ServletOutputStream out = resp.getOutputStream();
				) {
					IOUtils.copy(in,out);
					out.flush();
				}
			}
		} catch (SchedulerException se) {
			LOGGER.error(MessageFormat.format("Une erreur est survenue lors de la récupération du fichier log du job [{0}]", jobName), se);
			resp.reset();
			resp.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
			resp.getWriter().write(MessageHelper.getCoreMessage(BO_ERREUR_TELECHARGEMENT));
		} catch (IOException ioe) {
			LOGGER.error(MessageFormat.format("Une erreur est survenue lors de la lecture du fichier log du job [{0}]", jobName), ioe);
			resp.reset();
			resp.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
			resp.getWriter().write(MessageHelper.getCoreMessage(BO_ERREUR_TELECHARGEMENT));
		}
	}

	@Override
	protected void doGet(final HttpServletRequest req, final HttpServletResponse resp) throws ServletException, IOException {
		process(req, resp);
	}

	@Override
	protected void doPost(final HttpServletRequest req, final HttpServletResponse resp) throws ServletException, IOException {
		resp.setStatus(HttpServletResponse.SC_METHOD_NOT_ALLOWED);
		resp.getWriter().write(MessageHelper.getCoreMessage(BO_OPERATION_INTERDITE));
		return;
	}

	/**
	 * Définit les entêtes de la réponse HTTP pour le téléchargement
	 *
	 * @param f
	 *			Fichier de téléchargement concerné
	 * @param resp
	 *			Objet définissant la réponse HTTP
	 * @throws IOException
	 */
	private void definirEntetes(final File f, final HttpServletResponse resp) throws IOException {
		final String mimetype = (new MimetypesFileTypeMap()).getContentType(f);
		resp.setContentType((mimetype != null) ? mimetype : OCTETSTREAM_MIMETYPE);
		resp.setContentLength((int) f.length());
		resp.setHeader(CONTENT_DISPOSITION_FIELD, "attachment; filename=\"" + FilenameUtils.getName(f.getName()) + "\"");
	}

	@Override
	public void init() {
		//Pas d'init nécessaire
	}
}
