/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kportal.support;

import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.lang.management.ThreadInfo;
import java.lang.management.ThreadMXBean;
import java.util.ArrayList;

import org.apache.commons.lang3.StringUtils;
import org.quartz.JobExecutionContext;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.database.BasicPoolMgr;
import com.jsbsoft.jtf.database.InfosConnexion;
import com.kportal.core.config.PropertyHelper;
import com.kportal.scheduling.spring.quartz.LogReportJob;
import com.univ.multisites.Site;

public class DumpMonitor extends LogReportJob {

	public static final String TYPE_DUMP = "typeDump";
	public static final String THREADS = "threads";
	public static final String CONNEXIONS = "connexions";
	public static final String ID_BEAN = "dumpMonitor";
	public static final String ID_THREAD_MONITOR = "threadMonitor";
	public static final String THREADS_FOR_DUMP = "300";
	public static final String CONNEXIONS_FOR_DUMP = "300";
	private long nbThreadsForDump;
	private long nbConnexionsForDump;
	private String type;
	private ThreadMonitor threadMonitor;

	@Override
	public void perform() {
		System.currentTimeMillis();
		if (CONNEXIONS.equalsIgnoreCase(type)) {
			dumpPool();
		}
		else if (THREADS.equalsIgnoreCase(type)) {
			dumpThread();
		}
	}

	private void dumpPool() {
		try {
			StringBuilder sBuilder = new StringBuilder();
			String nr = System.getProperty("line.separator");
			sBuilder.append("Monitoring du pool"+ nr+ nr);
			sBuilder.append("max active : "+ BasicPoolMgr.getNombreConnexionsOuvertes()+ nr);
			sBuilder.append("max iddle : "+BasicPoolMgr.getNombreConnexionsDisponibles()+ nr);
			logger.info(sBuilder.toString());
			sBuilder = new StringBuilder();
			sBuilder.append("Traces d'appel du pool"+ nr+ nr);
			ArrayList<InfosConnexion> lstConnections = BasicPoolMgr.getCurrentConnections();
			sBuilder.append("nombre de connexions non fermées : " + lstConnections.size()+ nr);
			for (InfosConnexion infos : lstConnections) {
				if (!infos.estDisponible()) {
					sBuilder.append("Trace : " + infos.hashCode()+ nr);
					StackTraceElement[] stack = infos.getStackTrace();
					for (int i = stack.length - 1; i >= 0; i--) {
						sBuilder.append(" "+stack[i].getClassName() + " | " + stack[i].getMethodName() + " | l." + stack[i].getLineNumber()+ nr);
					}
				}
			}
			logger.info(sBuilder.toString());

		} catch (Exception e) {
			logger.error("Erreur dans DumpMonitor.dumpPool() : ", e);
		}
	}

	private void dumpThread() {
		try {

			// Get runtime information
			RuntimeMXBean runtime = ManagementFactory.getRuntimeMXBean();

			// Handle thread info
			ThreadMXBean threads = ManagementFactory.getThreadMXBean();
			StringBuilder sBuilder = new StringBuilder();
			String nr = System.getProperty("line.separator");

			sBuilder.append("Monitoring des threads"+ nr+ nr);
			sBuilder.append("thread-count : "+Long.toString(threads.getThreadCount())+ nr);
			sBuilder.append("total-started-thread-count : "+ Long.toString(threads.getTotalStartedThreadCount())+ nr);
			sBuilder.append("daemon-thread-count : "+Long.toString(threads.getDaemonThreadCount())+ nr);
			sBuilder.append("peak-thread-count : "+Long.toString(threads.getPeakThreadCount())+ nr);
			logger.info(sBuilder.toString());

			long totalCpuTime = 0l;
			long totalUserTime = 0l;
			ThreadInfo[] threadInfos = threads.getThreadInfo(threads.getAllThreadIds(), Integer.MAX_VALUE);

			sBuilder = new StringBuilder();
			sBuilder.append("Détail des threads"+nr +nr);
			for (ThreadInfo threadInfo : threadInfos) {
				if (threads.getThreadCpuTime(threadInfo.getThreadId()) > 0) {

					sBuilder.append("id : " + threadInfo.getThreadId()+ nr);
					sBuilder.append("nom : " + threadInfo.getThreadName());
					sBuilder.append("cpu-time-ms : " + Long.toString(threads.getThreadCpuTime(threadInfo.getThreadId()) / 1000000l)+ nr);
					sBuilder.append("blocked-count : " + Long.toString(threadInfo.getBlockedCount())+ nr);
					sBuilder.append("blocked-time : " + Long.toString(threadInfo.getBlockedTime())+ nr);
					sBuilder.append("waited-count : " + Long.toString(threadInfo.getWaitedCount())+ nr);
					sBuilder.append("waited-time : " + Long.toString(threadInfo.getWaitedTime())+ nr);
				}
				// Update our aggregate values
				totalCpuTime += threads.getThreadCpuTime(threadInfo.getThreadId());
				totalUserTime += threads.getThreadUserTime(threadInfo.getThreadId());
			}
			logger.info(sBuilder.toString());
			sBuilder = new StringBuilder();
			sBuilder.append("Temps d'utilisation globaux"+nr+nr);
			long totalCpuTimeMs = totalCpuTime / 1000000l;
			long totalUserTimeMs = totalUserTime / 1000000l;
			// Compute thread percentages
			long uptime = runtime.getUptime();
			double cpuPercentage = ((double) totalCpuTimeMs / (double) uptime) * 100.0;
			double userPercentage = ((double) totalUserTimeMs / (double) uptime) * 100.0;
			sBuilder.append("total-cpu-time-ms : " + Long.toString(totalCpuTimeMs)+nr);
			sBuilder.append("total-user-time-ms : " + Long.toString(totalUserTimeMs)+nr);
			sBuilder.append("uptime : " + Long.toString(uptime)+nr);
			sBuilder.append("total-cpu-percent : " + Double.toString(cpuPercentage)+nr);
			sBuilder.append("total-user-percent : " + Double.toString(userPercentage)+nr);
			logger.info(sBuilder.toString());
			logger.info("Dump des threads" +nr +threadMonitor.getFullThreadDump());

		} catch (Exception e) {
			logger.error("Erreur dans DumpMonitor.dumpThread() : ", e);
		}
	}

	@Override
	public void init(final JobExecutionContext jobCtx) {
		type = jobCtx.getMergedJobDataMap().getString(TYPE_DUMP);
		nbThreadsForDump = new Long(StringUtils.defaultIfEmpty(PropertyHelper.getCoreProperty("dumpMonitor.nbThreadsForDump"),THREADS_FOR_DUMP));
		nbConnexionsForDump = new Long(StringUtils.defaultIfEmpty(PropertyHelper.getCoreProperty("dumpMonitor.nbConnexionsForDump"),CONNEXIONS_FOR_DUMP));
		threadMonitor = (ThreadMonitor)ApplicationContextManager.getCoreContextBean(ID_THREAD_MONITOR);
	}

	@Override
	public boolean isReportByMail() {
		if (CONNEXIONS.equalsIgnoreCase(type)) {
			return BasicPoolMgr.getNombreConnexionsOuvertes() > nbConnexionsForDump;
		}
		else if (THREADS.equalsIgnoreCase(type)) {
			return ManagementFactory.getThreadMXBean().getThreadCount() > nbThreadsForDump;
		}
		return Boolean.FALSE;
	}

	@Override
	public String getLogFileName() {
		return DumpMonitor.class.getSimpleName().toLowerCase() + (type!=null ? "-" + type : "");
	}

	@Override
	public String getReportSubject() {
		String appli = StringUtils.defaultIfEmpty(PropertyHelper.getCoreProperty("application.nom"), Site.getSitePrincipal().getIntitule());
		if (CONNEXIONS.equalsIgnoreCase(type)) {
			return "["+appli+"][DumpMonitor] Alerte nombre de connexions > "+nbConnexionsForDump;
		}
		else if (THREADS.equalsIgnoreCase(type)) {
			return "["+appli+"][DumpMonitor] Alerte nombre de threads > " +nbThreadsForDump;
		}
		return "["+appli+"][DumpMonitor] Alerte";
	}

}
