/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.batch;

import java.io.File;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.Logger;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.FileAppender;

import com.jsbsoft.jtf.email.JSBMailbox;
import com.kportal.core.config.PropertyHelper;
import com.kportal.core.logging.DefaultRollingFileAppender;

public class LogReport {

	private final Logger logger;

	public Logger getLogger() {
		return logger;
	}

	private final String key;

	private String reportSubject;

	private final Level level;

	private boolean sendReportByMail;

	private String appenderMailKey;
	
    public LogReport(String key) {
		this.logger = (Logger) LoggerFactory.getLogger(key);
		this.key = key;
		this.level = Level.INFO;
		this.sendReportByMail = false;
		this.reportSubject = StringUtils.EMPTY;
		initMail();
		initAppender();
	}

	public LogReport(String key, Level level, boolean sendReportByMail, String subject) {
		this.logger = (Logger) LoggerFactory.getLogger(key);
		this.key = key;
		this.level = level;
		this.sendReportByMail = sendReportByMail;
		this.reportSubject = subject;
		initMail();
		initAppender();
	}

	private void initMail() {
		String mail = PropertyHelper.getCoreProperty("batch." + key + ".mail");
		if ("1".equals(mail)) {
			this.sendReportByMail = true;
		}
		String subject = PropertyHelper.getCoreProperty("batch." + key + ".mail.subject");
		if (StringUtils.isNotEmpty(subject)) {
			this.reportSubject = subject;
		}
	}

	private void initAppender() {
		try {
			// On crée le nouvel appender si il n'existe pas déjà
			if (logger.getAppender(key) == null) {
				DefaultRollingFileAppender appender = new DefaultRollingFileAppender(key);
				appender.setContext(logger.getLoggerContext());
				appender.start();
				// On ajoute l'appender au logger.
				logger.addAppender(appender);
			}
			// Appender destiné à l'envoi de mail
			if (sendReportByMail) {
				appenderMailKey = key +"-mail";
				DefaultRollingFileAppender appenderMail = new DefaultRollingFileAppender(appenderMailKey,appenderMailKey+"-"+System.currentTimeMillis());
				appenderMail.setContext(logger.getLoggerContext());
				appenderMail.setAppend(false);
				appenderMail.start();
				// On ajoute l'appender au logger.
				logger.addAppender(appenderMail);
			}
			logger.setLevel(level);
			logger.setAdditive(false);
		} catch (Exception e) {
			logger.error("Erreur d'initialisation du logger " + key + " : " + e.getMessage());
		}
	}

	public void info(String add) {
		logger.info(add);
	}

	public void error(String add) {
		logger.error(add);
	}

	public void debug(String add) {
		logger.debug(add);
	}

	public void close() {
		if (sendReportByMail){
			FileAppender<ILoggingEvent> appenderMail = (FileAppender<ILoggingEvent>) logger.getAppender(appenderMailKey);
			try {
				if (appenderMail != null) {
					File fLog = new File(appenderMail.getFile());
					if (fLog.exists()) {
						JSBMailbox mailbox = new JSBMailbox(false);
						String to = PropertyHelper.getCoreProperty("batch." + key + ".mail.to");
						if (StringUtils.isEmpty(to)) {
							to = PropertyHelper.getCoreProperty("mail.webmaster");
						}
						mailbox.sendSystemMsgWithAttachedFiles(to, reportSubject, "Ci-joint le rapport d'exécution du script " + key + ".", new String[]{appenderMail.getFile()});
					}
				}
			} catch (Exception e) {
				logger.error(e.getMessage(), e);
			} finally {
				if (appenderMail != null) {
					logger.detachAppender(appenderMail);
					appenderMail.stop();
					if ( !new File(appenderMail.getFile()).delete() ) {
						logger.error("Impossible de supprimer le fichier log {}",appenderMail.getFile());
					}
				}
			}
		}
	}
	
    public String getLogFilename() {
        FileAppender<ILoggingEvent> appender = (FileAppender<ILoggingEvent>) logger.getAppender(key);
        return appender.getFile();
    }
}
