/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.batch.exports;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.Files;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.LangueUtil;
import com.kportal.core.config.MessageHelper;
import com.kportal.core.config.PropertyHelper;
import com.kportal.scheduling.spring.quartz.LogReportJob;
import com.univ.objetspartages.om.Groupedsi;
import com.univ.objetspartages.om.Groupeutilisateur;
import com.univ.objetspartages.om.InfosGroupeDsi;
import com.univ.objetspartages.om.InfosRole;
import com.univ.objetspartages.om.Perimetre;
import com.univ.objetspartages.om.Role;
import com.univ.objetspartages.om.Rubrique;
import com.univ.objetspartages.om.Structure;
import com.univ.objetspartages.om.Utilisateur;
import com.univ.utils.ContexteDao;
import com.univ.utils.ExportCsvUtil;

/**
 * Classe exportant les rôles affectés aux utilisateurs, le tout au format CSV et stocké dans le storage.
 *
 * @author Pierre BURGET
 */
public class ExportRoles extends LogReportJob implements Runnable {

    private static final Logger LOG = LoggerFactory.getLogger(ExportRoles.class);

    /**
     * The Constant EXPORT_TYPE.
     */
    private static final String EXPORT_TYPE = "roles";

    private static final String CHEVRON = ">";

    /*
     * (non-Javadoc)
     *
     * @see java.lang.Runnable#run()
     */
    @Override
    public void run() {
        // Constitution de la map de langue
        // Les entêtes seront ajoutées au dernier moment si jamais les données sont vides (cas improbable)
        Map<Locale, List<String>> contentByLocale = new HashMap<Locale, List<String>>();
        for (Locale locale : LangueUtil.getLocales()) {
            contentByLocale.put(locale, new LinkedList<String>());
        }
        constitutionDonneesExport(contentByLocale);
        constitutionFichiersExport(contentByLocale);
        purge();
    }

    @Override
    public void perform() {
        run();
    }

    private void constitutionDonneesExport(Map<Locale, List<String>> contentByLocale) {
        // On constitue les données d'export pour chacune des langues de l'appli
        LOG.info("Constitution des données d'export de rôles pour chacune des langues");
        // Sélection de l'ensemble des utilisateurs
        try (ContexteDao ctx = new ContexteDao()) {
            Utilisateur utilisateur = new Utilisateur();
            utilisateur.setCtx(ctx);
            utilisateur.init();
            utilisateur.select("");
            while (utilisateur.nextItem()) {
                String userInfos = getUserInfos(utilisateur);
                addRolesLine(userInfos, utilisateur.getRoles(), ExportCsvUtil.DEFAULT_SEPARATOR, contentByLocale);
                // On formate l'affichage de ses rôles hérités par ses groupes DSI
                Vector<String> groupesUtilisateur = Groupeutilisateur.getVecteurGroupes(utilisateur.getCode());
                for (int i = 0; i < groupesUtilisateur.size(); i++) {
                    InfosGroupeDsi groupe = Groupedsi.renvoyerItemGroupeDsi(groupesUtilisateur.get(i));
                    if (groupe.getRoles().length() > 0) {
                        addRolesLine(userInfos, groupe.getRoles(), getSuffixeHeritage(groupe), contentByLocale);
                    }
                }
            }
        } catch (Exception e) {
            LOG.error("Erreur lors de la récupération des données d'export de rôles", e);
        }
        LOG.info("Fin de la constitution des données d'export de rôles");
    }

    private void constitutionFichiersExport(Map<Locale, List<String>> contentByLocale) {
        LOG.info("Constitution des fichiers d'export de rôles");
        for (Locale locale : LangueUtil.getLocales()) {
            List<String> datas = contentByLocale.get(locale);
            if (!datas.isEmpty()) {
                // On crée le fichier d'export et on conserve le timestamp de l'archive en cas de rollback
                String currentCtm = ExportCsvUtil.createNewExportFile(EXPORT_TYPE, locale);
                // Si la création du fichier d'export a réussi
                if (StringUtils.isNotEmpty(currentCtm) && !currentCtm.equals(ExportCsvUtil.ERROR)) {
                    // On récupère le fichier créé précédemment pour le renseigner
                    try (FileWriter fw = new FileWriter(new File(ExportCsvUtil.getExportFilePath(EXPORT_TYPE, locale)));) {
                        fw.write(MessageHelper.getCoreMessage(locale, "EXPORT_ROLES_HEADER_CSV") + "\n");
                        for (String data : datas) {
                            fw.write(data);
                        }
                    } catch (IOException e) {
                        LOG.error("Erreur lors de l'écriture du fichier d'export de rôles en locale " + locale, e);
                        if (!ExportCsvUtil.rollbackExportFile(EXPORT_TYPE, locale, currentCtm)) {
                            LOG.error("Erreur lors du rollback du fichier d'export de rôles en locale " + locale);
                        }
                    }
                } else {
                    // Pas de rollback, le fichier n'ayant pas été créé
                    LOG.error("Erreur lors de la création du fichier d'export de rôles en locale " + locale);
                }
            }
        }
        LOG.info("Fin de la constitution des fichiers d'export de rôles");
    }

    /**
     * permet de préparer pour affichage une liste de rôles, avec leurs périmètres
     *
     * @param roles l'ensemble des rôles à préparer pour affichage
     * @return la liste des rôles préparés pour l'affichage
     */
    private void addRolesLine(String prefixe, String roles, String suffixe, Map<Locale, List<String>> contentByLocale) throws Exception {
        final StringTokenizer st = new StringTokenizer(StringUtils.defaultString(roles), "[]");
        int i = 0;
        while (st.hasMoreTokens()) {
            final String val = st.nextToken();
            final int indexPointVirgule = val.indexOf(';');
            if (indexPointVirgule != -1) {
                final String codeRole = val.substring(0, indexPointVirgule);
                final InfosRole infosRole = Role.renvoyerItemRole(codeRole);
                //on verifie que le role existe
                if (infosRole.getCode().length() > 0) {
                    final String sPerimetre = val.substring(indexPointVirgule + 1);
                    final Perimetre perimetre = new Perimetre(sPerimetre);
                    final Perimetre perimetreAffectation = new Perimetre(perimetre.getCodeStructure(), perimetre.getCodeRubrique(), "", perimetre.getCodeGroupe(), "");
                    final Perimetre perimetrePermission = Perimetre.calculerPerimetrePermission(perimetreAffectation, new Perimetre(infosRole.getPerimetre()));
                    for (Locale locale : contentByLocale.keySet()) {
                        StringBuilder sb = new StringBuilder();
                        sb.append(prefixe);
                        sb.append(infosRole.getIntitule() + ExportCsvUtil.DEFAULT_SEPARATOR);
                        String intitulePerimetre = "";
                        if (perimetrePermission == null) {
                            intitulePerimetre = MessageHelper.getCoreMessage(locale, "EXPORT_ROLES_PERIMETRE_INVALIDE");
                            // Calcul Structure
                            sb.append(intitulePerimetre);
                            // Calcul Rubrique
                            sb.append(intitulePerimetre);
                            // Calcul Groupe
                            sb.append(intitulePerimetre);
                        } else {
                            // Calcul Structure
                            if (StringUtils.isEmpty(perimetrePermission.getCodeStructure())) {
                                intitulePerimetre = MessageHelper.getCoreMessage(locale, "EXPORT_ROLES_TOUTES");
                            } else if (perimetrePermission.getCodeStructure().equals("-")) {
                                intitulePerimetre = MessageHelper.getCoreMessage(locale, "EXPORT_ROLES_AUCUNE");
                            } else {
                                intitulePerimetre = Structure.getFilAriane(perimetrePermission.getCodeStructure(), CHEVRON) + ExportCsvUtil.DEFAULT_SEPARATOR;
                            }
                            sb.append(intitulePerimetre);
                            // Calcul Rubrique
                            if (StringUtils.isEmpty(perimetrePermission.getCodeRubrique())) {
                                intitulePerimetre = MessageHelper.getCoreMessage(locale, "EXPORT_ROLES_TOUTES");
                            } else if (perimetrePermission.getCodeRubrique().equals("-")) {
                                intitulePerimetre = MessageHelper.getCoreMessage(locale, "EXPORT_ROLES_AUCUNE");
                            } else {
                                intitulePerimetre = Rubrique.getFilAriane(perimetrePermission.getCodeRubrique(), CHEVRON) + ExportCsvUtil.DEFAULT_SEPARATOR;
                            }
                            sb.append(intitulePerimetre);
                            // Calcul Groupe
                            if (StringUtils.isEmpty(perimetrePermission.getCodeGroupe())) {
                                intitulePerimetre = MessageHelper.getCoreMessage(locale, "EXPORT_ROLES_TOUS");
                            } else if (perimetrePermission.getCodeGroupe().equals("-")) {
                                intitulePerimetre = MessageHelper.getCoreMessage(locale, "EXPORT_ROLES_AUCUN");
                            } else {
                                intitulePerimetre = Groupedsi.getFilAriane(perimetrePermission.getCodeGroupe(), CHEVRON) + ExportCsvUtil.DEFAULT_SEPARATOR;
                            }
                            sb.append(intitulePerimetre);
                        }
                        sb.append(suffixe + ";\n");
                        contentByLocale.get(locale).add(sb.toString());
                    }
                }
            }
        }
    }

    private String getSuffixeHeritage(InfosGroupeDsi groupe) {
        return Groupedsi.getFilAriane(groupe.getCode(), CHEVRON) + ExportCsvUtil.DEFAULT_SEPARATOR;
    }

    private String getUserInfos(Utilisateur utilisateur) {
        StringBuilder userInfos = new StringBuilder();
        userInfos.append(utilisateur.getCode()).append(ExportCsvUtil.DEFAULT_SEPARATOR);
        userInfos.append(utilisateur.getCodeLdap()).append(ExportCsvUtil.DEFAULT_SEPARATOR);
        userInfos.append(utilisateur.getNom()).append(ExportCsvUtil.DEFAULT_SEPARATOR);
        userInfos.append(utilisateur.getPrenom()).append(ExportCsvUtil.DEFAULT_SEPARATOR);
        return userInfos.toString();
    }

    private void purge() {
        // Récupération de la valeur de rétention en jour
        int days = PropertyHelper.getCorePropertyAsInt("export.roles.retention", 15);
        // Construction de la date de purge
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date(System.currentTimeMillis()));
        cal.add(Calendar.DATE, -days);
        long datePurge = cal.getTimeInMillis();
        File folder = new File(ExportCsvUtil.getFolderSpecificPath(EXPORT_TYPE));
        // On inspecte le répertoire
        if (folder.exists()) {
            // Pour chaque fichier, on regarde si on le purge ou non en fonction de la date de création
            for (File file : folder.listFiles()) {
                try {
                    BasicFileAttributes attributes = Files.readAttributes(file.toPath(), BasicFileAttributes.class);
                    if (attributes.creationTime().toMillis() < datePurge) {
                        if (!file.delete()) {
                            LOG.error("Erreur lors de la suppression du fichier " + file.toPath());
                        }
                    }
                } catch (IOException e) {
                    LOG.error("Erreur lors de la suppression du fichier " + file.toPath(), e);
                }
            }
        }
    }
}
