/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * Created on 12 mars 2004
 *
 */
package com.univ.batch.imports;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FilenameFilter;
import java.util.StringTokenizer;

import org.apache.commons.lang3.StringUtils;
import org.quartz.JobExecutionContext;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.core.CodeLibelle;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.jsbsoft.jtf.webutils.ContextePage;
import com.kportal.scheduling.spring.quartz.LogReportJob;
import com.univ.objetspartages.cache.CacheLibelleManager;
import com.univ.objetspartages.om.Libelle;
import com.univ.utils.Chaine;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;

// TODO: Auto-generated Javadoc
/**
 * The Class ImportLibelles.
 * 
 * @author malice
 * 
 *         Importation de libellés arguments fournis : un dossier ou un fichier. s'il s'agit d'un dossier, on importe tous les fichiers qui ont l'extension .txt s'il s'agit d'un
 *         fichier, on importe ce fichier Il s'agit d'un import différentiel : si le libellé existe déjà, on le met à jour, sinon, on l'ajoute
 */
public class ImportLibelles extends LogReportJob {

	/**
	 * clé permettant de retrouver le paramètre fichier dans la map du job
	 */
	private static final String SELECTED_FILE = "SELECTED_FILE";

	/** The s folder or file. */
	static String sFolderOrFile = "";

	/**
	 * Importe le fichier de libellés.
	 * 
	 * @param f
	 *            Fichier de libellés à importer
	 * @param _ctx
	 *            contexte
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public void importFichier(final File f, final ContextePage _ctx) throws Exception {
		final FileReader freader = new FileReader(f);
		final CacheLibelleManager cache = (CacheLibelleManager) ApplicationContextManager.getCoreContextBean(CacheLibelleManager.ID_BEAN);
		/* lecture du fichier */
		try (final BufferedReader fichLogique = new BufferedReader(freader)) {
			final Libelle l = new Libelle();
			l.setCtx(_ctx);
			/* chaque ligne correspond à un libellé */
			String ligneLue = fichLogique.readLine();
			while (ligneLue != null) {
				String typeLibelle = "";
				String code = "";
				String libelle = "";
				String langue = "";
				/* la ligne se présente ainsi :
				 * 'typelibelle','codelibelle','libelle','languelibelle'
				 * les quotes ne sont pas forcément présentes
				 * */
				final StringTokenizer st = new StringTokenizer(ligneLue, ",");
				int ind = 0;
				while (st.hasMoreTokens()) {
					final String item = st.nextToken();
					if (ind == 0) {
						typeLibelle = enleverQuote(item);
					}
					if (ind == 1) {
						code = enleverQuote(item);
					}
					if (ind == 2) {
						libelle = enleverQuote(item);
					}
					if (ind == 3) {
						langue = enleverQuote(item);
					}
					ind++;
				}
				/* insertion du libellé dans la base, ou mise à jour si déjà
				 * existant (pour le savoir, recherche par code,langue et type)
				 */
				boolean isCodeValide = Boolean.TRUE;
				try {
					Chaine.controlerCodeMetier(code);
				} catch (final ErreurApplicative e) {
					isCodeValide = Boolean.FALSE;
				}
				final boolean isTypeValide = cache.getListeTypesLibelles().containsKey(typeLibelle);
				if (isCodeValide && isTypeValide && (libelle.length() > 0) && (langue.length() > 0)) {
					l.init();
					final ClauseWhere whereCodeLangueType = new ClauseWhere(ConditionHelper.egalVarchar("CODE", code));
					whereCodeLangueType.and(ConditionHelper.egalVarchar("LANGUE", langue));
					whereCodeLangueType.and(ConditionHelper.egalVarchar("TYPE", typeLibelle));
					l.select(whereCodeLangueType.formaterSQL());
					if (l.nextItem()) {
						l.setLibelle(libelle);
						l.update();
						logger.debug("libellé mis à jour : " + libelle + "(code : " + code + ", langue : " + langue + ", type : " + CodeLibelle.lireLibelle(null, "type_libelle",
							typeLibelle) + ")");
					} else {
						if (l.select("WHERE LIBELLE='" + StringUtils.replace(libelle, "'", "\\'") + "' AND LANGUE='" + langue + "' AND TYPE='" + typeLibelle + "'") > 0) {
							logger.debug("Le même libellé existe déjà avec un code différent. Ce libellé ('" + libelle + "', langue : " + langue + ", type :" + typeLibelle + ") n'est donc pas importé.");
						} else {
							Libelle.addLibelle(_ctx, typeLibelle, code, libelle, langue);
							logger.debug("libellé ajouté : " + libelle + "(code : " + code + ", langue : " + langue + ", type : " + CodeLibelle.lireLibelle(null, "type_libelle",
								typeLibelle) + ")");
						}
					}
				}
				ligneLue = fichLogique.readLine();
			}
		}
	}

	@Override
	public void init(final JobExecutionContext arg0) {
		sFolderOrFile = (String) arg0.getMergedJobDataMap().get(SELECTED_FILE);
	}

	@Override
	public void perform() {
		try {
			final ContextePage ctx = new ContextePage("");
			final File folderorfile = new File(sFolderOrFile);
			File file = null;
			logger.debug((folderorfile.isDirectory() ? "Dossier " : "Fichier "));
			if (!folderorfile.exists()) {
				throw new Exception((folderorfile.isDirectory() ? "Dossier " : "Fichier ") + sFolderOrFile + " inexistant");
			} else if (!folderorfile.canRead()) {
				throw new Exception((folderorfile.isDirectory() ? "Dossier " : "Fichier ") + sFolderOrFile + " non accessible en lecture");
			}
			/* le paramètre fourni est un dossier */
			if (folderorfile.isDirectory()) {
				/* on importe tous les fichiers .TXT */
				final FilenameFilter filtre = new TXTFilter();
				final File[] listeFiles = folderorfile.listFiles(filtre);
				for (final File listeFile : listeFiles) {
					logger.debug("fichier : " + listeFile.getName());
					importFichier(listeFile, ctx);
				}
			} else {
				file = folderorfile;
				importFichier(file, ctx);
			}
		} catch (final Exception e) {
			logger.error("Erreur d'import de libellés", e);
		}
	}

	/**
	 * Enleve les quotes de debut et fin de chaine.
	 * 
	 * @param chaine
	 *            : la chaine dont on souhaite enlever les quotes
	 * 
	 * @return the string
	 * 
	 * @retun chaine: la chaine après traitement
	 */
	private String enleverQuote(final String chaine) {
		String res = chaine.trim();
		if (res.charAt(0) == '\'') {
			res = res.substring(1);
		}
		if (res.charAt(res.length() - 1) == '\'') {
			res = res.substring(0, res.length() - 1);
		}
		return res;
	}
}
