/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.batch.process;

import static com.univ.objetspartages.om.EtatFiche.A_SUPPRIMER;

import java.io.File;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.GregorianCalendar;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.Vector;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.Formateur;
import com.jsbsoft.jtf.core.LangueUtil;
import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.email.JSBMailbox;
import com.jsbsoft.jtf.webutils.ContextePage;
import com.kportal.core.config.PropertyHelper;
import com.kportal.core.webapp.WebAppUtil;
import com.kportal.extension.module.plugin.objetspartages.PluginFicheHelper;
import com.kportal.scheduling.spring.quartz.LogReportJob;
import com.univ.mediatheque.utils.MediathequeHelper;
import com.univ.multisites.InfosFicheComparator;
import com.univ.multisites.InfosFicheReferencee;
import com.univ.multisites.Site;
import com.univ.objetspartages.bean.HistoriqueBean;
import com.univ.objetspartages.om.DiffusionSelective;
import com.univ.objetspartages.om.EtatFiche;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.Media;
import com.univ.objetspartages.om.Metatag;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.objetspartages.om.Ressource;
import com.univ.objetspartages.om.Rubrique;
import com.univ.objetspartages.om.Rubriquepublication;
import com.univ.objetspartages.om.Structure;
import com.univ.objetspartages.om.Utilisateur;
import com.univ.objetspartages.util.AffichageHistorique;
import com.univ.objetspartages.util.RessourceUtil;
import com.univ.utils.Chaine;
import com.univ.utils.ContexteDao;
import com.univ.utils.ContexteUtil;
import com.univ.utils.FicheUnivMgr;
import com.univ.utils.URLResolver;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.condition.ConditionList;
import com.univ.utils.sql.criterespecifique.ConditionHelper;

/**
 * The Class ScanSite.
 */
public class ScanSite extends LogReportJob implements Runnable {

	/** The separateur langue. */
	private static String separateurLangue = ",LANGUE=";

	private static Logger LOGGER = LoggerFactory.getLogger(ScanSite.class);

	/**
	 * Calcul du répertoire d'une page
	 *
	 * @param url the url
	 * @return the directory
	 * @deprecated Méthode plus utilisée
	 */
	@Deprecated
	public String getDirectory(final String url) {
		final int irepertoire = url.lastIndexOf('/');
		String directory;
		if (irepertoire <= 6) {
			directory = url;
		} else {
			directory = url.substring(0, irepertoire);
		}
		return directory;
	}

	/**
	 * The main method.
	 *
	 * @param args the arguments
	 */
	public static void main(final String[] args) {
		final ScanSite capture = new ScanSite();
		final Thread thread = new Thread(capture);
		thread.start();
		try {
			// JB 20050530 : attente du thread
			thread.join();
		} catch (final Exception e) {
			LOGGER.error("Echec thread.join()", e);
		}
	}

	/**
	 *
	 *
	 * @see java.lang.Runnable#run()
	 */
	@Override
	public void run() {
		try {
			// ajout d'un contexte dans le thread local
			ContexteUtil.setContexteSansRequete();
			logger.info("***** Suppression des fichiers temporaires *****");
			final String tmpDir = WebAppUtil.getWorkDefaultPath();
			// PCO - 18/06/2010
			// pour le dossier TMP, suppression de tout son contenu, même des
			// dossiers.
			final File dossierTemporaire = new File(tmpDir);
			removeContenuDossier(dossierTemporaire);
			// => Fin patch
			// JSS 20040114 : suppression fichiers session
			final String tmpDirSession = WebAppUtil.getSessionsPath();
			// PCO - 18/06/2010
			// pour le dossier SESSION, suppression de tout son contenu, même
			// des dossiers.
			removeContenuDossier(new File(tmpDirSession));
			// => Fin patch
			String majReferences = StringUtils.defaultString(PropertyHelper.getCoreProperty("static.maj.references"), "0");
			/* Parsing des pages */
			final GregorianCalendar dateLimite = new GregorianCalendar();
			dateLimite.add(Calendar.DATE, -1);
			ContextePage ctx;
			logger.info("***** Traitement des fiches *****");
			logger.info("Purge des fiches obsolètes");
			logger.info("Synchronisation des métatags");
			logger.info("Gestion des alertes");
			try {
				for (final String codeObjet : ReferentielObjets.getListeCodesObjet()) {
					String libelleObjet = ReferentielObjets.getLibelleObjet(codeObjet);
					logger.info("***** " + libelleObjet + " *****");
					final ArrayList<Long> arrayMetatag = new ArrayList<>();
					final ArrayList<String> arrayRubriquePublication = new ArrayList<>();
					// new ArrayList();
					ctx = new ContextePage("");
					try {
						FicheUniv ficheUniv = ReferentielObjets.instancierFiche(codeObjet);
						if (ficheUniv == null) {
							continue;
						}
						ficheUniv.setCtx(ctx);
						ficheUniv.init();
						int count = ficheUniv.selectCodeLangueEtat("", "", "");
						if (count > 0) {
							while (ficheUniv.nextItem()) {
								// sauvegarde des fiches traitees
								arrayMetatag.add(ficheUniv.getIdFiche());
								if ("0003".equals(ficheUniv.getEtatObjet())) {
									arrayRubriquePublication.add(ficheUniv.getCode() + separateurLangue + ficheUniv.getLangue());
								}
								// Purge des fiches à supprimer, apercus,
								// sauvegardes automatiques
								if ("0004".equals(ficheUniv.getEtatObjet()) || "0005".equals(ficheUniv.getEtatObjet()) || "0006".equals(ficheUniv.getEtatObjet())) {
									String traceSuppression = StringUtils.EMPTY;
									if (A_SUPPRIMER.getEtat().equals(ficheUniv.getEtatObjet())) {
										traceSuppression = String.format("Suppression de la fiche '%s' de type '%s' et de langue '%s'", ficheUniv.getLibelleAffichable(), libelleObjet, LangueUtil.getDisplayName(ficheUniv.getLangue()));
										List<HistoriqueBean> historiqueFiche = AffichageHistorique.getHistorique(ficheUniv);
										Collections.sort(historiqueFiche);
										try {
											HistoriqueBean lastOperation = historiqueFiche.get(historiqueFiche.size() - 1);
											if (Metatag.getIntituleAction(Metatag.HISTORIQUE_SUPPRESSION).equals(lastOperation.getAction()) && StringUtils.isNotBlank(lastOperation.getUtilisateur())) {
												traceSuppression += String.format(" par l'utilisateur '%s'", lastOperation.getUtilisateur());
											}
										} catch (ArrayIndexOutOfBoundsException ignored) {
										}
									}
									FicheUnivMgr.supprimerFiche(ficheUniv, true);
									if (StringUtils.isNotBlank(traceSuppression)) {
										logger.info(traceSuppression);
									}
									continue;
								}
								if (ficheUniv instanceof DiffusionSelective) {
									if (((DiffusionSelective) ficheUniv).getDiffusionModeRestriction().equals("")) {
										((DiffusionSelective) ficheUniv).setDiffusionModeRestriction("0");
										ficheUniv.update();
									}
								}
								// Traitement des meta tags
								FicheUnivMgr.synchroniserMetaTag(ficheUniv, majReferences, true);
								// Gestion des alertes
								genererAlerte(codeObjet, ficheUniv);
							}// fin de la boucle sur les fiches du meme type
						}
						// Suppression des metatags obsolètes
						ContextePage ctxFiche = new ContextePage("");
						try {
							supprimerMetaTag(ctxFiche, codeObjet, arrayMetatag);
						} catch (final Exception e) {
							logger.error("Exception lors de la suppression des metatgs", e);
						} finally {
							ctxFiche.release();
						}
						ctxFiche = new ContextePage("");
						try {
							supprimerRubriquePublication(ctxFiche, codeObjet, arrayRubriquePublication);
						} catch (final Exception e) {
							logger.error("Exception lors de la suppression des rubriques de publication", e);
						} finally {
							ctxFiche.release();
						}
					} catch (final Exception e) {
						logger.error("Exception sur le traitement des fiches " + ReferentielObjets.getLibelleObjet(codeObjet), e);
					} finally {
						ctx.release();
					}
				}// fin for
			} catch (final Exception e) {
				logger.error("Ereur dans le traitement de la liste d'objet", e);
			}
			ctx = new ContextePage("");
			try {
				// synchronisation du référencement automatique
				logger.info("***** Rubriquage automatique des fiches *****");
				rubriquerFiches(ctx);
				// archivage des demandes de mot de passe
				logger.info("***** Archivage des demandes de mot de passe *****");
				archiverDemandesMotPasse();
				// purge fichiergw non associés à une fiche ou autre etat=0 ou code parent = ""
				// et purge fichiers physiques techniques fiches supprimées
				logger.info("***** Purge des fichiers joints *****");
				purgeFichiersgw(ctx);
			} catch (final Exception e) {
				logger.error("Echec maintenance", e);
			} finally {
				ctx.release();
			}
		} catch (final Exception e) {
			logger.error("Erreur d'execution du scansite : ", e);
		} finally {
			// suppression du contexte du thread local
			ContexteUtil.releaseContexteUniv();
		}
	}

	/**
	 * Suppression des méta-tags non référencés c'est à dire pour lequels la fiche correspondante a été supprimée.
	 *
	 * @param ctx            the _ctx
	 * @param codeObjet      the _code objet
	 * @param listeExistante the _liste existante
	 * @throws Exception the exception
	 */
	public void supprimerMetaTag(final OMContext ctx, final String codeObjet, final ArrayList<Long> listeExistante) throws Exception {
		if (listeExistante.size() > 0) {
			final StringBuilder sbListeIdFiches = new StringBuilder("");
			final Metatag metatag = new Metatag();
			metatag.init();
			metatag.setCtx(ctx);
			final String requete = "WHERE META_CODE_OBJET='" + codeObjet + "'";
			// on parcourt tous les metatag d'abord, et on recupere les id qui
			// n'existent plus
			ctx.getDatas().put("optimizedSelect", "1");
			metatag.selectNoCount(requete);
			while (metatag.nextItem()) {
				if (!listeExistante.contains(metatag.getMetaIdFiche())) {
					sbListeIdFiches.append(",'").append(metatag.getMetaIdFiche()).append("'");
				}
			}
			ctx.getDatas().remove("optimizedSelect");
			if (sbListeIdFiches.length() > 0) {
				// on supprime la première virgule de la liste des id
				sbListeIdFiches.deleteCharAt(0);
				final String query = "DELETE FROM METATAG WHERE META_CODE_OBJET='" + codeObjet + "' AND META_ID_FICHE IN (" + sbListeIdFiches.toString() + ")";
				try (PreparedStatement _stmt = ctx.getConnection().prepareStatement(query)) {
					_stmt.executeUpdate();
				} catch (final SQLException exc) {
					throw new Exception("ERROR_IN_METHOD supprimerMetatag() " + exc);
				}
			}
		}
	}

	/**
	 * Suppression des rubriques de publication non référencées c'est à dire pour lequels la fiche correspondante a été supprimée.
	 *
	 * @param ctx            the ctx
	 * @param codeObjet      the _code objet
	 * @param listeExistante the _liste existante
	 * @throws Exception the exception
	 */
	public void supprimerRubriquePublication(final OMContext ctx, final String codeObjet, final ArrayList<String> listeExistante) throws Exception {
		if (listeExistante.size() > 0) {
			final StringBuilder sbListeIdFiches = new StringBuilder("");
			final Rubriquepublication rubpub = new Rubriquepublication();
			rubpub.init();
			rubpub.setCtx(ctx);
			final String requete = "WHERE TYPE_FICHE_ORIG='" + codeObjet + "'";
			// on parcourt tous les metatag d'abord, et on recupere les id qui
			// n'existent plus
			ctx.getDatas().put("optimizedSelect", "1");
			rubpub.selectNoCount(requete);
			while (rubpub.nextItem()) {
				final String rubpubExistante = rubpub.getCodeFicheOrig() + separateurLangue + rubpub.getLangueFicheOrig();
				if (!listeExistante.contains(rubpubExistante)) {
					sbListeIdFiches.append(",'").append(rubpubExistante).append("'");
				}
			}
			ctx.getDatas().remove("optimizedSelect");
			if (sbListeIdFiches.length() > 0) {
				// on supprime la premiere virgule
				sbListeIdFiches.deleteCharAt(0);
				final String query = "DELETE FROM RUBRIQUEPUBLICATION WHERE TYPE_FICHE_ORIG='" + codeObjet + "' AND CONCAT(CODE_FICHE_ORIG,'" + separateurLangue + "',LANGUE_FICHE_ORIG) IN (" + sbListeIdFiches.toString() + ")";
				try (PreparedStatement stmt = ctx.getConnection().prepareStatement(query)) {
					stmt.executeUpdate();
				} catch (final SQLException exc) {
					throw new Exception("ERROR_IN_METHOD supprimerRubriquePublication() " + exc);
				}
			}
		}
	}

	/**
	 *
	 * @param ctx le contexte ne sert pas
	 * @throws Exception
	 * @deprecated Utiliser la méthode {@link ScanSite#archiverDemandesMotPasse()}
	 */
	@Deprecated
	public void archiverDemandesMotPasse(final OMContext ctx) throws Exception {
		archiverDemandesMotPasse();
	}

	/**
	 * Archivage des demandes de mot de passe ayant + de 1 jour.
	 *
	 * @throws Exception lors de la requête en BDD
	 */
	private void archiverDemandesMotPasse() throws Exception {
		final GregorianCalendar dateLimite = new GregorianCalendar();
		dateLimite.add(Calendar.DATE, -1);
		try (ContexteDao ctx = new ContexteDao()) {
			final PreparedStatement stmt = ctx.getConnection().prepareStatement("DELETE FROM CHANGEMENTMOTPASSE WHERE DATE_DEMANDE < ?");
			stmt.setDate(1, new Date(dateLimite.getTimeInMillis()));
			stmt.executeUpdate();
			stmt.close();
		} catch (final SQLException exc) {
			throw new Exception("ERROR_IN_METHOD ArchiverDemandesMotsDePasse " + exc);
		}
	}

	/**
	 * Purge des fichiers techniques qui ne sont plus référencées dans les fiches ou autre.
	 *
	 * @param ctx the ctx
	 */
	public void purgeFichiersgw(final OMContext ctx) {
		try {
			// Nettoie la table fichiergw
			final Ressource ressource = new Ressource();
			ressource.setCtx(ctx);
			ressource.init();
			final Media media = new Media();
			media.setCtx(ctx);
			media.init();
			int nbFichiersNonAssocies = 0;
			// etat=0 : non remonté ; etat=1 utilisé ; etat=2 a supprimer
			if (ressource.select("WHERE ETAT='0' OR ETAT='2' OR CODE_PARENT=''") > 0) {
				while (ressource.nextItem()) {
					logger.info("Suppression de la ressource d'id " + ressource.getIdRessource() + " ayant pour fichier source " + ressource.getSource());
					ressource.delete();
					nbFichiersNonAssocies++;
				}
			}
			logger.info(nbFichiersNonAssocies + " tuple(s) fichiergw supprimé(s) ");
			logger.info("***** Purge des fichiers techniques non référencés *****");
			final Set<String> tree = new TreeSet<>();
			/* Recherche de tous les fichiers joints référencés */
			if (media.select("") > 0) {
				while (media.nextItem()) {
					tree.add(media.getUrl());
					if (media.getUrlVignette().length() > 0) {
						tree.add(media.getUrlVignette());
					}
				}
			}
			/* Nettoyage des répertoires de stockage */
			// répertoire privé des fichiergw
			int nbFichiersNonReferences = 0;
			final File dir = new File(RessourceUtil.getAbsolutePath());
			nbFichiersNonReferences = nettoyerRepertoire(dir, tree);
			// répertoire public de la médiathèque
			// nettoyage des dossiers mediathéque.
			nbFichiersNonReferences += nettoyerMediatheque(tree);
			logger.info(nbFichiersNonReferences + " fichier(s) techniques supprimés ");
		} catch (final Exception e) {
			logger.error("Exception pendant la purge des fichiers : methode purgeFichiersgw()", e);
		}
	}

	/**
	 * Nettoyer repertoire.
	 *
	 * @param repertoire         the repertoire
	 * @param fichiersReferences the liste
	 * @return the int
	 */
	public int nettoyerRepertoire(final File repertoire, final Set<String> fichiersReferences) {
		int nbFichiersNonReferences = 0;
		if (repertoire.isDirectory()) {
			final File[] list = repertoire.listFiles();
			if (list != null) {
				for (final File fichierASupprimer : list) {
					// Appel récursif sur les sous-répertoires
					if (fichierASupprimer.isDirectory()) {
						nbFichiersNonReferences = nbFichiersNonReferences + nettoyerRepertoire(fichierASupprimer, fichiersReferences);
					} else if (!fichiersReferences.contains(fichierASupprimer.getName())) {
						if (fichierASupprimer.delete()) {
							logger.info("Suppression fichier " + repertoire.getAbsolutePath() + File.separator + fichierASupprimer.getName());
						} else {
							logger.error("Erreur de suppression du fichier " + repertoire.getAbsolutePath() + File.separator + fichierASupprimer.getName());
						}
						nbFichiersNonReferences++;
					}
				}
			}
		}
		return nbFichiersNonReferences;
	}

	/**
	 *
	 * @param ctx
	 * @param codeObjet
	 * @param ficheUniv
	 * @param mailbox
	 * @deprecated Méthode remplacée par sa version privée
	 */
	@Deprecated
	public void genererAlerte(final OMContext ctx, final String codeObjet, final FicheUniv ficheUniv, final JSBMailbox mailbox) {
		genererAlerte(codeObjet, ficheUniv);
	}

	/**
	 * Génération d'un page statique.
	 *
	 * @param codeObjet the _code objet
	 * @param ficheUniv the fiche univ
	 */
	private void genererAlerte(final String codeObjet, final FicheUniv ficheUniv) {
		try {
			if ((Formateur.estSaisie(ficheUniv.getDateAlerte()) && ficheUniv.getDateAlerte().compareTo(new Date(System.currentTimeMillis())) <= 0)) {
				final String mail = Utilisateur.getAdresseMail(ficheUniv.getCodeRedacteur());
				if (mail.length() > 0) {
					final String libelle = ReferentielObjets.getLibelleObjet(codeObjet);
					// recup de l'url du site web
					final String url_site = URLResolver.getAbsoluteBoUrl("/adminsite/", Site.getSitePrincipal());
					final StringBuilder message = new StringBuilder();
					message.append("Fiche concernée : fiche ").append(libelle).append(" intitulée ").append(ficheUniv.getLibelleAffichable()).append(" (code : ").append(ficheUniv.getCode()).append(").\n\n");
					message.append("Message de l'alerte : \n");
					message.append(ficheUniv.getMessageAlerte()).append("\n\n");
					message.append("Pour ne plus recevoir cette alerte, supprimez ou modifiez la date d'alerte de la fiche \n");
					message.append("dans le module d'administration du site web (").append(url_site).append(")\n\n");
					message.append("Cordialement.");
					final JSBMailbox mailbox = new JSBMailbox(false);
					mailbox.sendSystemMsg(mail, "Alerte site web", message.toString());
				}
			}
		} catch (final Exception e) {
			logger.error(String.format("Exception pendant génération alerte objet : %s code : %s langue : %s", codeObjet, ficheUniv.getCode(), ficheUniv.getLangue()), e);
		}
	}

	/**
	 * Cette méthode assigne des rubriques de publication de façon automatique.
	 *
	 * @param ctx the ctx
	 */
	public void rubriquerFiches(final OMContext ctx) {
		final Rubrique rubrique = new Rubrique();
		rubrique.init();
		rubrique.setCtx(ctx);
		final Rubriquepublication rubPub = new Rubriquepublication();
		rubPub.init();
		rubPub.setCtx(ctx);
		Metatag meta = new Metatag();
		meta.init();
		meta.setCtx(ctx);
		final TreeSet<InfosFicheReferencee> listeFichesAIndexer = new TreeSet<>(new InfosFicheComparator());
		try {
			// recherche des requêtes de publication automatique sur les
			// rubriques
			if (rubrique.select("WHERE REQUETES_RUBRIQUE_PUBLICATION != ''") > 0) {
				// boucle sur les rubriques
				while (rubrique.nextItem()) {
					boolean rubriqueModifiee = false;
					String messageInvalide = "";
					final Vector<String> vRequete = Chaine.getVecteurAccolades(rubrique.getRequetesRubriquePublication());
					final Iterator<String> it = vRequete.iterator();
					String codeObjet = "";
					// boucle sur les requetes
					while (it.hasNext()) {
						String requete = it.next();
						logger.debug("requete de multi-publication : " + requete);
						final String[] item = requete.split("/", -2);
						boolean requeteValide = true;
						boolean rechercheRubriqueRecursive = false;
						boolean rechercheStructureRecursive = false;
						// on vérifie que la rubrique est valide
						if (item[0].length() > 0) {
							codeObjet = ReferentielObjets.getCodeObjet(item[0]);
							if ("".equals(codeObjet)) {
								messageInvalide = "requete invalide : objet non trouve " + item[0];
								requeteValide = false;
							}
						}
						String codeStructure = item[1];
						if (codeStructure.length() > 0) {
							final String[] temp = codeStructure.split(";", -2);
							codeStructure = temp[0];
							if (!Structure.renvoyerItemStructure(codeStructure).getCode().equals(codeStructure)) {
								messageInvalide = "requete invalide : structure non trouvee " + codeStructure;
								requeteValide = false;
							} else if ("1".equals(temp[1])) {
								rechercheStructureRecursive = true;
							}
						}
						String codeRubrique = item[2];
						if (codeRubrique.length() > 0) {
							final String[] temp = codeRubrique.split(";", -2);
							codeRubrique = temp[0];
							if (!Rubrique.renvoyerItemRubrique(temp[0]).getCode().equals(codeRubrique)) {
								messageInvalide = "requete invalide : rubrique non trouvee " + codeRubrique;
								requeteValide = false;
							} else if ("1".equals(temp[1])) {
								rechercheRubriqueRecursive = true;
							}
						}
						String langue = "";
						if (item.length > 4) {
							langue = item[3];
							if (LangueUtil.getLocale(langue) == null) {
								messageInvalide = "requete invalide : langue non trouvee " + langue;
								requeteValide = false;
							}
						}
						// requete valide
						if (requeteValide) {
							final TreeSet<InfosFicheReferencee> listeFicheAvantSynchro = Rubriquepublication.supprimerRubriquePublicationAutomatique(ctx, rubrique.getCode(),
								requete, false);
							logger.debug("nb de fiches avant la synchro : " + listeFicheAvantSynchro.size());
							// requête sur la table méta
							final ClauseWhere whereFiche = new ClauseWhere();
							if (codeObjet.length() > 0) {
								whereFiche.setPremiereCondition(ConditionHelper.egalVarchar("META_CODE_OBJET", codeObjet));
							}
							if (codeStructure.length() > 0) {
								if (rechercheStructureRecursive) {
									final ConditionList conditionsStructure = new ConditionList(ConditionHelper.getConditionStructure("META_CODE_RATTACHEMENT", codeStructure));
									conditionsStructure.or(ConditionHelper.getConditionStructureMultiple("META_CODE_RATTACHEMENT_AUTRES", codeStructure));
									whereFiche.and(conditionsStructure);
								} else {
									final ConditionList conditionsStructure = new ConditionList(ConditionHelper.egalVarchar("META_CODE_RATTACHEMENT", codeStructure));
									conditionsStructure.or(ConditionHelper.like("META_CODE_RATTACHEMENT_AUTRES", codeStructure, "%[", "]%"));
									whereFiche.and(conditionsStructure);
								}
							}
							if (codeRubrique.length() > 0) {
								if (rechercheRubriqueRecursive) {
									whereFiche.and(ConditionHelper.getConditionRubrique("META_CODE_RUBRIQUE", codeRubrique));
								} else {
									whereFiche.and(ConditionHelper.egalVarchar("META_CODE_RUBRIQUE", codeRubrique));
								}
							}
							if (langue.length() > 0) {
								whereFiche.and(ConditionHelper.egalVarchar("META_LANGUE", langue));
							}
							// les rubriques de publication ne concernent que
							// les fiches en ligne
							whereFiche.and(ConditionHelper.egalVarchar("META_ETAT_OBJET", EtatFiche.EN_LIGNE.getEtat()));
							// enregistrement des fiches
							final int count = meta.select(whereFiche.formaterSQL());
							logger.debug("nb de fiches correspondant à la requête :" + count);
							if (count > 0) {
								while (meta.nextItem()) {
									final InfosFicheReferencee infosFiche = new InfosFicheReferencee();
									infosFiche.setCode(meta.getMetaCode());
									infosFiche.setLangue(meta.getMetaLangue());
									infosFiche.setType(meta.getMetaCodeObjet());
									infosFiche.setRequete(requete);
									if (!listeFicheAvantSynchro.contains(infosFiche)) {
										logger.debug("ajout à indexer de :" + infosFiche.getCode() + " " + infosFiche.getType());
										final Rubriquepublication rubP = new Rubriquepublication();
										rubP.init();
										rubP.setCtx(ctx);
										rubP.setTypeFicheOrig(infosFiche.getType());
										rubP.setCodeFicheOrig(infosFiche.getCode());
										rubP.setLangueFicheOrig(infosFiche.getLangue());
										rubP.setRubriqueDest(rubrique.getCode());
										rubP.setSourceRequete(requete);
										rubP.add();
										listeFichesAIndexer.add(infosFiche);
									} else {
										listeFicheAvantSynchro.remove(infosFiche);
									}
								}
							}
							// suppression des fiches obsoletes
							if (listeFicheAvantSynchro.size() > 0) {
								for (InfosFicheReferencee infosFiche : listeFicheAvantSynchro) {
									Rubriquepublication.supprimerRubriquePublicationFiche(ctx, infosFiche, true);
									logger.debug("suite à suppression, ajout à indexer de :" + infosFiche.getCode() + " " + infosFiche.getType());
									listeFichesAIndexer.add(infosFiche);
								}
							}
						}
						// requete non valide on supprime la rubrique de
						// publication
						else {
							listeFichesAIndexer.addAll(Rubriquepublication.supprimerRubriquePublicationAutomatique(ctx, rubrique.getCode(), requete, true));
							// vRequete.remove(requete);
							logger.warn("Probleme sur une requete automatique definie pour la rubrique " + rubrique.getIntitule() + " : " + messageInvalide + ", la requete est ignoree");
							rubriqueModifiee = true;
						}
						logger.debug("nb de fiches à indexer : " + listeFichesAIndexer.size());
					} // fin while sur requete
					// mise à jour de la rubrique si modifiée
					if (rubriqueModifiee) {
						rubrique.update();
					}
				} // fin while sur rubrique
				logger.info("re-indexation des nouvelles fiches rubriquées, nb de fiches à ré-indexer : " + listeFichesAIndexer.size());
				// réindexation de toutes les fiches modifiées
				for (InfosFicheReferencee infosFiche : listeFichesAIndexer) {
					// Lecture de la fiche
					final FicheUniv ficheAreindexee = ReferentielObjets.instancierFiche(ReferentielObjets.getNomObjet(infosFiche.getType()));
					ficheAreindexee.init();
					ficheAreindexee.setCtx(ctx);
					if (ficheAreindexee.selectCodeLangueEtat(infosFiche.getCode(), infosFiche.getLangue(), "0003") > 0) {
						ficheAreindexee.nextItem();
						ficheAreindexee.setDateProposition(new Date(System.currentTimeMillis()));
						ficheAreindexee.update();
						meta = FicheUnivMgr.lireMeta(ficheAreindexee);
						meta.setMetaDateProposition(ficheAreindexee.getDateProposition());
						FicheUnivMgr.enregistrerMeta(meta);
						PluginFicheHelper.synchroniserObjets(ficheAreindexee, meta, null);
					}
				}
			}
		} catch (final Exception e) {
			logger.error("Exception pendant le référencement des fiches : methode rubriquerFiches()", e);
		}
	}

	/**
	 * Remove all of the content of a directory .
	 *
	 * @param toRemove Directory or file to be removed
	 * @return false if failed
	 */
	protected boolean remove(final File toRemove) {
		boolean result = true;
		if (toRemove.isDirectory()) {
			final String list[] = toRemove.list();
			for (final String element : list) {
				final File file = new File(toRemove, element);
				if (file.isDirectory()) {
					result = remove(file);
				} else {
					if (!file.delete()) {
						result = false;
					}
				}
			}
		}
		return result;
	}

	@Override
	public void perform() {
		run();
	}

	/**
	 * Supprimer l'ensemble des fichiers de la médiathéque qui n'existent plus dans la base de données. Cette fonction tient compte des spécificité de la médiathéque à savoir : un
	 * dossier par type de média et pas de dossiers dans les dossiers de types de média.
	 *
	 * @param listeNomsFichersAGarder La liste des nom de fichier à garder.
	 * @return le nombre de fichiers supprimés.
	 * @author pierre.cosson (18/06/2010)
	 */
	public int nettoyerMediatheque(final Set<String> listeNomsFichersAGarder) {
		final File dossierMedias = new File(MediathequeHelper.getAbsolutePath());
		if (dossierMedias.exists() && dossierMedias.isDirectory()) {
			// Sous le dossier MEDIA un dossier apr type de média
			final File[] listeDossiersTypeMedia = dossierMedias.listFiles();
			int nbFichiersSupprimes = 0;
			for (final File enCours : listeDossiersTypeMedia) {
				if (enCours.isDirectory()) {
					// cas d'un dossier de type de média (on va le parcourir)
					File[] listeMediasTypes = enCours.listFiles();
					for (final File listeMediasType : listeMediasTypes) {
						File mediaType = listeMediasType;
						if (!mediaType.isDirectory() && !listeNomsFichersAGarder.contains(mediaType.getName())) {
							// on ne supprime pas les dossiers
							mediaType.delete();
							logger.info("Suppression fichier " + mediaType.getAbsolutePath());
							nbFichiersSupprimes++;
						}
					}
				} else if (!listeNomsFichersAGarder.contains(enCours.getName())) {
					// fichier qui n'est pas la liste des fichiers à garder
					enCours.delete();
					logger.info("Suppression fichier " + enCours.getAbsolutePath());
					nbFichiersSupprimes++;
				}
			}
			return nbFichiersSupprimes;
		}
		return 0;
	}

	/**
	 * Supprimer tous les éléments contenu dans un dossier (cette fonction supprime tous les fichiers et dossier)
	 *
	 * @param dossierTemporaire le dossier à vider
	 */
	private void removeContenuDossier(final File dossierTemporaire) {
		if (dossierTemporaire.exists() && dossierTemporaire.isDirectory()) {
			final File[] listeFichiersASuppr = dossierTemporaire.listFiles();
			for (final File element : listeFichiersASuppr) {
				File fichierASuppr = element;
				if (fichierASuppr.isDirectory()) {
					try {
						FileUtils.deleteDirectory(fichierASuppr);
					} catch (final Exception e) {
						// erreur lors de la suppression du dossier
						logger.error("Erreur lors de la suppression du dossier : " + fichierASuppr.getAbsolutePath(), e);
					}
				} else {
					fichierASuppr.delete();
				}
			}
		}
	}
}
