/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.batch.process;

import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;

import com.univ.objetspartages.om.*;
import org.slf4j.Logger;

import com.kportal.scheduling.spring.quartz.LogReportJob;
import com.univ.utils.RequeteGroupeUtil;

/**
 * Classe qui peut être lancée par un batch dans le but de synchroniser les groupes dynamiques. Pour effectuer cette synchronisation on parcours tous les types de groupes
 * dynamiques en ne prennant que ceux synchronisables et ayant l'option d'auto-synchronisation mise à la valeur 1 (cette option est à complété dans le JTF :
 * <strong>requete_groupe.NOM_REQUTE.auto_synchronisation</strong>).<br>
 * Pour fonctionner cette classe lit le JTF puis instancie les classes des groupes dynamiques pour les synchroniser. Tous les groupes de la requete seront instanciés.
 * 
 * @see RequeteGroupeDynamique
 * @see RequeteGroupeDynamiqueSynchronisable
 * @author Pierre Cosson
 */
public class SynchroniserGroupesDynamique extends LogReportJob implements Runnable {

	/**
	 * Fonction main lancée par un batch.
	 * 
	 * @param args
	 *            Les arguments.
	 */
	public static void main(final String[] args) {
		final SynchroniserGroupesDynamique synchroniserGroupesDynamique = new SynchroniserGroupesDynamique();
		final Thread thread = (new Thread(synchroniserGroupesDynamique));
		thread.start();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Runnable#run()
	 */
	@Override
	public void run() {
        // parcours de toutes les requetes de groupes dynamique
        final Hashtable<String, String> groups = Groupedsi.getListeRequetesGroupesPourAffichage();
        for (String key : groups.keySet()) {
            final InfosRequeteGroupe infosRequete = Groupedsi.renvoyerItemRequeteGroupe(key);
            final RequeteGroupeDynamique dynGroupe = RequeteGroupeUtil.instancierRequete(infosRequete);
            // on ne synchronise que les groupes qui sont synchronisables et
            // qui ont l'option de synchronisation automatique d'activée.
            if (dynGroupe != null && dynGroupe instanceof RequeteGroupeDynamiqueSynchronisable && ((RequeteGroupeDynamiqueSynchronisable) dynGroupe).isAutoSynchronise()) {
                logger.info("   --- DEBUT de la synchronisation " + "de la requete : " + key);
                // Synchronisation du groupe
                final SynchronisationGroupeDynamiqueEtat etatSync = ((RequeteGroupeDynamiqueSynchronisable) dynGroupe).synchronize();
                // affichage du résultat de la requête dans le logger
                if (etatSync.isValide()) {
                    this.displayLogSynchronisation(etatSync, logger);
                } else {
                    logger.info("          Erreur durant la synchronisation ");
                }
                logger.info("   --- FIN de la synchronisation" + " de la requete : " + key);
            }
        }
	}

	/**
	 * Display logger synchronisation.
	 * 
	 * @param etatSync
	 *            the etat sync
	 * @param logger
	 *            the logger
	 */
	public void displayLogSynchronisation(final SynchronisationGroupeDynamiqueEtat etatSync, final Logger logger) {
		final Collection<String> tmGroupesAjoutes = etatSync.getLibellesGroupeSynchroAjoutes();
		String groupesAjoutes = "";
		if (!tmGroupesAjoutes.isEmpty()) {
			final Iterator<String> itGroupesAjoutes = tmGroupesAjoutes.iterator();
			while (itGroupesAjoutes.hasNext()) {
				if (!groupesAjoutes.equals("")) {
					groupesAjoutes += ", ";
				}
				groupesAjoutes += itGroupesAjoutes.next();
			}
		}
		final Collection<String> tmGroupesSupprimes = etatSync.getLibellesGroupeSynchroSupprimes();
		String groupesSupprimes = "";
		if (!tmGroupesSupprimes.isEmpty()) {
			final Iterator<String> itGroupesSupprimes = tmGroupesSupprimes.iterator();
			while (itGroupesSupprimes.hasNext()) {
				if (!groupesSupprimes.equals("")) {
					groupesSupprimes += ", ";
				}
				groupesSupprimes += itGroupesSupprimes.next();
			}
		}
		final Collection<String> tmGroupesModifies = etatSync.getLibellesGroupeSynchroModifies();
		String groupesModifies = "";
		if (!tmGroupesModifies.isEmpty()) {
			final Iterator<String> itGroupesModifies = tmGroupesModifies.iterator();
			while (itGroupesModifies.hasNext()) {
				if (!groupesModifies.equals("")) {
					groupesModifies += ", ";
				}
				groupesModifies += itGroupesModifies.next();
			}
		}
		logger.info("          nombre de groupes dynamiques ajoutés : " + etatSync.getNbGroupesAjoutes());
		if (!groupesAjoutes.equals("")) {
			logger.info("          liste des groupes dynamiques ajoutés : " + groupesAjoutes);
		}
		logger.info("          nombre de groupes dynamiques modifiés : " + etatSync.getNbGroupesModifies());
		if (!groupesModifies.equals("")) {
			logger.info("          liste des groupes dynamiques modifiés : " + groupesModifies);
		}
		logger.info("          nombre de groupes dynamiques supprimés : " + etatSync.getNbGroupesSupprimes());
		if (!groupesSupprimes.equals("")) {
			logger.info("          liste des groupes dynamiques supprimés : " + groupesSupprimes);
		}
	}

	@Override
	public void perform() {
		run();
	}
}
