/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.collaboratif.dao.impl;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;

import com.jsbsoft.jtf.datasource.dao.impl.mysql.AbstractCommonDAO;
import com.jsbsoft.jtf.datasource.exceptions.AddToDataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.DataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.UpdateToDataSourceException;
import com.univ.collaboratif.bean.UserRolesCollaboratifBean;

/**
 * Created on 27/04/15.
 */
public class UserRolesCollaboratifDao extends AbstractCommonDAO<UserRolesCollaboratifBean> {

    public UserRolesCollaboratifDao() {
        tableName = "USER_ROLES_COLLABORATIF";
    }

    @Override
    protected UserRolesCollaboratifBean fill(ResultSet rs) throws DataSourceException {
        try {
            final UserRolesCollaboratifBean userRolesCollaboratifBean = new UserRolesCollaboratifBean();
            userRolesCollaboratifBean.setId(rs.getLong("ID_USER_ROLES_COLLABORATIF"));
            userRolesCollaboratifBean.setIdCollaboratif(rs.getLong("ID_COLLABORATIF"));
            userRolesCollaboratifBean.setCodeUser(rs.getString("CODE_USER"));
            userRolesCollaboratifBean.setCodeRole(rs.getString("CODE_ROLE"));
            return userRolesCollaboratifBean;
        } catch (SQLException e) {
            throw new DataSourceException("Une erreur est survenue lors de la récupération du bean UserRolesCollaboratifBean", e);
        }
    }

    @Override
    public UserRolesCollaboratifBean add(UserRolesCollaboratifBean userRolesCollaboratifBean) throws AddToDataSourceException {
        ResultSet rs = null;
        try (Connection connection = dataSource.getConnection();
            PreparedStatement stmt = connection.prepareStatement("INSERT INTO `USER_ROLES_COLLABORATIF` (`CODE_USER`, `CODE_ROLE`, `ID_COLLABORATIF`) VALUES (?, ?, ?)", Statement.RETURN_GENERATED_KEYS)) {
            stmt.setString(1, userRolesCollaboratifBean.getCodeUser());
            stmt.setString(2, userRolesCollaboratifBean.getCodeRole());
            stmt.setLong(3, userRolesCollaboratifBean.getIdCollaboratif());
            final int rowsAffected = stmt.executeUpdate();
            if (rowsAffected != 1) {
                throw new AddToDataSourceException(
                    String.format("Unable to add [%s] to table \"USER_ROLES_COLLABORATIF\" : %d row(s) affected", userRolesCollaboratifBean.toString(), rowsAffected));
            }
            rs = stmt.getGeneratedKeys();
            rs.next();
            userRolesCollaboratifBean.setId(rs.getLong(1));
        } catch (final Exception e) {
            throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"USER_ROLES_COLLABORATIF\"", userRolesCollaboratifBean.toString()), e);
        } finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (SQLException e) {
                    LOG.error(String.format("An error occurred closing Resultset during add() on table %s", this.tableName), e);
                }
            }
        }
        return userRolesCollaboratifBean;
    }

    @Override
    public UserRolesCollaboratifBean update(UserRolesCollaboratifBean userRolesCollaboratifBean) throws UpdateToDataSourceException {
        try (Connection connection = dataSource.getConnection();
            PreparedStatement stmt = connection.prepareStatement("UPDATE `USER_ROLES_COLLABORATIF` SET `CODE_USER` = ?, `CODE_ROLE` = ?, `ID_COLLABORATIF` = ? WHERE ID_USER_ROLES_COLLABORATIF = ?")) {
            stmt.setString(1, userRolesCollaboratifBean.getCodeUser());
            stmt.setString(2, userRolesCollaboratifBean.getCodeRole());
            stmt.setLong(3, userRolesCollaboratifBean.getIdCollaboratif());
            stmt.setLong(4, userRolesCollaboratifBean.getId());
            stmt.executeUpdate();
        } catch (Exception e) {
            throw new UpdateToDataSourceException(String.format("Unable to update [%s] from table \"USER_ROLES_COLLABORATIF\"", userRolesCollaboratifBean.toString()), e);
        }
        return userRolesCollaboratifBean;
    }

    public List<UserRolesCollaboratifBean> getByIdCollaboratif(Long id) {
        List<UserRolesCollaboratifBean> results = new ArrayList<>();
        try (final Connection connection = dataSource.getConnection();
            PreparedStatement stmt = connection.prepareStatement(String.format("SELECT * FROM `%s` WHERE ID_COLLABORATIF = ?", tableName))) {
            stmt.setLong(1, id);
            try (ResultSet rs = stmt.executeQuery()) {
                while (rs.next()) {
                    results.add(fill(rs));
                }
            }
        } catch (final SQLException e) {
            throw new DataSourceException(String.format("An error occurred retrieving objects from table %s", tableName), e);
        }
        return results;
    }

    public List<UserRolesCollaboratifBean> getByCodeUser(String code) {
        List<UserRolesCollaboratifBean> results = new ArrayList<>();
        try (final Connection connection = dataSource.getConnection();
            PreparedStatement stmt = connection.prepareStatement(String.format("SELECT * FROM `%s` WHERE CODE_USER = ?", tableName))) {
            stmt.setString(1, code);
            try (ResultSet rs = stmt.executeQuery()) {
                while (rs.next()) {
                    results.add(fill(rs));
                }
            }
        } catch (final SQLException e) {
            throw new DataSourceException(String.format("An error occurred retrieving objects from table %s", tableName), e);
        }
        return results;
    }

    public List<UserRolesCollaboratifBean> getByCodeRoleAndIdCollab(final String codeRole, final long idEspacecollaboratif) {
        List<UserRolesCollaboratifBean> results = new ArrayList<>();
        try (Connection connection = dataSource.getConnection();
            PreparedStatement stmt = connection.prepareStatement(String.format("SELECT * FROM `%s` WHERE CODE_ROLE = ? AND ID_COLLABORATIF = ?", tableName))) {
            stmt.setString(1, codeRole);
            stmt.setLong(2, idEspacecollaboratif);
            try (ResultSet rs = stmt.executeQuery()) {
                while (rs.next()) {
                    results.add(fill(rs));
                }
            }
        } catch (final SQLException e) {
            throw new DataSourceException(String.format("An error occurred retrieving objects from table %s", tableName), e);
        }
        return results;
    }

    public void deleteByIdCollaboratif(final Long idEspacecollaboratif) {
        try (Connection connection = dataSource.getConnection();
            PreparedStatement stmt = connection.prepareStatement(String.format("DELETE FROM `%s` WHERE ID_COLLABORATIF = ?", tableName))) {
            stmt.setLong(1, idEspacecollaboratif);
            stmt.execute();
        } catch (final SQLException e) {
            throw new DataSourceException(String.format("An error occurred retrieving objects from table %s", tableName), e);
        }
    }

    public UserRolesCollaboratifBean getByCodeUserAndIdCollab(final String codeUser, final Long idEspacecollaboratif) {
        UserRolesCollaboratifBean bean = null;
        try (Connection connection = dataSource.getConnection();
            PreparedStatement stmt = connection.prepareStatement(String.format("SELECT * FROM `%s` WHERE CODE_USER = ? AND ID_COLLABORATIF = ?", tableName))) {
            stmt.setString(1, codeUser);
            stmt.setLong(2, idEspacecollaboratif);
            try (ResultSet rs = stmt.executeQuery()) {
                if (rs.next()) {
                    bean = fill(rs);
                }
            }
        } catch (final SQLException e) {
            throw new DataSourceException(String.format("An error occurred retrieving objects from table %s", tableName), e);
        }
        return bean;
    }

    public void deleteByCodeUserAndIdCollab(String codeMembre, Long idEspacecollaboratif) {
        try (Connection connection = dataSource.getConnection();
            PreparedStatement stmt = connection.prepareStatement(String.format("DELETE FROM `%s` WHERE CODE_USER = ? AND ID_COLLABORATIF = ?", tableName))) {
            stmt.setString(1, codeMembre);
            stmt.setLong(2, idEspacecollaboratif);
            stmt.execute();
        } catch (final SQLException e) {
            throw new DataSourceException(String.format("An error occurred retrieving objects from table %s", tableName), e);
        }
    }

    public void deleteByCodeUser(String codeMembre) {
        try (Connection connection = dataSource.getConnection();
            PreparedStatement stmt = connection.prepareStatement(String.format("DELETE FROM `%s` WHERE CODE_USER = ?", tableName))) {
            stmt.setString(1, codeMembre);
            stmt.execute();
        } catch (final SQLException e) {
            throw new DataSourceException(String.format("An error occurred retrieving objects from table %s", tableName), e);
        }
    }
}
