/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.collaboratif.service;

import java.util.ArrayList;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.kportal.cache.CacheUtil;
import com.univ.collaboratif.om.Espacecollaboratif;
import com.univ.collaboratif.om.InfosEspaceCollaboratif;
import com.univ.utils.Chaine;
import com.univ.utils.ContexteDao;

/**
 * Cache manager for the espaces collaboratifs
 */
public class CacheEspaceManager {

    public static final String ID_BEAN = "cacheEspaceManager";

    public static final String CACHE_NAME = "CacheEspaceManager.cacheEspaces";

    private static final Logger LOG = LoggerFactory.getLogger(CacheEspaceManager.class);

    /**
     * Gets the single instance of CacheEspaceManager.
     *
     * @return single instance of CacheEspaceManager
     */
    public static CacheEspaceManager getInstance() {
        return (CacheEspaceManager) ApplicationContextManager.getBean(Espacecollaboratif.ID_EXTENSION, ID_BEAN);
    }

    /**
     * Récupère l'info espace dans le cache, en le mettant à jour si nécessaire
     * @param code Le code de l'espace souhaité
     * @return Le bean infosEspace souhaité
     */
    public InfosEspaceCollaboratif getItem(String code) {
        InfosEspaceCollaboratif infosEspace = (InfosEspaceCollaboratif) CacheUtil.getObjectValue(CACHE_NAME, code);
        if (infosEspace == null) {
            final Espacecollaboratif espacecollaboratif = new Espacecollaboratif();
            try (ContexteDao ctx = new ContexteDao()) {
                espacecollaboratif.setCtx(ctx);
                espacecollaboratif.init();
                espacecollaboratif.selectByCode(code);
                if (espacecollaboratif.nextItem()) {
                    infosEspace = new InfosEspaceCollaboratif(
                            espacecollaboratif.getIdEspacecollaboratif(),
                            espacecollaboratif.getCode(),
                            espacecollaboratif.getIntitule(),
                            espacecollaboratif.getGroupesMembres(),
                            espacecollaboratif.getGroupesConsultation(),
                            espacecollaboratif.getGroupesInscription(),
                            espacecollaboratif.getInscriptionFront(),
                            espacecollaboratif.getRolesMembre(),
                            espacecollaboratif.getTheme(),
                            espacecollaboratif.getDescription(),
                            espacecollaboratif.getCodeRubrique(),
                            espacecollaboratif.getLangue(),
                            "1".equals(espacecollaboratif.getActif()),
                            new ArrayList<>(Chaine.getVecteurPointsVirgules(espacecollaboratif.getServices())));
                    CacheUtil.updateObjectValue(CACHE_NAME, code, infosEspace);
                }
            } catch (final Exception e) {
                LOG.error("Problème de récupération de l'espace " + code, e);
            }
        }
        return infosEspace;
    }

    /**
     * Flush an espace cache item
     * @param code The code of the espace to flush
     */
    public void flushItem(String code) {
        CacheUtil.flush(CACHE_NAME, code);
    }
}
