/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.datagrid.processus;

import static org.apache.commons.lang3.StringUtils.containsIgnoreCase;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.collections.Predicate;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.session.SessionUtilisateur;
import com.kportal.extension.module.composant.ComposantLibelle;
import com.univ.datagrid.bean.ResultatDatagrid;
import com.univ.datagrid.bean.ResultatLibelle;
import com.univ.datagrid.bean.ResultatLibelle.LibelleParLangue;
import com.univ.datagrid.utils.DatagridUtils;
import com.univ.objetspartages.cache.CacheLibelleManager;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.Libelle;
import com.univ.utils.ContexteUtil;
import com.univ.utils.SessionUtil;
import com.univ.utils.sql.RequeteSQL;
import com.univ.utils.sql.clause.ClauseOrderBy;
import com.univ.utils.sql.clause.ClauseOrderBy.SensDeTri;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;

public class LibelleDatagrid extends AbstractServiceDatagrid {

	public static final String ID_BEAN = "libelleDatagrid";

	private static final Logger LOG = LoggerFactory.getLogger(LibelleDatagrid.class);

	private final CacheLibelleManager cache = (CacheLibelleManager) ApplicationContextManager.getCoreContextBean(CacheLibelleManager.ID_BEAN);

	@Override
	public List<ResultatDatagrid> traiterRechercheDepuisRequete(final HttpServletRequest req) {
		final AutorisationBean autorisations = (AutorisationBean) SessionUtil.getInfosSession(req).get(SessionUtilisateur.AUTORISATIONS);
		if (isPermissionNonValide(autorisations)) {
			return Collections.<ResultatDatagrid> emptyList();
		}
		List<ResultatDatagrid> resultats = null;
		final RequeteSQL requete = constuireRequete(req);
		final Libelle libelleCourant = new Libelle();
		libelleCourant.init();
		libelleCourant.setCtx(ContexteUtil.getContexteUniv());
		try {
			libelleCourant.selectNoCount(requete.formaterRequete());
			resultats = mapperResultatRechercheDepuisLibelle(libelleCourant);
		} catch (final Exception e) {
			LOG.error("erreur de traitement de datagrid", e);
		}
		return resultats;
	}

	private RequeteSQL constuireRequete(final HttpServletRequest req) {
		final RequeteSQL requete = new RequeteSQL();
		final ClauseWhere where = new ClauseWhere();
		final String libelle = req.getParameter("LIBELLE");
		if (StringUtils.isNotBlank(libelle)) {
			where.setPremiereCondition(ConditionHelper.egalVarchar("LIBELLE", libelle));
		}
		final String code = req.getParameter("CODE");
		if (StringUtils.isNotBlank(code)) {
			where.and(ConditionHelper.egalVarchar("CODE", code));
		}
		final String type = req.getParameter("TYPE");
		if (StringUtils.isNotBlank(type) && !"0000".equals(type)) {
			where.and(ConditionHelper.egalVarchar("TYPE", type));
		}
		final String langue = req.getParameter("LANGUE");
		if (StringUtils.isNotBlank(langue) && !"0000".equals(langue)) {
			where.and(ConditionHelper.egalVarchar("LANGUE", langue));
		}
		final ClauseOrderBy orderBy = new ClauseOrderBy();
		orderBy.orderBy("TYPE", SensDeTri.ASC).orderBy("LANGUE", SensDeTri.ASC).orderBy("LIBELLE", SensDeTri.ASC);
		requete.where(where);
		requete.orderBy(orderBy);
		return requete;
	}

	private List<ResultatDatagrid> mapperResultatRechercheDepuisLibelle(final Libelle libelle) throws Exception {
		final List<ResultatDatagrid> resultats = new ArrayList<>();
		final Map<String, ResultatLibelle> libellesParTypeCode = new HashMap<>();
		while (libelle.nextItem()) {
			final ResultatLibelle lib = new ResultatLibelle();
			lib.setCode(libelle.getCode());
			lib.setType(cache.getListeTypesLibelles().get(libelle.getType()));
			lib.setUrlModification(DatagridUtils.getUrlActionLibelle(String.valueOf(libelle.getIdLibelle()), "MODIFIERPARID"));
			if (libelle.isLibelleSupprimable()) {
				lib.setUrlSuppression(DatagridUtils.getUrlActionLibelle(String.valueOf(libelle.getIdLibelle()), "SUPPRIMERPARID"));
			}
			if (!libellesParTypeCode.containsKey(libelle.getType() + libelle.getCode())) {
				lib.addLibelleParLangue(libelle.getLangue(), libelle.getLibelle());
				libellesParTypeCode.put(libelle.getType() + libelle.getCode(), lib);
			} else {
				final ResultatLibelle libelleCourant = libellesParTypeCode.get(libelle.getType() + libelle.getCode());
				libelleCourant.addLibelleParLangue(libelle.getLangue(), libelle.getLibelle());
			}
		}
		resultats.addAll(libellesParTypeCode.values());
		return resultats;
	}

	private boolean isPermissionNonValide(final AutorisationBean autorisations) {
		return autorisations == null || !autorisations.isWebMaster() && !autorisations.possedePermission(ComposantLibelle.getPermissionGestion());
	}

	@Override
	protected Predicate retrouvePredicate(final String critereRecherche) {
		return new Predicate() {

			@Override
			public boolean evaluate(final Object object) {
				boolean isARetourner = Boolean.FALSE;
				if (object instanceof ResultatLibelle) {
					final ResultatLibelle libelle = (ResultatLibelle) object;
					isARetourner = containsIgnoreCase(libelle.getType(), critereRecherche) || containsIgnoreCase(libelle.getCode(), critereRecherche);
					if (!isARetourner) {
						for (final LibelleParLangue libParLangue : libelle.getLibelleParLangue()) {
							isARetourner = containsIgnoreCase(libParLangue.getLibelle(), critereRecherche) || containsIgnoreCase(libParLangue.getLangue(), critereRecherche);
							if (isARetourner) {
								break;
							}
						}
					}
				}
				return isARetourner;
			}
		};
	}
}
