/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.datagrid.processus;

import java.sql.Date;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Predicate;
import org.apache.commons.collections.Transformer;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.Formateur;
import com.jsbsoft.jtf.core.LangueUtil;
import com.jsbsoft.jtf.session.SessionUtilisateur;
import com.kportal.cms.objetspartages.Objetpartage;
import com.univ.datagrid.bean.ResultatDatagrid;
import com.univ.datagrid.bean.ResultatFicheDatagrid;
import com.univ.datagrid.filtre.FiltreFicheUniv;
import com.univ.datagrid.utils.DatagridUtils;
import com.univ.multisites.InfosSite;
import com.univ.multisites.Site;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.DiffusionSelective;
import com.univ.objetspartages.om.EtatFiche;
import com.univ.objetspartages.om.FicheRattachementsSecondaires;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.Metatag;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.objetspartages.om.Rubrique;
import com.univ.url.UrlManager;
import com.univ.utils.Chaine;
import com.univ.utils.ContexteUniv;
import com.univ.utils.ContexteUtil;
import com.univ.utils.DateUtil;
import com.univ.utils.FicheUnivHelper;
import com.univ.utils.SessionUtil;
import com.univ.utils.URLResolver;
import com.univ.utils.recherche.RechercheMultificheHelper;
import com.univ.utils.recherche.ResultatRechercheMultifiche;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;

public class MultiFicheDatagrid extends AbstractServiceDatagrid {

	/** l'id Spring du bean. */
	public static final String ID_BEAN = "multiFicheDatagrid";

	private static Logger LOG = LoggerFactory.getLogger(MultiFicheDatagrid.class);

	@Override
	public List<ResultatDatagrid> traiterRechercheDepuisRequete(final HttpServletRequest req) {
		final AutorisationBean autorisations = (AutorisationBean) SessionUtil.getInfosSession(req).get(SessionUtilisateur.AUTORISATIONS);
		if (autorisations == null) {
			return Collections.emptyList();
		}
		final ContexteUniv ctx = ContexteUtil.getContexteUniv();
		final String libelle = req.getParameter("TITRE");
		final String codeObjet = req.getParameter("CODE_OBJET");
		final String codeFiche = req.getParameter("CODE_FICHE");
		final String codeRubrique = req.getParameter("CODE_RUBRIQUE_RECHERCHE");
		final String codeRattachement = req.getParameter("CODE_RATTACHEMENT");
		final String codeRedacteur = req.getParameter("CODE_REDACTEUR");
		final String idMeta = req.getParameter("ID_META");
		final String urlFiche = req.getParameter("URL_FICHE");
		final String langue = req.getParameter("LANGUE");
		final String etatObjet = req.getParameter("ETAT_OBJET");
		// Récupération des critère de recherche de date
		final Date dateDebutCreation = recupererAttributDate(req, "DATE_CREATION_DEBUT");
		final Date dateFinCreation = recupererAttributDate(req, "DATE_CREATION_FIN");
		final Date dateDebutModification = recupererAttributDate(req, "DATE_MODIFICATION_DEBUT");
		final Date dateFinModification = recupererAttributDate(req, "DATE_MODIFICATION_FIN");
		final Date dateDebutMiseEnLigne = recupererAttributDate(req, "DATE_MISE_EN_LIGNE_DEBUT");
		final Date dateFinMiseEnLigne = recupererAttributDate(req, "DATE_MISE_EN_LIGNE_FIN");
		ResultatRechercheMultifiche resultatRechercheMultifiche = null;
		List<ResultatDatagrid> resultats = null;
		try {
			resultatRechercheMultifiche = RechercheMultificheHelper.rerchercherParmisToutesLesFiches(ctx, autorisations, libelle, codeObjet, codeFiche, codeRubrique, codeRattachement,
				codeRedacteur, idMeta, urlFiche, dateDebutCreation, dateFinCreation, dateDebutModification, dateFinModification, dateDebutMiseEnLigne, dateFinMiseEnLigne, langue,
				etatObjet, String.valueOf(DatagridUtils.getNombreMaxDatagrid()));
			resultats = mapperResultatRechercheDepuisMeta(resultatRechercheMultifiche.getResultats());
		} catch (final Exception e) {
			LOG.error("impossible d'executer la recherche", e);
			return Collections.<ResultatDatagrid> emptyList();
		}
		return resultats;
	}

	private Date recupererAttributDate(final HttpServletRequest req, final String nomAttribut) {
		Date dateDepuisRequete = null;
		try {
			dateDepuisRequete = DateUtil.parseDate(req.getParameter(nomAttribut));
			if (!Formateur.estSaisie(dateDepuisRequete)) {
				dateDepuisRequete = null;
			}
		} catch (final Exception e) {}
		return dateDepuisRequete;
	}

	public static List<ResultatDatagrid> mapperResultatRechercheDepuisMeta(final Collection<Metatag> metatags) {
		final List<ResultatDatagrid> resultats = new ArrayList<>();
		final Map<String, String> libellesParEtat = ReferentielObjets.getEtatsObjet();
		final Map<String, String> classParEtat = FicheUnivHelper.getClassEtatsObjet();
		for (final Metatag leMeta : metatags) {
			final ResultatFicheDatagrid resultat = new ResultatFicheDatagrid();
			resultat.setFilAriane(Rubrique.getFilAriane(leMeta.getMetaCodeRubrique(), ">"));
			resultat.setId(leMeta.getIdMetatag());
			resultat.setRubrique((Rubrique.getIntitule(leMeta.getMetaCodeRubrique())));
			resultat.setLibelle(leMeta.getMetaLibelleFiche());
			resultat.setEtatFiche(libellesParEtat.get(leMeta.getMetaEtatObjet()));
			resultat.setClassEtatFiche(classParEtat.get(leMeta.getMetaEtatObjet()));
			resultat.setLangue(LangueUtil.getLocale(leMeta.getMetaLangue()).getLanguage());
			resultat.setUrlDrapeauLangue(LangueUtil.getPathImageDrapeau(leMeta.getMetaLangue()));
			resultat.setTypeObjet(ReferentielObjets.getLibelleObjet(leMeta.getMetaCodeObjet()));
			resultat.setDateModification(new java.util.Date(leMeta.getMetaDateModification().getTime()));
			resultats.add(resultat);
		}
		return resultats;
	}

	@Override
	public void traiterAction(final HttpServletRequest req) {
		final ContexteUniv ctx = ContexteUtil.getContexteUniv();
		final AutorisationBean autorisations = ctx.getAutorisation();
		final String[] idsMetas = req.getParameterValues("idsMetas");
		final String action = req.getParameter("ACTION");
		if (idsMetas != null && idsMetas.length > 0 && StringUtils.isNotBlank(action)) {
			@SuppressWarnings("unchecked")
			final Collection<Long> idsMetasATraiter = CollectionUtils.collect(Arrays.asList(idsMetas), new Transformer() {

				@Override
				public Object transform(final Object input) {
					return Long.valueOf((String) input);
				}
			});
			try {
				if ("SUPPRIMER".equals(action)) {
					FicheUnivHelper.supprimerListeFichesParIdsMeta(idsMetasATraiter, autorisations);
				} else if ("ARCHIVER".equals(action)) {
					FicheUnivHelper.archiverListeFichesParIdsMeta(idsMetasATraiter, autorisations);
				} else if ("EN_LIGNE".equals(action)) {
					FicheUnivHelper.mettreEnLigneListeFichesParIdsMeta(idsMetasATraiter, autorisations);
				}
			} catch (final Exception e) {
				LOG.error("impossible de traiter l'action : " + action + " sur les Ids metas : " + StringUtils.join(idsMetas, " ,") + " pour l'utilisateur : " + ctx.getCode(), e);
			}
		}
	}

	@Override
	protected Predicate retrouvePredicate(final String critereRecherche) {
		return new FiltreFicheUniv(critereRecherche);
	}

	@Override
	public List<ResultatDatagrid> postTraitementResultat(final List<ResultatDatagrid> resultats) {
		if (CollectionUtils.isNotEmpty(resultats)) {
			final ContexteUniv ctx = ContexteUtil.getContexteUniv();
			final Map<String, ResultatFicheDatagrid> resultatsParId = getResultatParIDMetatag(resultats);
			final AutorisationBean autorisations = ctx.getAutorisation();
			final Map<String, Objetpartage> objetPartageParCode = ReferentielObjets.getReferentiel().getObjetsByCode();
			final Metatag leMeta = new Metatag();
			leMeta.setCtx(ctx);
			final ClauseWhere whereIdMeta = new ClauseWhere(ConditionHelper.in("ID_METATAG", resultatsParId.keySet()));
			try {
				leMeta.select(whereIdMeta.formaterSQL());
				while (leMeta.nextItem()) {
					final ResultatFicheDatagrid resultat = resultatsParId.get(String.valueOf(leMeta.getIdMetatag()));
					modifUrlEnLigneResultat(leMeta, resultat);
					modifUrlsContributionsResultat(autorisations, objetPartageParCode, leMeta, resultat);
				}
			} catch (final Exception e1) {
				LOG.error("erreur dans la requete ou le parcours des resultats", e1);
			}
		}
		return resultats;
	}

	private void modifUrlsContributionsResultat(final AutorisationBean autorisations, final Map<String, Objetpartage> objetPartageParCode, final Metatag leMeta,
		final ResultatFicheDatagrid resultat) {
		final FicheUniv fiche = instacierFicheUniv(leMeta, objetPartageParCode);
		if (autorisations != null && autorisations.estAutoriseAModifierLaFiche(fiche)) {
			resultat.setUrlModification(DatagridUtils.getUrlActionFiche(leMeta.getMetaCodeObjet(), leMeta.getMetaIdFiche(), "MODIFIER"));
		}
		if (autorisations != null && autorisations.estAutoriseASupprimerLaFiche(fiche) && !EtatFiche.A_SUPPRIMER.getEtat().equals(leMeta.getMetaEtatObjet())) {
			resultat.setUrlSuppression(DatagridUtils.getUrlActionFiche(leMeta.getMetaCodeObjet(), leMeta.getMetaIdFiche(), "SUPPRIMER"));
		}
	}

	private static FicheUniv instacierFicheUniv(final Metatag meta, final Map<String, Objetpartage> objetPartageParCode) {
		final Objetpartage objetCourant = objetPartageParCode.get(meta.getMetaCodeObjet());
		FicheUniv ficheUniv = null;
		if (objetCourant != null) {
			try {
				ficheUniv = (FicheUniv) Class.forName(objetCourant.getNomClasse()).newInstance();
			} catch (IllegalAccessException | ClassNotFoundException | InstantiationException e) {
				return null;
			}
		} else {
			return null;
		}
		ficheUniv.setCodeRattachement(meta.getMetaCodeRattachement());
		ficheUniv.setCodeRubrique(meta.getMetaCodeRubrique());
		if (ficheUniv instanceof FicheRattachementsSecondaires) {
			((FicheRattachementsSecondaires) ficheUniv).setCodeRattachementAutres(Chaine.convertirPointsVirgulesEnAccolades(meta.getMetaCodeRattachementAutres()));
		}
		ficheUniv.setCodeRedacteur(meta.getMetaCodeRedacteur());
		ficheUniv.setCodeValidation(meta.getMetaCodeValidation());
		ficheUniv.setCode(meta.getMetaCode());
		ficheUniv.setLangue(meta.getMetaLangue());
		ficheUniv.setEtatObjet(meta.getMetaEtatObjet());
		if (ficheUniv instanceof DiffusionSelective) {
			((DiffusionSelective) ficheUniv).setDiffusionPublicVise(meta.getMetaDiffusionPublicVise());
			((DiffusionSelective) ficheUniv).setDiffusionModeRestriction(meta.getMetaDiffusionModeRestriction());
			((DiffusionSelective) ficheUniv).setDiffusionPublicViseRestriction(meta.getMetaDiffusionPublicViseRestriction());
		}
		return ficheUniv;
	}

	private void modifUrlEnLigneResultat(final Metatag leMeta, final ResultatFicheDatagrid resultat) {
		if (EtatFiche.EN_LIGNE.getEtat().equals(leMeta.getMetaEtatObjet())) {
			final InfosSite siteFiche = Site.determinerSiteRubrique(leMeta.getMetaCodeRubrique());
			resultat.setUrl(URLResolver.getAbsoluteUrl(UrlManager.calculerUrlFiche(leMeta), siteFiche, URLResolver.STANDARD_ACTION));
		}
	}

	private Map<String, ResultatFicheDatagrid> getResultatParIDMetatag(final List<ResultatDatagrid> resultats) {
		final Map<String, ResultatFicheDatagrid> resultatsParId = new HashMap<>();
		for (final ResultatDatagrid resultatDatagrid : resultats) {
			resultatsParId.put(String.valueOf(resultatDatagrid.getId()), (ResultatFicheDatagrid) resultatDatagrid);
		}
		return resultatsParId;
	}
}
