/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.datagrid.processus;

import static org.apache.commons.lang3.StringUtils.containsIgnoreCase;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.collections.Predicate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.LangueUtil;
import com.univ.datagrid.bean.ResultatDatagrid;
import com.univ.datagrid.bean.ResultatRubrique;
import com.univ.datagrid.utils.DatagridUtils;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.Perimetre;
import com.univ.objetspartages.om.PermissionBean;
import com.univ.objetspartages.om.Rubrique;
import com.univ.utils.ContexteUniv;
import com.univ.utils.ContexteUtil;

public class RubriqueDatagrid extends AbstractServiceDatagrid {

	private static final Logger LOG = LoggerFactory.getLogger(RubriqueDatagrid.class);

	public static final String ID_BEAN = "rubriqueDatagrid";

	@Override
	public List<ResultatDatagrid> traiterRechercheDepuisRequete(final HttpServletRequest req) {
		List<ResultatDatagrid> resultats = null;
		final Rubrique rubriquePourRequete = new Rubrique();
		final ContexteUniv ctx = ContexteUtil.getContexteUniv();
		rubriquePourRequete.setCtx(ctx);
		try {
			rubriquePourRequete.select(req.getParameter("CODE"), req.getParameter("LANGUE"), req.getParameter("INTITULE"), req.getParameter("CODE_SAISI"));
			resultats = mapperResultatRechercheDepuisRubrique(rubriquePourRequete);
		} catch (final Exception e) {
			LOG.error("impossible de récuperer les resultats de la recherche", e);
		}
		if (resultats == null) {
			resultats = Collections.<ResultatDatagrid> emptyList();
		}
		return resultats;
	}

	/**
	 * A partir de l'objet Rubrique, on retrouve l'ensemble des rubriques et on les map dans une liste de résultat exploitable
	 *
	 * @param rubriquePourRequete
	 * @return
	 * @throws Exception
	 */
	public static List<ResultatDatagrid> mapperResultatRechercheDepuisRubrique(final Rubrique rubriquePourRequete) throws Exception {
		final List<ResultatDatagrid> resultats = new ArrayList<>();
		final AutorisationBean autorisations = ContexteUtil.getContexteUniv().getAutorisation();
		final Map<String, String> languageParCodeLangue = DatagridUtils.getLanguageParCodeLangue();
		final PermissionBean permissionSuppressionRubrique = new PermissionBean("TECH", "rub", "S");
		final PermissionBean permissionModificationRubrique = new PermissionBean("TECH", "rub", "M");
		while (rubriquePourRequete.nextItem()) {
			final Perimetre perimetreRubriqueCourante = new Perimetre("*", rubriquePourRequete.getCode(), "*", "*", "");
			if (autorisations != null && (autorisations.possedePermission(permissionModificationRubrique, perimetreRubriqueCourante) || autorisations.possedePermission(
				permissionSuppressionRubrique, perimetreRubriqueCourante))) {
				final ResultatRubrique res = new ResultatRubrique();
				res.setLibelle(rubriquePourRequete.getIntitule());
				res.setRubriqueMere(Rubrique.getIntituleComplet(rubriquePourRequete.getCode(), Boolean.FALSE));
				res.setCode(rubriquePourRequete.getCode());
				res.setLangue(languageParCodeLangue.get(rubriquePourRequete.getLangue()));
				res.setUrlDrapeauLangue(LangueUtil.getPathImageDrapeau(rubriquePourRequete.getLangue()));
				if (autorisations.possedePermission(permissionModificationRubrique, perimetreRubriqueCourante)) {
					res.setUrlModification(DatagridUtils.getUrlActionRubrique(rubriquePourRequete.getCode(), "MODIFIERRUBRIQUEPARCODE"));
				}
				if (autorisations.possedePermission(permissionSuppressionRubrique, perimetreRubriqueCourante)) {
					res.setUrlSuppression(DatagridUtils.getUrlActionRubrique(rubriquePourRequete.getCode(), "SUPPRIMERRUBRIQUEPARCODE"));
				}
				resultats.add(res);
			}
		}
		return resultats;
	}

	@Override
	protected Predicate retrouvePredicate(final String critereRecherche) {
		return new Predicate() {

			@Override
			public boolean evaluate(final Object object) {
				boolean isARetourner = Boolean.FALSE;
				if (object instanceof ResultatRubrique) {
					final ResultatRubrique rubrique = (ResultatRubrique) object;
					isARetourner = containsIgnoreCase(rubrique.getLibelle(), critereRecherche) || containsIgnoreCase(rubrique.getRubriqueMere(), critereRecherche) || containsIgnoreCase(
						rubrique.getLangue(), critereRecherche) || containsIgnoreCase(rubrique.getCode(), critereRecherche);
				}
				return isARetourner;
			}
		};
	}
}
