/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.datagrid.processus;

import static org.apache.commons.lang3.StringUtils.containsIgnoreCase;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Predicate;
import org.apache.commons.collections.Transformer;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.session.SessionUtilisateur;
import com.kportal.core.webapp.WebAppUtil;
import com.kportal.extension.module.composant.ComposantUtilisateur;
import com.univ.collaboratif.om.Espacecollaboratif;
import com.univ.datagrid.bean.ResultatDatagrid;
import com.univ.datagrid.bean.ResultatUtilisateur;
import com.univ.datagrid.utils.DatagridUtils;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.Groupeutilisateur;
import com.univ.objetspartages.om.InfosStructure;
import com.univ.objetspartages.om.Structure;
import com.univ.objetspartages.om.Utilisateur;
import com.univ.utils.ContexteUniv;
import com.univ.utils.ContexteUtil;
import com.univ.utils.SessionUtil;

public class UtilisateurDatagrid extends AbstractServiceDatagrid {

	private static final Logger LOG = LoggerFactory.getLogger(UtilisateurDatagrid.class);

	public static final String ID_BEAN = "utilisateurDatagrid";

	@Override
	public List<ResultatDatagrid> traiterRechercheDepuisRequete(final HttpServletRequest req) {
		final AutorisationBean autorisations = (AutorisationBean) SessionUtil.getInfosSession(req).get(SessionUtilisateur.AUTORISATIONS);
		if (isPermissionNonValide(autorisations)) {
			return Collections.emptyList();
		}
		final ContexteUniv ctx = ContexteUtil.getContexteUniv();
		final String code = req.getParameter("CODE");
		final String nom = req.getParameter("NOM");
		final String prenom = req.getParameter("PRENOM");
		final String profil = req.getParameter("PROFIL_DSI");
		final String groupe = req.getParameter("GROUPE_DSI");
		final String codeRattachement = req.getParameter("CODE_RATTACHEMENT");
		final String adresseMail = req.getParameter("ADRESSE_MAIL");
		final Utilisateur utilisateurPourRecherche = new Utilisateur();
		utilisateurPourRecherche.init();
		utilisateurPourRecherche.setCtx(ctx);
		List<ResultatDatagrid> resultats = null;
		try {
			utilisateurPourRecherche.select(code, nom, prenom, StringUtils.EMPTY, profil, groupe, codeRattachement, adresseMail);
			resultats = mapperResultatRecherche(utilisateurPourRecherche, groupe, ctx.getLangue(), autorisations);
		} catch (final Exception e) {
			LOG.error("impossible d'executer la recherche", e);
			resultats = Collections.emptyList();
		}
		return resultats;
	}

	private boolean isPermissionNonValide(final AutorisationBean autorisations) {
		return autorisations == null || !autorisations.isWebMaster() && !autorisations.possedePermission(ComposantUtilisateur.getPermissionConsultation());
	}

	private List<ResultatDatagrid> mapperResultatRecherche(final Utilisateur utilisateurPourRecherche, final String groupe, final String langue,
		final AutorisationBean autorisations) throws Exception {
		final List<ResultatDatagrid> resultats = new ArrayList<>();
		int nbUtilisateur = 0;
		final boolean isUtilisateurGestion = autorisations.possedePermission(ComposantUtilisateur.getPermissionGestion()) || autorisations.isWebMaster();
		while (utilisateurPourRecherche.nextItem() && nbUtilisateur < DatagridUtils.getNombreMaxDatagrid()) {
			final ResultatUtilisateur res = new ResultatUtilisateur();
			res.setId(utilisateurPourRecherche.getIdUtilisateur());
			res.setCode(utilisateurPourRecherche.getCode());
			res.setCivilite(utilisateurPourRecherche.getCivilite());
			res.setNom(utilisateurPourRecherche.getNom());
			res.setMail(utilisateurPourRecherche.getAdresseMail());
			res.setLibelle(utilisateurPourRecherche.getNom() + " " + utilisateurPourRecherche.getPrenom());
			res.setPrenom(utilisateurPourRecherche.getPrenom());
			res.setArboStructure(Structure.getFilAriane(utilisateurPourRecherche.getCodeRattachement(), langue));
			final InfosStructure info = Structure.renvoyerItemStructure(utilisateurPourRecherche.getCodeRattachement(), langue);
			res.setStructure(info.getLibelleAffichable());
			if (isUtilisateurGestion) {
				final StringBuilder baseUrlAction = new StringBuilder(WebAppUtil.SG_PATH + "?EXT=core&PROC=SAISIE_UTILISATEUR&ID=");
				baseUrlAction.append(String.valueOf(utilisateurPourRecherche.getIdUtilisateur()));
				baseUrlAction.append("&ACTION=");
				res.setUrlModification(baseUrlAction.toString() + "MODIFIERPARID");
				res.setUrlSuppression(baseUrlAction.toString() + "SUPPRIMERPARID");
				res.setUrlSuppressionGroupe(WebAppUtil.SG_PATH + "?EXT=core&PROC=SAISIE_GROUPEDSI&ACTION=SUPPRIMERDUGROUPE&GROUPE_DSI=" + groupe + "&UTILISATEUR=" + utilisateurPourRecherche.getIdUtilisateur());
			}
			resultats.add(res);
			nbUtilisateur++;
		}
		return resultats;
	}

	@Override
	public void traiterAction(final HttpServletRequest req) {
		final ContexteUniv ctx = ContexteUtil.getContexteUniv();
		final String[] idsUtilisateur = req.getParameterValues("idsUtilisateur");
		final String action = req.getParameter("ACTION");
		final AutorisationBean autorisation = ctx.getAutorisation();
		if (isActionValide(action, autorisation, idsUtilisateur)) {
			@SuppressWarnings("unchecked")
			final Collection<Long> idsMetasATraiter = CollectionUtils.collect(Arrays.asList(idsUtilisateur), new Transformer() {

				@Override
				public Object transform(final Object input) {
					return Long.valueOf((String) input);
				}
			});
			try {
				if ("SUPPRIMER".equals(action)) {
					for (final Long idUser : idsMetasATraiter) {
						final Utilisateur utilisateur = new Utilisateur();
						utilisateur.init();
						utilisateur.setCtx(ctx);
						utilisateur.setIdUtilisateur(idUser);
						utilisateur.retrieve();
						Espacecollaboratif.supprimerUtilisateur(utilisateur.getCode());
						Groupeutilisateur.deleteParUtilisateur(ctx, utilisateur.getCode());
						utilisateur.delete();
					}
				}
			} catch (final Exception e) {
				LOG.error("impossible de traiter : " + action + " sur les Ids metas : " + StringUtils.join(idsUtilisateur, " ,") + " pour l'utilisateur : " + ctx.getCode(), e);
			}
		}
	}

	private boolean isActionValide(final String action, final AutorisationBean autorisation, final String[] idsUtilisateur) {
		final boolean isAutorise = autorisation != null && (autorisation.possedePermission(ComposantUtilisateur.getPermissionGestion()) || autorisation.isWebMaster());
		final boolean isUtilisateursValide = idsUtilisateur != null && idsUtilisateur.length > 0;
		return isAutorise && isUtilisateursValide && StringUtils.isNotBlank(action);
	}

	@Override
	protected Predicate retrouvePredicate(final String critereRecherche) {
		return new Predicate() {

			@Override
			public boolean evaluate(final Object object) {
				boolean isARetourner = Boolean.FALSE;
				if (object instanceof ResultatUtilisateur) {
					final ResultatUtilisateur groupes = (ResultatUtilisateur) object;
					isARetourner = containsIgnoreCase(groupes.getLibelle(), critereRecherche) || containsIgnoreCase(groupes.getMail(), critereRecherche) || containsIgnoreCase(
						groupes.getStructure(), critereRecherche) || containsIgnoreCase(groupes.getCode(), critereRecherche);
				}
				return isARetourner;
			}
		};
	}
}
