/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.identity.dao.impl;

import com.jsbsoft.jtf.datasource.dao.impl.mysql.AbstractCommonDAO;
import com.jsbsoft.jtf.datasource.exceptions.AddToDataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.DataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.UpdateToDataSourceException;
import com.univ.identity.bean.IdentityBean;
import com.univ.identity.om.Identity;
import com.univ.utils.ContexteDao;
import com.univ.utils.sql.RequeteSQL;
import com.univ.utils.sql.clause.ClauseLimit;
import com.univ.utils.sql.clause.ClauseOrderBy;
import com.univ.utils.sql.clause.ClauseOrderBy.SensDeTri;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.utils.sql.criterespecifique.LimitHelper;
import com.univ.utils.sql.operande.TypeOperande;
import org.apache.commons.lang3.StringUtils;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.map.SerializationConfig;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.List;

public class IdentityDAO<I extends IdentityBean> extends AbstractCommonDAO<I> {

    protected final ObjectMapper mapper = new ObjectMapper();

    public IdentityDAO() {
        this.tableName = "IDENTITY";
        mapper.configure(SerializationConfig.Feature.FAIL_ON_EMPTY_BEANS, false);
    }

    @Override
    public I add(final I identityBean) throws AddToDataSourceException {
        ResultSet rs = null;
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement("insert into IDENTITY (CIVILITE, NOM, PRENOM, ADRESSE, CODE_POSTAL, VILLE, PAYS, TELEPHONE, EMAIL, JAVA_TYPE, DATA) values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)", Statement.RETURN_GENERATED_KEYS)) {
            final String datas = mapper.writeValueAsString(identityBean);
            stmt.setObject(1, identityBean.getCivilite(), Types.VARCHAR);
            stmt.setObject(2, identityBean.getNom(), Types.VARCHAR);
            stmt.setObject(3, identityBean.getPrenom(), Types.VARCHAR);
            stmt.setObject(4, identityBean.getAdresse(), Types.VARCHAR);
            stmt.setObject(5, identityBean.getCodePostal(), Types.VARCHAR);
            stmt.setObject(6, identityBean.getVille(), Types.VARCHAR);
            stmt.setObject(7, identityBean.getPays(), Types.VARCHAR);
            stmt.setObject(8, identityBean.getTelephone(), Types.VARCHAR);
            stmt.setObject(9, identityBean.getEmail(), Types.VARCHAR);
            stmt.setObject(10, identityBean.getClass().getName(), Types.BLOB);
            stmt.setObject(11, datas, Types.VARCHAR);
            final int rowsAffected = stmt.executeUpdate();
            if (rowsAffected != 1) {
                throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"%s\" : %d row(s) affected", identityBean.toString(), tableName, rowsAffected));
            }
            rs = stmt.getGeneratedKeys();
            rs.next();
            identityBean.setIdIdentity(rs.getLong(1));
        } catch (final Exception e) {
            throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"%s\"", identityBean.toString(), tableName), e);
        } finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (final SQLException e) {
                    LOG.error(String.format("An error occured closing Resultset during add() on table %s", tableName), e);
                }
            }
        }
        return identityBean;
    }

    @Override
    public I update(final I identityBean) throws UpdateToDataSourceException {
        try (Connection connection = dataSource.getConnection();
             PreparedStatement stmt = connection.prepareStatement("update IDENTITY set CIVILITE = ?, NOM = ?, PRENOM = ?, ADRESSE = ?, CODE_POSTAL = ?, VILLE = ?, PAYS = ?, TELEPHONE = ?, EMAIL = ?, JAVA_TYPE = ?, DATA = ? where ID_IDENTITY = ?")) {
            final String datas = mapper.writeValueAsString(identityBean);
            // put parameters into statement
            stmt.setObject(1, identityBean.getCivilite(), Types.VARCHAR);
            stmt.setObject(2, identityBean.getNom(), Types.VARCHAR);
            stmt.setObject(3, identityBean.getPrenom(), Types.VARCHAR);
            stmt.setObject(4, identityBean.getAdresse(), Types.VARCHAR);
            stmt.setObject(5, identityBean.getCodePostal(), Types.VARCHAR);
            stmt.setObject(6, identityBean.getVille(), Types.VARCHAR);
            stmt.setObject(7, identityBean.getPays(), Types.VARCHAR);
            stmt.setObject(8, identityBean.getTelephone(), Types.VARCHAR);
            stmt.setObject(9, identityBean.getEmail(), Types.VARCHAR);
            stmt.setObject(10, identityBean.getClass().getName(), Types.BLOB);
            stmt.setObject(11, datas, Types.VARCHAR);
            stmt.setObject(12, identityBean.getIdIdentity(), Types.BIGINT);
            stmt.executeUpdate();
            stmt.close();
        } catch (final Exception e) {
            throw new UpdateToDataSourceException(String.format("Unable to update [%s] from table \"%s\"", identityBean.toString(), tableName), e);
        }
        return identityBean;
    }

    @SuppressWarnings("unchecked")
    @Override
    protected I fill(final ResultSet rs) throws DataSourceException {
        I identityBean;
        try {
            final Class<?> className = Class.forName(rs.getString("JAVA_TYPE"));
            identityBean = (I) mapper.readValue(rs.getString("DATA"), className);
            identityBean.setIdIdentity(rs.getLong("ID_IDENTITY"));
            identityBean.setCivilite(rs.getString("CIVILITE"));
            identityBean.setNom(rs.getString("NOM"));
            identityBean.setPrenom(rs.getString("PRENOM"));
            identityBean.setAdresse(rs.getString("ADRESSE"));
            identityBean.setCodePostal(rs.getString("CODE_POSTAL"));
            identityBean.setVille(rs.getString("VILLE"));
            identityBean.setPays(rs.getString("PAYS"));
            identityBean.setTelephone(rs.getString("TELEPHONE"));
            identityBean.setEmail(rs.getString("EMAIL"));
            identityBean.setData(rs.getString("DATA"));
        } catch (final Exception e) {
            throw new DataSourceException("An error occured trying to map resultset to object", e);
        }
        return identityBean;
    }

    public List<I> select(final Identity<I> identity, final String civilite, final String nom, final String prenom, final String adresse, final String codePostal,
                          final String ville, final String pays, final String telephone, final String email, final String nombre) throws DataSourceException {
        try (final ContexteDao ctx = new ContexteDao()) {
            final RequeteSQL requeteSelect = new RequeteSQL();
            final ClauseWhere where = new ClauseWhere();
            if (identity != null && identity.getIdIdentity() != null) {
                where.and(ConditionHelper.egal("ID_IDENTITY", identity.getIdIdentity(), TypeOperande.LONG));
            }
            if (StringUtils.isNotEmpty(civilite)) {
                where.and(ConditionHelper.egalVarchar("CIVILITE", civilite));
            }
            if (StringUtils.isNotEmpty(nom)) {
                where.and(ConditionHelper.egalVarchar("NOM", nom));
            }
            if (StringUtils.isNotEmpty(prenom)) {
                where.and(ConditionHelper.egalVarchar("PRENOM", prenom));
            }
            if (StringUtils.isNotEmpty(adresse)) {
                where.and(ConditionHelper.egalVarchar("ADRESSE", adresse));
            }
            if (StringUtils.isNotEmpty(codePostal)) {
                where.and(ConditionHelper.egalVarchar("CODE_POSTAL", codePostal));
            }
            if (StringUtils.isNotEmpty(ville)) {
                where.and(ConditionHelper.egalVarchar("VILLE", ville));
            }
            if (StringUtils.isNotEmpty(pays)) {
                where.and(ConditionHelper.egalVarchar("PAYS", pays));
            }
            if (StringUtils.isNotEmpty(telephone)) {
                where.and(ConditionHelper.egalVarchar("TELEPHONE", telephone));
            }
            if (StringUtils.isNotEmpty(email)) {
                where.and(ConditionHelper.egalVarchar("EMAIL", email));
            }
            requeteSelect.where(where);
            final ClauseOrderBy orderBy = new ClauseOrderBy();
            orderBy.orderBy("T1.ID_IDENTITY", SensDeTri.ASC);
            requeteSelect.orderBy(orderBy);
            final ClauseLimit limite = LimitHelper.ajouterCriteresLimitesEtOptimisation(ctx, nombre);
            requeteSelect.limit(limite);
            return select(requeteSelect.formaterRequete());
        }
    }
}
