/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.mediatheque.container;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import com.univ.objetspartages.om.Media;

/**
 * Il s'agit d'un objet qui permet de contenir un ensemble de {@link Media} ainsi que des informations sur l'affichage de ces derniers : intitule, tri et style. Cela veut dire que
 * le même style sera appliqué à tous les média de cette objet.
 * 
 * @author Pierre Cosson
 * @see ClassifieurMedia
 */
public class ClassificationMedia implements Comparable<ClassificationMedia> {

	/**
	 * L'identifiant auto-généré de la classification (regénrer à chaque initialisation).
	 */
	private String code;

	/**
	 * Le code de classification définissant les médias à insérer dans cette classification.
	 */
	private String codeClassification;

	/**
	 * L'intitulé de la classification.
	 */
	private String intitule;

	/**
	 * La liste de médias qui correspondent à la classifiaction. Cette liste est mise à jour par un {@link ClassifieurMedia} et vidée à chaque initialisation.
	 */
	private List<Media> medias;

	/**
	 * Constructeur.
	 * 
	 * @param codeClassification
	 *            Le code de classification (en lien direct avec les {@link ClassificationMedia} puisque ce code est dépendant du classificateur).
	 * @param intitule
	 *            L'intitulé de la classification.
	 */
	public ClassificationMedia() {
		this.init();
	}

	/**
	 * Initialiser le code et les medias de l'insatnce.
	 */
	public void init() {
		String randomValue = String.valueOf(new Random().nextInt(10000));
		this.code = String.valueOf(System.currentTimeMillis() + randomValue);
		this.medias = new ArrayList<Media>();
	}

	/**
	 * Gets the code classification.
	 * 
	 * @return the codeClassification
	 */
	public String getCodeClassification() {
		return codeClassification;
	}

	/**
	 * Sets the medias.
	 * 
	 * @param medias
	 *            the medias to set
	 */
	public void setMedias(List<Media> medias) {
		this.medias = medias;
	}

	/**
	 * Gets the code.
	 * 
	 * @return the code
	 */
	public String getCode() {
		return code;
	}

	/**
	 * Gets the intitule.
	 * 
	 * @return the intitule
	 */
	public String getIntitule() {
		return intitule;
	}

	/**
	 * Gets the medias.
	 * 
	 * @return the medias
	 */
	public List<Media> getMedias() {
		return medias;
	}

	/**
	 * Ajouter un média à la classification.
	 * 
	 * @param media
	 *            Le {@link Media} à ajouter.
	 */
	public void addMedia(Media media) {
		this.medias.add(media);
	}

	/**
	 * Savoir si cette instance contient des {@link Media}.
	 * 
	 * @return TRUE s'il n'y a pas de {@link Media} sinon FALSE.
	 */
	public boolean isEmpty() {
		return this.medias.isEmpty();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	@Override
	public int compareTo(ClassificationMedia o) {
		return this.getIntitule().compareToIgnoreCase(o.getIntitule());
	}

	public void setCodeClassification(String codeClassification) {
		this.codeClassification = codeClassification;
	}

	public void setIntitule(String intitule) {
		this.intitule = intitule;
	}
}
