/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.migration;

import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.sql.SQLException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Properties;

import javax.imageio.ImageIO;
import javax.imageio.ImageWriter;
import javax.imageio.stream.ImageOutputStream;
import javax.servlet.jsp.JspWriter;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;

import ch.qos.logback.classic.encoder.PatternLayoutEncoder;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.FileAppender;

import com.jsbsoft.jtf.core.LangueUtil;
import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.webutils.ContextePage;
import com.kportal.core.config.PropertyHelper;
import com.kportal.core.webapp.WebAppUtil;
import com.univ.collaboratif.om.Espacecollaboratif;
import com.univ.collaboratif.om.Fichiergw;
import com.univ.mediatheque.utils.MediathequeHelper;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.Media;
import com.univ.objetspartages.om.Metatag;
import com.univ.objetspartages.om.Photo;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.objetspartages.om.Ressource;
import com.univ.objetspartages.util.RessourceUtil;
import com.univ.utils.FileUtil;
import com.univ.utils.ImageInfo;
import com.univ.utils.PhotoUtil;

/**
 * The Class MigrationPhototheque.
 * 
 * @author Romain
 * 
 *         To change the template for this generated type comment go to Window - Preferences - Java - Code Generation - Code and Comments
 */
public class MigrationPhototheque implements Runnable {

	private static final String LOG_DEBUG = "DEBUG";

	private static final String LOG_ERROR = "ERROR";

	private static final String LOG_INFO = "INFO";

	private static ch.qos.logback.classic.Logger logger = (ch.qos.logback.classic.Logger) LoggerFactory.getLogger(MigrationPhototheque.class);

	private static JspWriter outWriter = null;

	public static void main(String[] args) {
		MigrationPhototheque capture = new MigrationPhototheque();
		Thread thread = (new Thread(capture));
		thread.start();
		try {
			thread.join();
		} catch (Exception e) {
			logger.error("Echec thread.join()", e);
		}
	}

	/**
	 * Migrer.
	 * 
	 * @param out
	 *            the out
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public static void migrer(JspWriter out) throws Exception {
		outWriter = out;
		FileAppender<ILoggingEvent> appender;
		MigrationPhototheque mig = new MigrationPhototheque();
		appender = mig.initialisationLog();
		mig.migrerPhototheque();
		logger.detachAppender(appender);
		//appender.close();
	}

	public void migrerPhototheque() throws Exception {
		logger.info("Date Début migration phototheque: " + new java.util.Date(System.currentTimeMillis()));
		ecrireLog("DEBUT du traitement ...", LOG_INFO);
		ContextePage ctx = new ContextePage("");
		ctx.setConnectionLongue(true);
		// migration des photos en media
		migrerPhotos(ctx);
		// migration des fichiergw en ressource + creation d'un media
		migrerFichiergw(ctx);
		migrerReferenceAuMedia(ctx);
		ecrireLog("FIN du traitement.", LOG_INFO);
		ctx.release();
		logger.info("Date fin migration phototheque: " + new java.util.Date(System.currentTimeMillis()));
	}

	/**
	 * Migrer documents.
	 *
	 * @param ctx
	 *            the ctx
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public static void migrerPhotos(ContextePage ctx) throws Exception {
		ecrireLog("*** Migration des photos en medias de type photo ***", LOG_INFO);
		//out.flush();
		int nbModifs = 0;
		int nbErreur = 0;
		int nbMigres = 0;
		// creation du répertoire /medias/photo/ si nécessaire
		String repertoire = MediathequeHelper.getAbsolutePath();
		File fRepertoire = new File(repertoire);
		if (!fRepertoire.exists()) {
			fRepertoire.mkdir();
		}
		repertoire = repertoire + File.separator + "photo";
		fRepertoire = new File(repertoire);
		if (!fRepertoire.exists()) {
			fRepertoire.mkdir();
		}
		Photo photo = new Photo();
		photo.init();
		photo.setCtx(ctx);
		String sRepertoirePhotoOld = "";
		// tous les photos presentes dans la base sont migrees
		photo.select("");
		while (photo.nextItem()) {
			try {
				sRepertoirePhotoOld = "";
				boolean isAlreadySync = false;
				Media media = new Media();
				media.setCtx(ctx);
				media.init();
				media.setIdMedia(photo.getIdPhoto());
				media.setTitre(photo.getTitre());
				media.setLegende(photo.getLegende());
				media.setDescription(photo.getDescription());
				String url = photo.getUrl();
				sRepertoirePhotoOld = url.substring(0, url.indexOf("/") + 1);
				if (url.endsWith("gif") || url.endsWith("GIF")) {
					media.setFormat("image/gif");
				} else {
					media.setFormat("image/jpeg");
				}
				url = url.substring(url.indexOf("/") + 1);
				media.setUrl(url);
				media.setSource(url);
				media.setIsMutualise("0");
				media.setTypeRessource("photo");
				media.setTypeMedia(photo.getTypePhoto());
				media.setCodeRubrique(photo.getCodeRubrique());
				media.setCodeRattachement(photo.getCodeStructure());
				media.setUrlVignette("");
				media.setCodeRedacteur(photo.getCodeRedacteur());
				media.setDateCreation(photo.getDateCreation());
				media.setMetaKeywords(photo.getMetaKeywords());
				// gestion des donnees specifiques
				String sDatas = "";
				Properties properties = new Properties();
				properties.load(new ByteArrayInputStream(sDatas.getBytes()));
				properties.setProperty("LARGEUR", photo.getLargeur().toString());
				properties.setProperty("HAUTEUR", photo.getHauteur().toString());
				ByteArrayOutputStream baos = new ByteArrayOutputStream();
				properties.store(baos, "");
				sDatas = baos.toString();
				media.setSpecificData(sDatas);
				// gestion des traductions
				Hashtable<String, String> hLangue = LangueUtil.getListeLangues();
				sDatas = "";
				properties = new Properties();
				properties.load(new ByteArrayInputStream(sDatas.getBytes()));
				// français par defaut, supprimé ici
				if (hLangue.containsKey("0")) {
					hLangue.remove("0");
				}
				String sMetaKeywords = photo.getMetaKeywords();
				if (hLangue.size() > 0) {
					String sLangue = "";
					String sLg = "";
					int idxDebut = 0;
					int idxFin = 0;
					Enumeration<String> en = hLangue.keys();
					while (en.hasMoreElements()) {
						sLg = en.nextElement();
						if ((idxDebut = sMetaKeywords.indexOf("[" + sLg)) != -1) {
							idxFin = sMetaKeywords.indexOf("]", idxDebut);
							if (idxFin != -1) {
								sLangue = sMetaKeywords.substring(idxDebut + 1, idxFin);
								sMetaKeywords = sMetaKeywords.substring(0, idxDebut) + sMetaKeywords.substring(idxFin + 1, sMetaKeywords.length());
								String[] itemLg = sLangue.split("/", -2);
								if (itemLg.length == 3) {
									properties.setProperty("LEGENDE_" + sLg, itemLg[1]);
									properties.setProperty("DESCRIPTION_" + sLg, itemLg[2]);
								} else {
									properties.setProperty("LEGENDE_" + sLg, "");
									properties.setProperty("DESCRIPTION_" + sLg, "");
								}
							}
						} else {
							properties.setProperty("LEGENDE_" + sLg, "");
							properties.setProperty("DESCRIPTION_" + sLg, "");
						}
					}
				}
				baos = new ByteArrayOutputStream();
				properties.store(baos, "");
				sDatas = baos.toString();
				media.setTraductionData(sDatas);
				// deplacement du fichier
				File oldPhoto = new File(photo.getPathAbsolu());
				File newMedia = null;
				if (oldPhoto.exists()) {
					newMedia = new File(media.getPathAbsolu());
					if (!newMedia.exists()) {
						int nResCopierFichier = FileUtil.copierFichier(oldPhoto, newMedia);
						media.setPoids(new Integer(nResCopierFichier / 1024));
					} else {
						isAlreadySync = true;
						nbMigres++;
					}
				} else {
					throw new Exception("fichier " + photo.getPathAbsolu() + " inexistant");
				}
				if (!isAlreadySync) {
					// generation de la vignette
					if (newMedia != null) {
						try {
							ImageInfo ii;
							InputStream inpStream;
							int largeur;
							int hauteur;
							String critere = PropertyHelper.getCoreProperty(MediathequeHelper.CRITERES_VIGNETTE_PROPERTIES_KEY);
							String[] lstLimite = critere.split("/", -2);
							boolean vignetteOk = false;
							String PathVignette = PropertyHelper.getCoreProperty("phototheque.path") + File.separator + sRepertoirePhotoOld + "v_" + media.getUrl();
							File file = new File(PathVignette);
							if (file.exists()) {
								ii = new ImageInfo();
								inpStream = new FileInputStream(PathVignette);
								ii.setInput(inpStream);
								ii.check();
								ii.close();
								lstLimite = critere.split("/", -2);
								largeur = Integer.parseInt(lstLimite[1]);
								hauteur = Integer.parseInt(lstLimite[2]);
								if (ii.getWidth() >= largeur || ii.getHeight() >= hauteur) {
									FileUtil.copierFichier(file, new File(repertoire + File.separator + "v_" + media.getUrl()));
									vignetteOk = true;
								}
							}
							if (vignetteOk == false) {
								File fVignette = newMedia;
								lstLimite = critere.split("/", -2);
								ii = new ImageInfo();
								inpStream = new FileInputStream(photo.getPathAbsolu());
								ii.setInput(inpStream);
								ii.check();
								ii.close();
								largeur = Integer.parseInt(lstLimite[1]);
								hauteur = Integer.parseInt(lstLimite[2]);
								//On ne redimensionne que si la taille était plus grande ...
								if (largeur < ii.getWidth() && hauteur < ii.getHeight()) {
									String extension = FileUtil.getExtension(media.getSource());
									//resize cree une copie de l'image lorsque le dernier param est a false
									fVignette = resize(fVignette.getAbsolutePath(), extension, largeur, hauteur, false);
									FileUtil.copierFichier(fVignette, new File(repertoire + File.separator + "v_" + media.getUrl()));
								} else {
									FileUtil.copierFichier(fVignette, new File(repertoire + File.separator + "v_" + media.getUrl()));
								}
							}
							media.setUrlVignette("v_" + media.getUrl());
						} catch (Exception e) {
							ecrireLog("ERREUR lors de la creation de vignette de la photo " + media.getUrl() + " Exception :" + e, LOG_ERROR);
						}
					}
					// creation de l'objet media
					try {
						media.add();
						nbModifs++;
					} catch (Exception e) {}
				}
			} catch (Exception e) {
				ecrireLog("Erreur sur la photo " + photo.getPathAbsolu() + " : " + e.getMessage(), LOG_ERROR);
				nbErreur++;
			}
		}
		ecrireLog(nbModifs + " photo(s) migrée(s) ", LOG_INFO);
		ecrireLog(nbMigres + " photo(s) déja migrée(s).", LOG_INFO);
		ecrireLog(nbErreur + " erreur(s).", LOG_INFO);
	}

	private static void migrerReferenceAuMedia(ContextePage ctx) throws Exception {
		Metatag metaDonnees = new Metatag();
		metaDonnees.setCtx(ctx);
		metaDonnees.init();
		metaDonnees.select("WHERE META_LISTE_REFERENCES LIKE '%[photo;%'");
		while (metaDonnees.nextItem()) {
			String listeReference = metaDonnees.getMetaListeReferences();
			String[] listeIdMedia = StringUtils.substringsBetween(listeReference, "[photo;", "]");
			for (String idMedia : listeIdMedia) {
				migrerRessource(ctx, metaDonnees, idMedia);
			}
		}
	}

	private static void migrerRessource(OMContext ctx, Metatag metaDonnees, String idMedia) throws Exception {
		if (isRessourceAMigrer(idMedia)) {
			String codeParent = formaterCodeParent(metaDonnees);
			Ressource ressource = new Ressource();
			ressource.init();
			ressource.setCtx(ctx);
			int nbResultat = ressource.select("WHERE ID_MEDIA=" + idMedia);
			boolean dejaImportee = Boolean.FALSE;
			while (ressource.nextItem() && !dejaImportee) {
				dejaImportee = codeParent.equals(ressource.getCodeParent());
			}
			Media media = new Media();
			media.init();
			media.setCtx(ctx);
			boolean mediaExist = media.select("WHERE ID_MEDIA=" + idMedia) > 0;
			if (!dejaImportee && mediaExist) {
				ressource.init();
				ressource.setIdMedia(Long.valueOf(idMedia));
				ressource.setCodeParent(codeParent);
				ressource.setOrdre(nbResultat);
				ressource.setEtat("1");
				ressource.add();
				ecrireLog("reference du media " + idMedia + " à la fiche " + codeParent + " importée", LOG_INFO);
			} else {
				ecrireLog("reference du media " + idMedia + " à la fiche " + codeParent + " déjà importée", LOG_INFO);
			}
		}
	}

	private static String formaterCodeParent(Metatag metaDonnees) {
		return metaDonnees.getMetaIdFiche() + ",TYPE=IMG_" + metaDonnees.getMetaCodeObjet() + ",CODE=" + metaDonnees.getMetaCode();
	}

	/**
	 * L"idMedia commence par F si c'est un fichiergw. Les ressources sont déjà gérés.
	 */
	private static boolean isRessourceAMigrer(String idMedia) {
		return !idMedia.startsWith("F");
	}

	public static File resize(String path, String format, int newLargeur, int newHauteur, boolean replace) throws Exception {
		File fOld = new File(path);
		File fNew = new File(fOld.getParent(), "tmp_" + System.currentTimeMillis());
		Iterator<ImageWriter> writers = ImageIO.getImageWritersByFormatName(format);
		ImageWriter writer = writers.next();
		BufferedImage image = ImageIO.read(fOld);
		BufferedImage bImage = PhotoUtil.scale(image, newLargeur, newHauteur, format);
		if (!image.equals(bImage)) {
			ImageOutputStream ios = ImageIO.createImageOutputStream(fNew);
			writer.setOutput(ios);
			writer.write(bImage);
			ios.close();
			writer.dispose();
		} else {
			fNew = fOld;
		}
		if (replace) {
			FileUtil.copierFichier(fNew, fOld, true);
			return fOld;
		}
		return fNew;
	}

	/**
	 * Migrer photos albums.
	 *
	 * @param ctx
	 *            the ctx
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public static void migrerFichiergw(ContextePage ctx) throws Exception {
		ecrireLog("*** Migration des fichiergw ***", LOG_INFO);
		//out.flush();
		int nbModifs = 0;
		int nbMigres = 0;
		int nbErreur = 0;
		Fichiergw f = new Fichiergw();
		f.setCtx(ctx);
		f.init();
		// on recupere ts les fichiergw sauf les vignettes
		if (f.select("WHERE ETAT = '1' AND TYPE_FICHIER != 'vignette'") > 0) {
			while (f.nextItem()) {
				ContextePage ctx2 = new ContextePage("");
				try {
					Long idMedia = new Long(0);
					Media media = new Media();
					media.setCtx(ctx2);
					media.init();
					ecrireLog("Migration fichier " + f.getPathAbsolu() + ";IdFichiergw=" + f.getIdFichiergw() + "code=" + f.getCode(), LOG_INFO);
					// pas de media associe
					if (f.getIdPhoto().equals("")) {
						try {
							idMedia = new Long(f.getCode());
							media.setIdMedia(idMedia);
						} catch (Exception e) {
							// le fichier ne possede pas de code
						}
						media.setTitre(f.getLibelleFichierJoint());
						media.setLegende("");
						media.setDescription(f.getCommentaireVersion());
						media.setFormat(f.getFormatFichierJoint());
						media.setUrl(f.getPathFichierJoint());
						media.setSource(f.getNomFichierJoint());
						media.setIsMutualise("2");
						media.setCodeRedacteur(f.getAuteurVersion());
						media.setDateCreation(f.getDateVersion());
						try {
							media.setPoids(new Integer(f.getPoidsFichierJoint()));
						} catch (Exception e) {}
						if (f.getTypeFichier().equals("fichier") || f.getTypeFichier().equals("document") || f.getTypeFichier().equals("lien")) {
							media.setTypeRessource("fichier");
						} else if (f.getTypeFichier().equals("photo") || f.getTypeFichier().equals("image") || f.getTypeFichier().equals("img") || f.getTypeFichier().equals("logo") || f.getTypeFichier().equals(
								"galerie")) {
							media.setTypeRessource("photo");
						}
						// gestion de la vignette
						if (f.getIdVignette().length() > 0) {
							Fichiergw vignette = new Fichiergw();
							vignette.setCtx(ctx2);
							vignette.init();
							try {
								vignette.setIdFichiergw(new Long(f.getIdVignette()));
								vignette.retrieve();
								media.setUrlVignette(vignette.getPathFichierJoint());
								// deplacement du fichier
								File file = new File(vignette.getPathAbsolu());
								//La vignette doit se trouver sous fichiergw
								if (file.exists()) {
									File file2 = new File(media.getPathVignetteAbsolu());
									if (!file2.exists()) {
										FileUtil.copierFichier(file, file2);
									}
								} else {
									//Si ce n'est pas le cas on regarde dans photo
									File filePhoto = new File(PropertyHelper.getCoreProperty("phototheque.path") + File.separator + vignette.getPathFichierJoint());
									if (filePhoto.exists()) {
										File file2 = new File(media.getPathVignetteAbsolu());
										if (!file2.exists()) {
											FileUtil.copierFichier(filePhoto, file2);
										}
									}
								}
							} catch (Exception e) {}
						}
						// ajout du media
						try {
							if (media.select("WHERE URL='" + media.getUrl() + "'") == 0) {
								media.add();
								idMedia = media.getIdMedia();
							} else {
								//FDE 2010048 correction : si le media existe déja, on le recupere pour creer eventuellement d'autres ressources
								media.nextItem();
								idMedia = media.getIdMedia();
							}
						} catch (Exception e) {}
					} else {
						// on teste la presence du media photo
						try {
							media.setIdMedia(new Long(f.getIdPhoto()));
							media.retrieve();
							idMedia = media.getIdMedia();
						} catch (Exception e) {}
					}
					if (!idMedia.equals(new Long(0))) {
						boolean objetParentExiste = Boolean.FALSE;
						// creation de la ressource
						Ressource ressource = new Ressource();
						ressource.setCtx(ctx2);
						ressource.init();
						if (f.getCodeParent().matches("^([0-9]+)(,TYPE=)([0-9]+)$")) {
							ressource.setCodeParent(f.getCodeParent() + ",NO=1");
						} else {
							ressource.setCodeParent(f.getCodeParent());
						}
						ressource.setEtat(f.getEtat());
						ressource.setIdMedia(idMedia);
						try {
							ressource.setOrdre(new Integer(f.getNumeroVersion()));
						} catch (Exception e) {}
						//On test l'existence du parent avant de creer la ressource
						String codeParent = ressource.getCodeParent();
						FicheUniv ficheUniv = null;
						// en apercu on ne controle pas la fiche car elle a déja été supprimée
						if (isAppartientAFicheUniv(ressource) && isFichierAImporter(ressource.getEtat())) {
							String idFiche = codeParent.substring(0, codeParent.indexOf(","));
							String typeObjet = recupererTypeFicheParent(codeParent);
							ficheUniv = ReferentielObjets.instancierFiche(ReferentielObjets.getNomObjet(typeObjet));
							try {
								if (ficheUniv != null) {
									ficheUniv.setCtx(ctx2);
									ficheUniv.init();
									ficheUniv.setIdFiche(new Long(idFiche));
									ficheUniv.retrieve();
									//retreive genere une excption si on ne trouve pas la fiche, on ne met donc pas le booleen a true
									objetParentExiste = Boolean.TRUE;
								}
							} catch (Exception e) {
								ecrireLog("Erreur lors du test de l'existence de la FicheUniv : " + idFiche + " Exception: " + e + "Class=" + ficheUniv.getClass(), LOG_ERROR);
							}
						} else if (isFichierEspaceCollaboratif(codeParent) && isFichierAImporter(ressource.getEtat())) {
							Espacecollaboratif espaceParent = Espacecollaboratif.getEspace(getCodeEspaceCollaboratif(codeParent));
							objetParentExiste = espaceParent != null;
						}
						//Si la ressource n'existe pas et que la fiche parente existe bien
						if (ressource.select("WHERE CODE_PARENT='" + ressource.getCodeParent() + "' AND ID_MEDIA='" + idMedia + "' ") == 0 && objetParentExiste) {
							ressource.add();
							ecrireLog("=> OK", LOG_INFO);
							nbModifs++;
						} else {
							if (objetParentExiste) {
								ecrireLog("=> déja migré", LOG_INFO);
								nbMigres++;
							}
						}
					} else {
						ecrireLog("=> fichiergw code 0, pas de creation de media ", LOG_INFO);
					}
				} catch (Exception e) {
					if (e instanceof SQLException) {
						nbMigres++;
						ecrireLog("=> déja migré", LOG_INFO);
					} else {
						ecrireLog("Erreur sur le fichier " + f.getPathAbsolu() + " : " + e.getMessage(), LOG_ERROR);
						nbErreur++;
					}
				} finally {
					if (ctx2 != null) {
						ctx2.release();
					}
				}
			}
		}
		ecrireLog(nbModifs + " fichiergw(s) migré(s).", LOG_INFO);
		ecrireLog(nbMigres + " fichiergw(s) déja migre(s).", LOG_INFO);
		ecrireLog(nbErreur + " erreur(s).", LOG_INFO);
	}

	private static String getCodeEspaceCollaboratif(String codeParent) {
		String codeEspace = null;
		String[] codeParentSpliter = codeParent.split(",");
		if (codeParentSpliter.length > 0) {
			codeEspace = codeParentSpliter[0];
		}
		return codeEspace;
	}

	private static Boolean isAppartientAFicheUniv(Ressource ressource) {
		String codeParent = ressource.getCodeParent();
		return StringUtils.isNotEmpty(codeParent) && !codeParent.startsWith("TYPE") && !isFichierEspaceCollaboratif(codeParent);
	}

	private static boolean isFichierAImporter(String etat) {
		return !"0".equals(etat) && !"2".equals(etat);
	}

	private static boolean isFichierEspaceCollaboratif(String codeParent) {
		return codeParent.contains("LOGO_ESPACE");
	}

	private static String recupererTypeFicheParent(String codeParent) {
		String typeObjet = codeParent.substring(codeParent.indexOf("TYPE=") + 5);
		if (typeObjet.length() > 4) {
			if (typeObjet.indexOf("_") != -1) {
				typeObjet = typeObjet.substring(typeObjet.indexOf("_") + 1, typeObjet.indexOf("_") + 5);
			}
			if (typeObjet.indexOf(",") != -1) {
				typeObjet = typeObjet.substring(0, typeObjet.indexOf(","));
			}
		}
		return typeObjet;
	}

	/**
	 * Migrer photos albums.
	 * 
	 * @param out
	 *            the out
	 * @param ctx
	 *            the ctx
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public static void migrerVignette(JspWriter out, ContextePage ctx) throws Exception {
		out.println("<br />*** Migration des vignettes ***<br />");
		out.flush();
		int nbModifs = 0;
		int nbMigres = 0;
		int nbErreur = 0;
		Media media = new Media();
		media.setCtx(ctx);
		media.init();
		String newPath = "";
		String oldPath = "";
		String mediathequePath = MediathequeHelper.getAbsolutePath();
		String fichiergwPath = RessourceUtil.getAbsolutePath();
		// on recupere les medias avec vignette
		if (media.select("WHERE URL_VIGNETTE != ''") > 0) {
			while (media.nextItem()) {
				newPath = "";
				oldPath = "";
				try {
					// new path
					newPath = mediathequePath;
					if (media.getTypeRessource().length() > 0) {
						newPath += "/" + media.getTypeRessource().toLowerCase();
					}
					newPath += "/" + media.getUrlVignette();
					File fileCible = new File(newPath);
					if (!fileCible.exists()) {
						// ancien path
						oldPath = fichiergwPath + "/" + media.getUrlVignette();
						File fileOrig = new File(oldPath);
						if (fileOrig.exists()) {
							FileUtil.copierFichier(fileOrig, fileCible);
							nbModifs++;
						} else {
							nbErreur++;
							out.println("Erreur vignette inexistante : " + fileOrig + "<br />");
							out.flush();
						}
					}
				} catch (Exception e) {
					nbErreur++;
					out.println("Erreur " + e.getMessage() + "<br />");
					out.flush();
				}
				nbMigres++;
			}
		}
		out.println(nbModifs + " vignette(s) déplacée(s) sur un total de " + nbMigres + "<br />");
		out.println(nbErreur + " erreur(s).<br />");
		out.flush();
	}

	@Override
	public void run() {
		FileAppender<ILoggingEvent> appender;
		try {
			// On crée le nouveau 'logger'.
			appender = initialisationLog();
			migrerPhototheque();
			logger.detachAppender(appender);
			//appender.close();
		} catch (Exception e) {
			logger.error("Erreur lors du traitement de migration > Message :" + e.getMessage() + "StackTrace :");
		}
	}

	private FileAppender<ILoggingEvent> initialisationLog() throws Exception {
		FileAppender<ILoggingEvent> appender = null;
		String sFileLog = "";
		try {
			// On crée le nouveau 'logger'.
			String logPath = WebAppUtil.getLogsPath();
			sFileLog = logPath + File.separator + "MigrationPhototheque.log";
			appender = new FileAppender<ILoggingEvent>();
			//( new SimpleLayout(),sFileLog);
			appender.setFile(sFileLog);
			appender.setEncoder(new PatternLayoutEncoder());
			// On 'démarre' l'appender.
			appender.setAppend(false);
			//appender.activateOptions();
			logger.detachAndStopAllAppenders();
			// On ajoute l'appender file au logger.
			logger.addAppender(appender);
		} catch (Exception e) {
			// TODO Auto-generated catch block
			logger.error("Erreur lors du traitement de migration > Message :" + e.getMessage() + "StackTrace :");
		}
		return appender;
	}

	private static void ecrireLog(String sTexteLog, String sMode) throws Exception {
		if (outWriter != null) {
			outWriter.println(sTexteLog + "<br />");
			outWriter.flush();
		}
		if (sMode.equals(LOG_INFO)) {
			logger.info(sTexteLog);
		} else if (sMode.equals(LOG_ERROR)) {
			logger.error(sTexteLog);
		} else if (sMode.equals(LOG_DEBUG)) {
			logger.debug(sTexteLog);
		}
	}


	public static void setOutWriter(JspWriter outWriter) {
		MigrationPhototheque.outWriter = outWriter;
	}
}
