/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.multisites;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import com.univ.multisites.bean.impl.InfosSiteImpl;
import com.univ.objetspartages.cache.CacheInfosSiteManager;
import com.univ.objetspartages.om.InfosRubriques;
import com.univ.objetspartages.om.Rubrique;

/**
 * Façade pour accéder au site
 */
public class Site {

	public static Map<String, InfosSite> getListeInfosSites() {
		Map<String, InfosSite> infosSiteParCode = CacheInfosSiteManager.getDonneesSites().getInfosSitesParCode();
		if (infosSiteParCode == null) {
			infosSiteParCode = Collections.emptyMap();
		}
		return infosSiteParCode;
	}

	public static InfosSite getSitePrincipal() {
		return CacheInfosSiteManager.getDonneesSites().getSitePrincipal();
	}

	public static InfosSite renvoyerItemSiteParRubrique(final String rubrique) {
		Map<String, InfosSite> infosSiteParRubrique = CacheInfosSiteManager.getDonneesSites().getInfosSitesParCodeRubrique();
		if (infosSiteParRubrique == null) {
			infosSiteParRubrique = Collections.emptyMap();
		}
		return infosSiteParRubrique.get(rubrique);
	}

	public static InfosSite renvoyerItemSite(final String alias) {
		Map<String, InfosSite> infosSiteParCode = CacheInfosSiteManager.getDonneesSites().getInfosSitesParCode();
		if (infosSiteParCode == null) {
			infosSiteParCode = Collections.emptyMap();
		}
		InfosSite itemSite = infosSiteParCode.get(alias);
		if (itemSite == null) {
			itemSite = getSitePrincipal();
		}
		return itemSite;
	}

	/**
	 * Récupération d'un site stocké en mémoire.
	 *
	 * @param host
	 *            the host
	 *
	 * @return the infos site
	 */
	public static InfosSite renvoyerItemSiteParHost(final String host) {
		Map<String, InfosSite> infosSiteParHost = CacheInfosSiteManager.getDonneesSites().getInfosSitesParHost();
		if (infosSiteParHost == null) {
			infosSiteParHost = Collections.emptyMap();
		}
		InfosSite siteParHost = infosSiteParHost.get(host);
		if (siteParHost == null) {
			siteParHost = new InfosSiteImpl();
		}
		return siteParHost;
	}

	/**
	 * Récupère le site de la rubrique dont le code est fourni en paramètre.
	 *
	 *
	 * @param codeRubrique
	 *            Le code de la rubrique dont on souhaite connaitre le site
	 * @param renvoyerSiteGlobal
	 *            si aucun site n'est trouvé, doit on renvoyer le site principal?
	 *
	 * @return l'InfosSite de la rubrique ou null si le code est vide ou que renvoyerSiteGlobal est à false et que le site n'est pas trouvé ou le site principal si
	 *         renvoyerSiteGlobal est à true
	 */
	public static InfosSite determinerSiteRubrique(final String codeRubrique, final boolean renvoyerSiteGlobal) {
		if (StringUtils.isEmpty(codeRubrique)) {
			return null;
		}
		InfosRubriques infosRubrique = Rubrique.renvoyerItemRubrique(codeRubrique);
		while (infosRubrique.getNiveau() > 0) {
			final String codeRubriqueCourante = infosRubrique.getCode();
			InfosSite infosSiteRubrique = CacheInfosSiteManager.getDonneesSites().getInfosSitesParCodeRubrique().get(codeRubriqueCourante);
			if (infosSiteRubrique == null) {
				infosRubrique = infosRubrique.getRubriqueMere();
			} else {
				return infosSiteRubrique;
			}
		}
		if (renvoyerSiteGlobal) {
			return getSitePrincipal();
		} else {
			return null;
		}
	}

	/**
	 * Determiner site rubrique.
	 *
	 * @param codeRubrique
	 *            Le code de la rubrique dont on souhaite connaitre le site
	 *
	 * @return l'InfosSite de la rubrique ou null si le code est vide ou que renvoyerSiteGlobal est à false et que le site n'est pas trouvé ou le site principal si
	 *         renvoyerSiteGlobal est à true
	 */
	public static InfosSite determinerSiteRubrique(final String codeRubrique) {
		return determinerSiteRubrique(codeRubrique, true);
	}

	/**
	 * Calcule le site global à partir du cache.
	 *
	 * @return le site global
	 * @deprecated utiliser getSitePrincipal
	 */
	@SuppressWarnings("deprecation")
	@Deprecated
	public static InfosSite getSiteGlobal() {
		return CacheInfosSiteManager.getDonneesSites().getSiteGlobal();
	}

	/**
	 * liste des hosts et alias des sites actifs
	 *
	 * @return la liste des hosts et alias
	 */
	public static ArrayList<String> getListeCompleteHostsEtAlias() {
		return new ArrayList<>(CacheInfosSiteManager.getDonneesSites().getUrlsDefinis());
	}
}
