/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.multisites.dao;

import java.util.Collection;

import com.jsbsoft.jtf.exception.ErreurDonneeNonTrouve;
import com.univ.multisites.InfosSite;

/**
 * Gestion des accés à la source de données qui contient les {@link InfosSite}
 * 
 * @author pierre.cosson
 * 
 */
public interface InfosSiteDao {

	/**
	 * Récupérer un {@link InfosSite} en spécifiant son code (appelé aussi alias).
	 * 
	 * @param code
	 *            Code du {@link InfosSite} à récupérer. Ce paramétre était aussi appelé alias.
	 * @return L'objet {@link InfosSite} chargé correspondant au code.
	 * @exception <ul>
	 *            <li>Exception {@link ErreurDonneeNonTrouve} : le code ne correspond à aucun site</li>
	 *            <li>Autres erreurs durant le chargement du site</li>
	 *            </ul>
	 */
	InfosSite getInfosSite(String code) throws Exception;

	/**
	 * Récupérer tous les {@link InfosSite} gérés par la source de données.
	 * 
	 * @return La liste des {@link InfosSite} contenus par la source de données. Cette liste peut être vide.
	 * @throws Exception
	 *             Erreur durant les traitements.
	 */
	Collection<InfosSite> getListeInfosSites() throws Exception;

	/**
	 * Crééer un {@link InfosSite} <br/>
	 * <br/>
	 * <br/>
	 * REMARQUE : l'idenfiant du bean sera mis à jour par la source de données.
	 * 
	 * @param infosSite
	 *            l'élément à créer dans la source de données.
	 * @throws Exception
	 *             <ul>
	 *             <li>Erreurs d'écriture dans la source de données</li>
	 *             </ul>
	 */
	void creer(InfosSite infosSite) throws Exception;

	/**
	 * Mettre à jour la {@link InfosSite} en source de données.
	 * 
	 * @param infosSite
	 *            L' {@link InfosSite} à mettre à jour avec ses données à jour.
	 * @throws Exception
	 *             <ul>
	 *             <li> {@link ErreurDonneeNonTrouve} : aucun {@link InfosSite} ne correspond à l'identifiant du bean passé en paramètre (pas de modification possible).</li>
	 *             <li>Erreurs d'écriture dans la source de données</li>
	 *             </ul>
	 */
	void miseAJour(InfosSite infosSite) throws Exception;

	/**
	 * Supprimer un {@link InfosSite} via son code.
	 * 
	 * @param code
	 *            code de l'{@link InfosSite} à supprimer.
	 * @throws Exception
	 *             <ul>
	 *             <li>Erreurs d'écriture dans la source de données</li>
	 *             </ul>
	 */
	void supprimer(String code) throws Exception;
}
