/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.multisites.service.cache;

import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.univ.multisites.InfosSite;
import com.univ.multisites.bean.impl.InfosSiteImpl;
import com.univ.multisites.service.ServiceInfosSiteFactory;
import com.univ.utils.URLResolver;

/**
 * Bean contenant la liste des sites de l'application en fonction des critères les plus souvent utilisés : les hosts, les code des rubriques, les alias, ...
 *
 */
public class DonneesInfosSite implements Serializable {

	/**
	 *
	 */
	private static final long serialVersionUID = -5768970279225460911L;

	private static final Logger LOGGER = LoggerFactory.getLogger(DonneesInfosSite.class);

	private final Map<String, InfosSite> infosSitesParCode;

	private final Map<String, InfosSite> infosSitesParHost;

	private final Map<String, InfosSite> infosSitesParCodeRubrique;

	private final Set<String> urlsDefinis;

	private InfosSite sitePrincipal;

	private InfosSite siteGlobal;

	private static final String HTTPS = "https";

	private static final String HTTP = "http";

	public DonneesInfosSite() {
		final Collection<InfosSite> listeInfosSite = getInfosSitesDefinis();
		infosSitesParCode = new HashMap<>(listeInfosSite.size());
		infosSitesParCodeRubrique = new HashMap<>(listeInfosSite.size());
		infosSitesParHost = new HashMap<>();
		urlsDefinis = new HashSet<>();
		for (final InfosSite infosSite : listeInfosSite) {
			if (infosSite.isActif()) {
				if (infosSite.isSitePrincipal()) {
					sitePrincipal = infosSite;
				}
				if (StringUtils.isEmpty(infosSite.getCodeRubrique())) {
					siteGlobal = infosSite;
				}
				infosSitesParCode.put(infosSite.getAlias(), infosSite);
				infosSitesParCodeRubrique.put(infosSite.getCodeRubrique(), infosSite);
				if (StringUtils.isNotEmpty(infosSite.getHttpHostname())) {
					infosSitesParHost.put(infosSite.getHttpHostname(), infosSite);
					urlsDefinis.add(URLResolver.getBasePath(infosSite, Boolean.FALSE));
				}
				if (StringUtils.isNotEmpty(infosSite.getHttpsHostname())) {
					urlsDefinis.add(URLResolver.getBasePath(infosSite, Boolean.TRUE));
					infosSitesParHost.put(infosSite.getHttpsHostname(), infosSite);
				}
				for (final String hostAlias : infosSite.getListeHostAlias()) {
					if (StringUtils.isNotEmpty(hostAlias)) {
						infosSitesParHost.put(hostAlias, infosSite);
						urlsDefinis.add(URLResolver.getBasePath(HTTP, hostAlias, infosSite.getHttpPort()));
						urlsDefinis.add(URLResolver.getBasePath(HTTPS, hostAlias, infosSite.getHttpsPort()));
					}
				}
			}
		}
		if (sitePrincipal == null && infosSitesParHost.size() >= 1) {
			sitePrincipal = (InfosSite) CollectionUtils.get(infosSitesParHost.values(), 0);
		}
		if (siteGlobal == null) {
			siteGlobal = sitePrincipal;
		}
		LOGGER.info("Chargement de " + infosSitesParCode.size() + " site(s) OK");
	}

	private Collection<InfosSite> getInfosSitesDefinis() {
		Collection<InfosSite> listeInfosSite = null;
		try {
			listeInfosSite = ServiceInfosSiteFactory.getServiceInfosSite().getListeTousInfosSites();
			if(CollectionUtils.isNotEmpty(listeInfosSite)) {
				boolean hasSitePrincipal = false;
				for(InfosSite siteCourant : listeInfosSite){
					if(siteCourant.isSitePrincipal()){
						hasSitePrincipal = true;
						break;
					}
				}
				if(!hasSitePrincipal) {
					final InfosSiteImpl newSitePrincipal = (InfosSiteImpl) listeInfosSite.iterator().next();
					newSitePrincipal.setSitePrincipal(Boolean.TRUE);
					ServiceInfosSiteFactory.getServiceInfosSite().modifier(newSitePrincipal, null); //Le deuxième paramètre n'est pas utilisé dans l'implémentation ServiceInfosSiteProcessus
				}
			}
		} catch (final Exception e) {
			LOGGER.error("impossible de récupérer la liste des sites déclarés dans l'application");
		}
		if (listeInfosSite == null) {
			listeInfosSite = Collections.emptyList();
		}
		return listeInfosSite;
	}

	/**
	 * 
	 * @return la liste des sites actifs de l'application par leur code ou une map vide si il n'y en a aucun de déclaré.
	 */
	public Map<String, InfosSite> getInfosSitesParCode() {
		return infosSitesParCode;
	}

	/**
	 * 
	 * @return la liste des sites actifs de l'application par leur host.
	 */
	public Map<String, InfosSite> getInfosSitesParHost() {
		return infosSitesParHost;
	}

	/**
	 * 
	 * @return la liste des sites actifs de l'application par leur code de rubrique.
	 */
	public Map<String, InfosSite> getInfosSitesParCodeRubrique() {
		return infosSitesParCodeRubrique;
	}

	/**
	 * 
	 * @return l'ensemble des urls des sites actifs.
	 */
	public Set<String> getUrlsDefinis() {
		return urlsDefinis;
	}

	/**
	 * 
	 * @return le site principal de l'application
	 */
	public InfosSite getSitePrincipal() {
		return sitePrincipal;
	}

	/**
	 * 
	 * @return le site global. (pas de code de rubrique ?!?)
	 * @deprecated utiliser getSitePrincipal
	 */
	@Deprecated
	public InfosSite getSiteGlobal() {
		return siteGlobal;
	}
}
