/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.cache;

import java.util.HashMap;
import java.util.Hashtable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.CodeLibelle;
import com.kdecole.cache.aspect.StaticCacheFlush;
import com.kdecole.cache.aspect.StaticCacheGet;
import com.kportal.extension.ExtensionHelper;
import com.univ.objetspartages.om.InfosLibelle;
import com.univ.objetspartages.om.Libelle;
import com.univ.utils.ContexteDao;

public class CacheLibelleManager {

	public static final String ID_BEAN = "cacheLibelleManager";

	private static final Logger LOGGER = LoggerFactory.getLogger(CacheLibelleManager.class);

	@StaticCacheGet(key = "CacheLibelleManager.getListeInfosLibelles", contextual = false)
	public HashMap<String, InfosLibelle> getListeInfosLibelles() {
		final ContexteDao ctx = new ContexteDao();
		final HashMap<String, InfosLibelle> listeLibelles = new HashMap<>();
		try {
			// Charge en mémoire la liste des libelles
			final Libelle libelle = new Libelle();
			libelle.setCtx(ctx);
			libelle.selectNoCount("");
			int count = 0;
			InfosLibelle infosLibelle = null;
			while (libelle.nextItem()) {
				// Instanciation de l'InfosRubriques
				infosLibelle = new InfosLibelle(libelle);
				// Ajout dans la bibliothèque de rubriques
				listeLibelles.put(infosLibelle.getType() + infosLibelle.getCode().toLowerCase() + infosLibelle.getLangue(), infosLibelle);
				count++;
			}
			LOGGER.info("Chargement de " + count + " libelles OK");
		} catch (final Exception e) {
			LOGGER.info("Erreur de chargement des libelles", e);
		} finally {
			ctx.close();
		}
		return listeLibelles;
	}

	/* Renvoie une Hashtable contenant la concaténation de la table type_libelle.dat et des tables
	 * type_libelle.dat propres à chaque développement spécifique.
	 * Cette Hashtable est chargée en mémoire au premier appel.
	 */
	@StaticCacheGet(key = "CacheLibelleManager.getListeTypesLibelles", contextual = false)
	public Hashtable<String, String> getListeTypesLibelles() {
		final Hashtable<String, String> listeTypesLibelles = new Hashtable<String, String>();
		try {
			for (final String idExt : ExtensionHelper.getExtensionManager().getExtensions().keySet()) {
				Hashtable<String, String> h = CodeLibelle.lireTable(idExt, "type_libelle", null);
				listeTypesLibelles.putAll(h);
				h = CodeLibelle.lireTable(idExt, idExt + "_type_libelle", null);
				listeTypesLibelles.putAll(h);
			}
			LOGGER.info("Chargement des types de libelle OK");
		} catch (final Exception e) {
			LOGGER.info("Erreur de chargement des types de libelle");
		}
		return listeTypesLibelles;
	}

	@StaticCacheFlush(key = "CacheLibelleManager.getListeInfosLibelles", contextual = false)
	public void flushListeInfosLibelles() {}

	@StaticCacheFlush(key = "CacheLibelleManager.getListeTypesLibelles", contextual = false)
	public void flushListeTypesLibelles() {}

	public void flushAll() {
		flushListeInfosLibelles();
		flushListeTypesLibelles();
	}
}