/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.cache;

import java.util.HashMap;
import java.util.Iterator;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.kportal.cache.AbstractCacheManager;
import com.univ.objetspartages.om.InfosRubriques;
import com.univ.objetspartages.om.Media;
import com.univ.objetspartages.om.Rubrique;
import com.univ.utils.ContexteDao;

@Component
public class CacheRubriqueManager extends AbstractCacheManager {

	public static final String ID_BEAN = "cacheRubriqueManager";

	private static final Logger LOGGER = LoggerFactory.getLogger(CacheRubriqueManager.class);

	public static final String KEY_CACHE = "CacheRubriqueManager.cacheRubriques";

	public static CacheRubriqueManager getInstance() {
		return (CacheRubriqueManager) ApplicationContextManager.getCoreContextBean(ID_BEAN);
	}

	private HashMap<String, InfosRubriques> getCacheRubriques() throws Exception {
		final HashMap<String, InfosRubriques> listeRubriques = new HashMap<>();
		// Initialise la rubrique de niveau supérieur
		final InfosRubriques infosRubriqueTop = new InfosRubriques(Rubrique.CODE_RUBRIQUE_ROOT);
		listeRubriques.put(infosRubriqueTop.getCode(), infosRubriqueTop);
		int count = 0;
		try (ContexteDao ctx = new ContexteDao()) {
			// Charge en mémoire la liste des rubriques
			final Rubrique rubrique = new Rubrique();
			rubrique.setCtx(ctx);
			rubrique.selectNoCount(StringUtils.EMPTY);
			InfosRubriques infosRubrique = null;
			while (rubrique.nextItem()) {
				// Instanciation de l'InfosRubriques
				infosRubrique = new InfosRubriques(rubrique);
				// Bandeau de navigation
				if (rubrique.getIdBandeau() != 0L) {
					infosRubrique.setIdMediaBandeau(rubrique.getIdBandeau());
					infosRubrique.setUrlBandeau(Media.getMedia(rubrique.getIdBandeau()).getUrlAbsolue());
				}
				// Picto associé
				if (rubrique.getIdPicto() != 0L) {
					infosRubrique.setIdMediaPicto(rubrique.getIdPicto());
					infosRubrique.setUrlPicto(Media.getMedia(rubrique.getIdPicto()).getUrlAbsolue());
				}
				// Ajout dans la bibliothèque de rubriques
				listeRubriques.put(infosRubrique.getCode(), infosRubrique);
				count++;
			}
		} catch (final Exception e) {
			LOGGER.error("impossible de récupérer les rubriques", e);
			throw e;
		}
		// Effectue les rattachements de rubrique et calcule les niveaux
		final Iterator<InfosRubriques> listRubriqueIt = listeRubriques.values().iterator();
		InfosRubriques rubrique = null;
		InfosRubriques rubriqueMere = null;
		while (listRubriqueIt.hasNext()) {
			rubrique = listRubriqueIt.next();
			if (rubrique.getCode() != Rubrique.CODE_RUBRIQUE_ROOT) {
				rubriqueMere = listeRubriques.get(rubrique.getCodeRubriqueMere());
				if (rubriqueMere == null || rubriqueMere.getCode().length() == 0) {
					// La rubrique mère est le sommet de l'arbre
					rubriqueMere = infosRubriqueTop;
				}
				// Rattache la rubrique à sa rubrique mère
				rubrique.rattacheA(rubriqueMere);
			}
		}
		LOGGER.info("Chargement de " + count + " rubriques OK");
		return listeRubriques;
	}

	@Override
	public Object getObjectToCache() throws Exception {
		return getCacheRubriques();
	}

	@Override
	public String getCacheName() {
		return KEY_CACHE;
	}

	@Override
	public Object getObjectKey() {
		return KEY_CACHE;
	}

	@Override
	public Object call() {
		final Object res = super.call();
		if (res != null) {
			return res;
		} else {
			return new HashMap<String, InfosRubriques>();
		}
	}
}
