/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.cache;

import java.util.HashMap;
import java.util.Locale;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.LangueUtil;
import com.kdecole.cache.aspect.StaticCacheFlush;
import com.kdecole.cache.aspect.StaticCacheGet;
import com.univ.objetspartages.om.ElementArboStructure;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.InfosStructure;
import com.univ.objetspartages.om.Media;
import com.univ.objetspartages.om.Metatag;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.objetspartages.om.Structure;
import com.univ.objetspartages.om.StructureModele;
import com.univ.utils.ContexteDao;

public class CacheStructureManager implements ICacheFicheManager {

	public static final String ID_BEAN = "cacheStructureManager";

	private static final Logger LOGGER = LoggerFactory.getLogger(CacheStructureManager.class);

	@StaticCacheGet(key = "CacheStructureManager.getListeStructures", contextual = false)
	public HashMap<String, ElementArboStructure> getListeStructures() {
		final HashMap<String, ElementArboStructure> listeStructure = new HashMap<>();
		try (ContexteDao ctx = new ContexteDao()) {
			// Charge en mémoire les listes des différents types de structures
			for (final String codeObjet : ReferentielObjets.getListeCodesObjet()) {
				FicheUniv ficheUniv = ReferentielObjets.instancierFiche(codeObjet);
				if (ficheUniv != null && ficheUniv instanceof StructureModele) {
					final StructureModele structure = (StructureModele) ficheUniv;
					structure.setCtx(ctx);
					final int count = structure.select("where ETAT_OBJET = '0003'");
					if (count > 0) {
						// chargement des métas en une seule requete
						HashMap<Long, String> hashMetaStructure = new HashMap<>();
						final Metatag meta = new Metatag();
						meta.init();
						meta.setCtx(ctx);
						if (meta.select(codeObjet, null) > 0) {
							while (meta.nextItem()) {
								hashMetaStructure.put(meta.getMetaIdFiche(), meta.getMetaInTree());
							}
						}
						while (structure.nextItem()) {
							// Instanciation de l'InfosStructure
							InfosStructure infosStructure = new InfosStructure(structure);
							if (structure.getIdBandeau() != 0) {
								infosStructure.setUrlBandeau(Media.getMedia(structure.getIdBandeau()).getUrlAbsolue());
							}
							if (hashMetaStructure.containsKey(structure.getIdFiche())) {
								infosStructure.setVisibleInFront("1".equals(hashMetaStructure.get(structure.getIdFiche())));
							}
							// Instanciation de l'ElementArboStructure si
							// nécessaire (gestion de la langue)
							ElementArboStructure eltArbo = listeStructure.get(structure.getCode());
							if (eltArbo == null) {
								// Ajout dans la bibliothèque de structures
								eltArbo = new ElementArboStructure(structure.getCode());
								listeStructure.put(structure.getCode(), eltArbo);
							}
							eltArbo.addInfosStructure(infosStructure);
						}
						LOGGER.info("Chargement de " + count + " structures de type " + ReferentielObjets.getLibelleObjet(codeObjet) + " OK");
					}
				}
			}
			// Initialise la structure de niveau supérieur
			ElementArboStructure eltArboTop = listeStructure.get(Structure.CODE_STRUCTURE_ROOT);
			if (eltArboTop == null) {
				eltArboTop = new ElementArboStructure(Structure.CODE_STRUCTURE_ROOT);
				listeStructure.put(eltArboTop.getCode(), eltArboTop);
			}
			for (int i = 0; i < LangueUtil.getNbLangues(); i++) {
				InfosStructure infosStructTop = new InfosStructure();
				infosStructTop.setCode(Structure.CODE_STRUCTURE_ROOT);
				infosStructTop.setLangue(String.valueOf(i));
				eltArboTop.addInfosStructure(infosStructTop);
			}
			// Effectue les rattachements de structure et calcule les niveaux
			for (ElementArboStructure elementArboStructure : listeStructure.values()) {
				for (InfosStructure infosStructure : elementArboStructure.getListeInfosStructure().values()) {
					rattacheStructure(listeStructure, infosStructure);
				}
			}
		} catch (final Exception e) {
			LOGGER.error("Impossible de charger le cache des structures", e);
		}
		return listeStructure;
	}

	/**
	 * Rattache une structure a sa structure mere, emule la structure mere si elle n'existe pas dans la langue demandée.
	 *
	 * @param structure
	 *            the structure
	 *
	 */
	private static void rattacheStructure(final HashMap<String, ElementArboStructure> listeStructure, InfosStructure structure) {
		if (!Structure.CODE_STRUCTURE_ROOT.equals(structure.getCode())) {
			InfosStructure structureMere = null;
			final ElementArboStructure elementArbo = listeStructure.get(structure.getCodeRattachement());
			if (elementArbo != null) {
				structureMere = elementArbo.getInfosStructure(structure.getLangue());
				if (structureMere == null) {
					// si la structure dans la langue demandée n'est pas disponible,
					// on prend la première structure existante ayant le même code
					if (!elementArbo.getListeInfosStructure().isEmpty()) {
						structure = elementArbo.getListeInfosStructure().values().iterator().next();
					}
				}
			}
			if (structureMere == null || structureMere.getCode().length() == 0) {
				// La structure mère est le sommet de l'arbre
				structureMere = listeStructure.get(Structure.CODE_STRUCTURE_ROOT).getInfosStructure(structure.getLangue());
			} else if (!structureMere.getLangue().equals(structure.getLangue())) {
				// La structure mère n'a pas été traduite dans la langue demandée
				final Locale locale = LangueUtil.getLocale(structure.getLangue());
				LOGGER.warn("La structure " + structure.getCodeRattachement() + " n'a pas été traduite en " + (locale != null ? locale.getDisplayLanguage() : "langue " + structure.getLangue()));
				structureMere = (InfosStructure) structureMere.clone();
				structureMere.setLangue(structure.getLangue());
				rattacheStructure(listeStructure, structureMere);
				listeStructure.get(structureMere.getCode()).addInfosStructure(structureMere);
			}
			// Rattache la structure à sa structure mère
			structure.rattacheA(structureMere);
		}
	}

	@Override
	@StaticCacheFlush(key = "CacheStructureManager.getListeStructures", contextual = false)
	public void flush() {}
}
