/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.dao.impl;

import java.sql.Connection;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Predicate;
import org.apache.commons.lang3.StringUtils;

import com.jsbsoft.jtf.core.Formateur;
import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.datasource.exceptions.AddToDataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.DataSourceException;
import com.jsbsoft.jtf.datasource.exceptions.UpdateToDataSourceException;
import com.univ.objetspartages.bean.ArticleBean;
import com.univ.objetspartages.dao.AbstractFicheDAO;
import com.univ.objetspartages.om.Article;
import com.univ.utils.Chaine;
import com.univ.utils.ContexteUtil;
import com.univ.utils.sql.RequeteSQL;
import com.univ.utils.sql.clause.ClauseJoin;
import com.univ.utils.sql.clause.ClauseLimit;
import com.univ.utils.sql.clause.ClauseOrderBy;
import com.univ.utils.sql.clause.ClauseOrderBy.SensDeTri;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ClauseJoinHelper;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.utils.sql.criterespecifique.LimitHelper;
import com.univ.utils.sql.criterespecifique.OrderByHelper;
import com.univ.utils.sql.criterespecifique.RequeteSQLHelper;

public class ArticleDAO extends AbstractFicheDAO<ArticleBean> {

	public ArticleDAO() {
		tableName = "ARTICLE";
	}
	
	@Override
	public ArticleBean add(final ArticleBean fiche) throws AddToDataSourceException {
		ResultSet rs = null;
		try (Connection connection = dataSource.getConnection();
			 PreparedStatement stmt = connection.prepareStatement("insert into ARTICLE (ID_ARTICLE , TITRE , SOUS_TITRE , DATE_ARTICLE , ID_VIGNETTE , CHAPEAU , CORPS , THEMATIQUE , ORDRE , CODE , CODE_RUBRIQUE , CODE_RATTACHEMENT , META_KEYWORDS , META_DESCRIPTION , TITRE_ENCADRE , CONTENU_ENCADRE , ENCADRE_RECHERCHE , ENCADRE_RECHERCHE_BIS , DATE_ALERTE , MESSAGE_ALERTE , DATE_CREATION , DATE_PROPOSITION , DATE_VALIDATION , DATE_MODIFICATION , CODE_REDACTEUR , CODE_VALIDATION , LANGUE , ETAT_OBJET , NB_HITS , DIFFUSION_PUBLIC_VISE , DIFFUSION_MODE_RESTRICTION , DIFFUSION_PUBLIC_VISE_RESTRICTION) values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)", Statement.RETURN_GENERATED_KEYS)) {
			stmt.setObject(1, fiche.getIdArticle(), Types.BIGINT);
			stmt.setObject(2, fiche.getTitre(), Types.VARCHAR);
			stmt.setObject(3, fiche.getSousTitre(), Types.LONGVARCHAR);
			stmt.setObject(4, fiche.getDateArticle(), Types.DATE);
			stmt.setObject(5, fiche.getIdVignette(), Types.BIGINT);
			stmt.setObject(6, fiche.getChapeau(), Types.LONGVARCHAR);
			stmt.setObject(7, fiche.getCorps(), Types.LONGVARCHAR);
			stmt.setObject(8, fiche.getThematique(), Types.VARCHAR);
			stmt.setObject(9, fiche.getOrdre(), Types.INTEGER);
			stmt.setObject(10, fiche.getCode(), Types.VARCHAR);
			stmt.setObject(11, fiche.getCodeRubrique(), Types.VARCHAR);
			stmt.setObject(12, fiche.getCodeRattachement(), Types.VARCHAR);
			stmt.setObject(13, fiche.getMetaKeywords(), Types.LONGVARCHAR);
			stmt.setObject(14, fiche.getMetaDescription(), Types.LONGVARCHAR);
			stmt.setObject(15, fiche.getTitreEncadre(), Types.VARCHAR);
			stmt.setObject(16, fiche.getContenuEncadre(), Types.LONGVARCHAR);
			stmt.setObject(17, fiche.getEncadreRecherche(), Types.VARCHAR);
			stmt.setObject(18, fiche.getEncadreRechercheBis(), Types.VARCHAR);
			stmt.setObject(19, fiche.getDateAlerte(), Types.DATE);
			stmt.setObject(20, fiche.getMessageAlerte(), Types.LONGVARCHAR);
			stmt.setObject(21, fiche.getDateCreation(), Types.TIMESTAMP);
			stmt.setObject(22, fiche.getDateProposition(), Types.TIMESTAMP);
			stmt.setObject(23, fiche.getDateValidation(), Types.TIMESTAMP);
			stmt.setObject(24, fiche.getDateModification(), Types.TIMESTAMP);
			stmt.setObject(25, fiche.getCodeRedacteur(), Types.VARCHAR);
			stmt.setObject(26, fiche.getCodeValidation(), Types.VARCHAR);
			stmt.setObject(27, fiche.getLangue(), Types.VARCHAR);
			stmt.setObject(28, fiche.getEtatObjet(), Types.VARCHAR);
			stmt.setObject(29, fiche.getNbHits(), Types.BIGINT);
			stmt.setObject(30, fiche.getDiffusionPublicVise(), Types.VARCHAR);
			stmt.setObject(31, fiche.getDiffusionModeRestriction(), Types.VARCHAR);
			stmt.setObject(32, fiche.getDiffusionPublicViseRestriction(), Types.VARCHAR);
			int rowsAffected = stmt.executeUpdate();
			if (rowsAffected != 1) {
				throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"%s\" : %d row(s) affected", fiche.toString(), tableName, rowsAffected));
			}
			rs = stmt.getGeneratedKeys();
			rs.next();
			fiche.setIdArticle(rs.getLong(1));
		} catch (SQLException e) {
			throw new AddToDataSourceException(String.format("Unable to add [%s] to table \"%s\"", fiche.toString(), tableName), e);
		} finally {
			if (rs != null) {
				try{
					rs.close();
				} catch (SQLException e) {
					LOG.error(String.format("An error occured closing Resultset during add() on table %s", tableName), e);
				}
			}
		}
		return fiche;
	}

	@Override
	public ArticleBean update(final ArticleBean fiche) throws UpdateToDataSourceException {
		try (Connection connection = dataSource.getConnection();
			 PreparedStatement stmt = connection.prepareStatement("update ARTICLE set ID_ARTICLE = ?, TITRE = ?, SOUS_TITRE = ?, DATE_ARTICLE = ?, ID_VIGNETTE = ?, CHAPEAU = ?, CORPS = ?, THEMATIQUE = ?, ORDRE = ?, CODE = ?, CODE_RUBRIQUE = ?, CODE_RATTACHEMENT = ?, META_KEYWORDS = ?, META_DESCRIPTION = ?, TITRE_ENCADRE = ?, CONTENU_ENCADRE = ?, ENCADRE_RECHERCHE = ?, ENCADRE_RECHERCHE_BIS = ?, DATE_ALERTE = ?, MESSAGE_ALERTE = ?, DATE_CREATION = ?, DATE_PROPOSITION = ?, DATE_VALIDATION = ?, DATE_MODIFICATION = ?, CODE_REDACTEUR = ?, CODE_VALIDATION = ?, LANGUE = ?, ETAT_OBJET = ?, NB_HITS = ?, DIFFUSION_PUBLIC_VISE = ?, DIFFUSION_MODE_RESTRICTION = ?, DIFFUSION_PUBLIC_VISE_RESTRICTION = ? where ID_ARTICLE = ?")) {
			// put parameters into statement
			stmt.setObject(1, fiche.getIdArticle(), Types.BIGINT);
			stmt.setObject(2, fiche.getTitre(), Types.VARCHAR);
			stmt.setObject(3, fiche.getSousTitre(), Types.LONGVARCHAR);
			stmt.setObject(4, fiche.getDateArticle(), Types.DATE);
			stmt.setObject(5, fiche.getIdVignette(), Types.BIGINT);
			stmt.setObject(6, fiche.getChapeau(), Types.LONGVARCHAR);
			stmt.setObject(7, fiche.getCorps(), Types.LONGVARCHAR);
			stmt.setObject(8, fiche.getThematique(), Types.VARCHAR);
			stmt.setObject(9, fiche.getOrdre(), Types.INTEGER);
			stmt.setObject(10, fiche.getCode(), Types.VARCHAR);
			stmt.setObject(11, fiche.getCodeRubrique(), Types.VARCHAR);
			stmt.setObject(12, fiche.getCodeRattachement(), Types.VARCHAR);
			stmt.setObject(13, fiche.getMetaKeywords(), Types.LONGVARCHAR);
			stmt.setObject(14, fiche.getMetaDescription(), Types.LONGVARCHAR);
			stmt.setObject(15, fiche.getTitreEncadre(), Types.VARCHAR);
			stmt.setObject(16, fiche.getContenuEncadre(), Types.LONGVARCHAR);
			stmt.setObject(17, fiche.getEncadreRecherche(), Types.VARCHAR);
			stmt.setObject(18, fiche.getEncadreRechercheBis(), Types.VARCHAR);
			stmt.setObject(19, fiche.getDateAlerte(), Types.DATE);
			stmt.setObject(20, fiche.getMessageAlerte(), Types.LONGVARCHAR);
			stmt.setObject(21, fiche.getDateCreation(), Types.TIMESTAMP);
			stmt.setObject(22, fiche.getDateProposition(), Types.TIMESTAMP);
			stmt.setObject(23, fiche.getDateValidation(), Types.TIMESTAMP);
			stmt.setObject(24, fiche.getDateModification(), Types.TIMESTAMP);
			stmt.setObject(25, fiche.getCodeRedacteur(), Types.VARCHAR);
			stmt.setObject(26, fiche.getCodeValidation(), Types.VARCHAR);
			stmt.setObject(27, fiche.getLangue(), Types.VARCHAR);
			stmt.setObject(28, fiche.getEtatObjet(), Types.VARCHAR);
			stmt.setObject(29, fiche.getNbHits(), Types.BIGINT);
			stmt.setObject(30, fiche.getDiffusionPublicVise(), Types.VARCHAR);
			stmt.setObject(31, fiche.getDiffusionModeRestriction(), Types.VARCHAR);
			stmt.setObject(32, fiche.getDiffusionPublicViseRestriction(), Types.VARCHAR);
			stmt.setObject(33, fiche.getIdArticle(), Types.BIGINT);
			stmt.executeUpdate();
		} catch (SQLException e) {
			throw new UpdateToDataSourceException(String.format("Unable to update [%s] from table \"%s\"", fiche.toString(), tableName), e);
		}
		return fiche;
	}

	@Override
	protected ArticleBean fill(ResultSet rs) throws DataSourceException {
		ArticleBean article = new ArticleBean();
		try {
			// get output from result set
			article.setIdArticle(new Long(rs.getLong("ID_ARTICLE")));
			article.setTitre(rs.getString("TITRE"));
			article.setSousTitre(rs.getString("SOUS_TITRE"));
			article.setDateArticle(rs.getDate("DATE_ARTICLE"));
			article.setIdVignette(new Long(rs.getLong("ID_VIGNETTE")));
			article.setChapeau(rs.getString("CHAPEAU"));
			article.setCorps(rs.getString("CORPS"));
			article.setThematique(rs.getString("THEMATIQUE"));
			article.setOrdre(new Integer(rs.getInt("ORDRE")));
			article.setCode(rs.getString("CODE"));
			article.setCodeRubrique(rs.getString("CODE_RUBRIQUE"));
			article.setCodeRattachement(rs.getString("CODE_RATTACHEMENT"));
			article.setMetaKeywords(rs.getString("META_KEYWORDS"));
			article.setMetaDescription(rs.getString("META_DESCRIPTION"));
			article.setTitreEncadre(rs.getString("TITRE_ENCADRE"));
			article.setContenuEncadre(rs.getString("CONTENU_ENCADRE"));
			article.setEncadreRecherche(rs.getString("ENCADRE_RECHERCHE"));
			article.setEncadreRechercheBis(rs.getString("ENCADRE_RECHERCHE_BIS"));
			article.setDateAlerte(rs.getDate("DATE_ALERTE"));
			article.setMessageAlerte(rs.getString("MESSAGE_ALERTE"));
			article.setDateCreation(new Date(rs.getTimestamp("DATE_CREATION").getTime()));
			article.setDateProposition(new Date(rs.getTimestamp("DATE_PROPOSITION").getTime()));
			article.setDateValidation(new Date(rs.getTimestamp("DATE_VALIDATION").getTime()));
			article.setDateModification(new Date(rs.getTimestamp("DATE_MODIFICATION").getTime()));
			article.setCodeRedacteur(rs.getString("CODE_REDACTEUR"));
			article.setCodeValidation(rs.getString("CODE_VALIDATION"));
			article.setLangue(rs.getString("LANGUE"));
			article.setEtatObjet(rs.getString("ETAT_OBJET"));
			article.setNbHits(new Long(rs.getLong("NB_HITS")));
			article.setDiffusionPublicVise(rs.getString("DIFFUSION_PUBLIC_VISE"));
			article.setDiffusionModeRestriction(rs.getString("DIFFUSION_MODE_RESTRICTION"));
			article.setDiffusionPublicViseRestriction(rs.getString("DIFFUSION_PUBLIC_VISE_RESTRICTION"));
		} catch (SQLException e) {
			throw new DataSourceException("An error occured trying to map resultset to object", e);
		}
		return article;
	}
	
	/**
	 * Selection d'une instance de l'objet Article a partir de l'ensemble des criteres.
	 * 
	 * @param titre
	 *            the titre
	 * @param dateDebut
	 *            the date debut
	 * @param dateFin
	 *            the date fin
	 * @param thematique
	 *            the thematique
	 * @param codeRubrique
	 *            the code rubrique
	 * @param codeRattachement
	 *            the code rattachement
	 * @param etatObjet
	 *            the etat objet
	 * @param langue
	 *            the langue
	 * @param codeRedacteur
	 *            the code redacteur
	 * @param nombre
	 *            the nombre
	 * @param ordre
	 *            the ordre
	 * @param ids
	 *            the ids
	 * @param code
	 *            the code
	 * 
	 * @return the int
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public List<ArticleBean> select(Article article, String titre, Date dateDebut, Date dateFin, String thematique, String codeRubrique, String codeRattachement, String etatObjet, String langue,
		String codeRedacteur, String nombre, String ordre, String ids, String code) throws Exception {
		ClauseWhere where = new ClauseWhere();
		if (StringUtils.isNotEmpty(titre)) {
			where.setPremiereCondition(ConditionHelper.rechercheMots("TITRE", titre));
		}
		if (Formateur.estSaisie(dateDebut)) {
			where.and(ConditionHelper.critereDateDebut("DATE_ARTICLE", dateDebut));
		}
		if (Formateur.estSaisie(dateFin)) {
			where.and(ConditionHelper.critereDateFin("DATE_ARTICLE", dateFin));
		}
		if (StringUtils.isNotEmpty(thematique) && !"0000".equals(thematique)) {
			where.and(ConditionHelper.likePourValeursMultiple("THEMATIQUE", thematique));
		}
		if (StringUtils.isNotEmpty(ids) && !"0000".equals(ids)) {
			where.and(ConditionHelper.in("ID_ARTICLE", Chaine.getVecteurPointsVirgules(ids)));
		}
		if (StringUtils.isNotEmpty(codeRattachement)) {
			where.and(ConditionHelper.getConditionStructure("CODE_RATTACHEMENT", codeRattachement));
		}
		if (StringUtils.isNotEmpty(langue) && !"0000".equals(langue)) {
			where.and(ConditionHelper.egalVarchar("T1.LANGUE", langue));
		}
		if (StringUtils.isNotEmpty(etatObjet) && !"0000".equals(etatObjet)) {
			where.and(ConditionHelper.egalVarchar("T1.ETAT_OBJET", etatObjet));
		}
		if (StringUtils.isNotEmpty(codeRedacteur)) {
			where.and(ConditionHelper.egalVarchar("CODE_REDACTEUR", codeRedacteur));
		}
		if (StringUtils.isNotEmpty(code)) {
			where.and(ConditionHelper.egalVarchar("T1.CODE", code));
		}
		OMContext ctx = ContexteUtil.getContexteUniv();
		final RequeteSQL requeteSelect = RequeteSQLHelper.getRequeteGenerique(where, ctx, article, codeRubrique);
		ClauseOrderBy orderBy = new ClauseOrderBy();
		if (StringUtils.isNotEmpty(ordre)) {
			orderBy = OrderByHelper.reconstruireClauseOrderBy(ordre);
		} else {
			orderBy.orderBy("T1.ORDRE", SensDeTri.ASC).orderBy("T1.TITRE", SensDeTri.ASC);
		}
		if (orderBy != null && StringUtils.contains(orderBy.formaterSQL(),"META.")) {
			boolean existJoinMeta = CollectionUtils.exists(requeteSelect.getJointures(), new Predicate() {
				@Override
				public boolean evaluate(Object o) {
					ClauseJoin jointure = (ClauseJoin)o;
					return StringUtils.contains(jointure.getNomTable() , "METATAG");
				}
			});
			if (!existJoinMeta) {
				requeteSelect.join(ClauseJoinHelper.creerJointureMetaTag(article));
			}
		}
		requeteSelect.orderBy(orderBy);
		ClauseLimit limite = LimitHelper.ajouterCriteresLimitesEtOptimisation(ctx, nombre);
		requeteSelect.limit(limite);
		return select(requeteSelect.formaterRequete());

	}
}
