/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.sql.Date;
import java.util.List;
import java.util.StringTokenizer;

import org.apache.commons.lang3.StringUtils;
import org.w3c.dom.Node;

import com.jsbsoft.jtf.core.Formateur;
import com.jsbsoft.jtf.core.LangueUtil;
import com.jsbsoft.jtf.database.OMContext;
import com.kportal.cms.objetspartages.annotation.FicheAnnotation;
import com.kportal.cms.objetspartages.annotation.GetterAnnotation;
import com.kportal.cms.objetspartages.annotation.SetterAnnotation;
import com.univ.objetspartages.bean.ArticleBean;
import com.univ.objetspartages.dao.impl.ArticleDAO;
import com.univ.utils.ContexteUtil;
import com.univ.utils.DateUtil;
import com.univ.utils.FicheUnivMgr;
import com.univ.utils.RequeteUtil;
import com.univ.utils.UnivWebFmt;
import com.univ.xml.NodeUtil;

/**
 * Classe representant un objet article.
 */
@FicheAnnotation()
public class Article extends AbstractFiche<ArticleBean> implements DiffusionSelective, FicheExportableXML {

	/**
	 * Renvoie le libelle a afficher (methode statique utilisee pour les jointures entre fiches).
	 *
	 * @param codes
	 *            the codes
	 * @param langue
	 *            the langue
	 *
	 * @return the libelle affichable
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String getLibelleAffichable(final String codes, String langue) throws Exception {
		String res = "";
		if (StringUtils.isEmpty(codes)) {
			return res;
		}
		final Article article = new Article();
		article.init();
		if (StringUtils.isEmpty(langue)) {
			langue = "0";
		}
		final StringTokenizer st = new StringTokenizer(codes, ";");
		while (st.hasMoreTokens()) {
			final String code = st.nextToken();
			// On cherche d'abord la version en ligne puis les autres versions
			int count = article.selectCodeLangueEtat(code, langue, "0003");
			if (count == 0) {
				count = article.selectCodeLangueEtat(code, langue, "");
			}
			if (count > 0) {
				if (res.length() > 1) {
					res += ";";
				}
				if (article.nextItem()) {
					res += article.getLibelleAffichable();
				}
			} else {
				res += "-";
			}
		}
		return res;
	}

	/**
	 * Renvoie la fiche demandee (methode statique utilisee pour les jointures entre fiches).
	 *
	 * @param code
	 *            the code
	 * @param langue
	 *            the langue
	 *
	 * @return the fiche article
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static Article getFicheArticle(final String code, String langue) throws Exception {
		Article fiche = null;
		if (code != null && code.length() > 0) {
			if (langue == null || langue.length() == 0) {
				langue = "0"; // francais par defaut
			}
			fiche = new Article();
			fiche.init();
			int count = fiche.selectCodeLangueEtat(code, langue, "0003");
			if (count == 0) // si pas de version en ligne
			{
				count = fiche.selectCodeLangueEtat(code, langue, "");
			}
			if (count == 0) // si pas de version dans la langue demandee
			{
				count = fiche.selectCodeLangueEtat(code, "", "0003");
			}
			if (count > 0) {
				fiche.nextItem();
			}
		}
		return fiche;
	}

	/**
	 * Initialise l'objet metier.
	 */
	@Override
	public void init() {
		super.init();
		setSousTitre("");
		setDateArticle(new Date(System.currentTimeMillis()));
		setIdVignette((long) 0);
		setChapeau("");
		setCorps("");
		setThematique("");
		setOrdre(1000);
	}

	public void setOrdre(final Integer ordre) {
		persistenceBean.setOrdre(ordre);
	}

	public Integer getOrdre() {
		return persistenceBean.getOrdre();
	}

	public void setThematique(final String thematique) {
		persistenceBean.setThematique(thematique);
	}

	public String getThematique() {
		return persistenceBean.getThematique();
	}

	@SetterAnnotation(isToolbox = true)
	public void setCorps(final String corps) {
		persistenceBean.setCorps(corps);
	}

	@GetterAnnotation(isToolbox = true)
	public String getCorps() {
		return persistenceBean.getCorps();
	}

	public void setChapeau(final String chapeau) {
		persistenceBean.setChapeau(chapeau);
	}

	public String getChapeau() {
		return persistenceBean.getChapeau();
	}

	public void setIdVignette(final Long idVignette) {
		persistenceBean.setIdVignette(idVignette);
	}

	public Long getIdVignette() {
		return persistenceBean.getIdVignette();
	}

	public void setDateArticle(final Date dateArticle) {
		persistenceBean.setDateArticle(dateArticle);
	}

	public Date getDateArticle() {
		return persistenceBean.getDateArticle();
	}

	public void setSousTitre(final String sousTitre) {
		persistenceBean.setSousTitre(sousTitre);
	}

	public String getSousTitre() {
		return persistenceBean.getSousTitre();
	}

	/**
	 * Renvoie le libelle a afficher.
	 *
	 * @return the libelle affichable
	 */
	@Override
	public String getLibelleAffichable() {
		return getTitre();
	}

	/**
	 * Renvoie le contenu formate en HTML pour l'attribut corps.
	 *
	 * @return the formated corps
	 *
	 * @throws Exception
	 *             the exception
	 */
	public String getFormatedCorps() throws Exception {
		return UnivWebFmt.formaterEnHTML(ContexteUtil.getContexteUniv(), getCorps());
	}

	/**
	 * Renvoie le libelle pour l'attribut thematique.
	 *
	 * @return the libelle thematique
	 *
	 */
	public String getLibelleThematique() {
		return Libelle.getLibelle("04", getThematique(), LangueUtil.getLocale(Integer.parseInt(getLangue())));
	}

	/**
	 * Renvoie le libelle pour l'attribut thematique.
	 *
	 * @return the libelle thematique
	 *
	 */
	public List<String> getListLibelleThematique() {
		return Libelle.getLibelleSousFormeDeListe("04", getThematique(), LangueUtil.getLocale(Integer.parseInt(getLangue())));
	}

	/**
	 * Renvoie la liste des fichiers joints de la fiche.
	 *
	 * @return the liste fichiers
	 *
	 * @throws Exception
	 *             the exception
	 */
	public List<Ressource> getListeFichiers() throws Exception {
		return Ressource.getListeFichier(this);
	}

	/**
	 * Renvoie la liste des fichiers indicés joints de la fiche.
	 *
	 * @return the liste fichiers indices
	 *
	 * @throws Exception
	 *             the exception
	 */
	public List<Ressource> getListeFichiersIndices() throws Exception {
		return Ressource.getListeFichierIndice(this);
	}

	/**
	 * Renvoie la rubrique de la fiche.
	 *
	 * @return the infos rubrique
	 *
	 */
	public InfosRubriques getInfosRubrique() {
		return Rubrique.renvoyerItemRubrique(getCodeRubrique());
	}

	/**
	 * Renvoie la structure de la fiche.
	 *
	 * @return the infos structure
	 *
	 * @throws Exception
	 *             the exception
	 */
	public InfosStructure getInfosStructure() throws Exception {
		return Structure.renvoyerItemStructure(getCodeRattachement(), getLangue());
	}

	/**
	 * Renvoie les metadonnees de la fiche.
	 *
	 * @return the meta
	 *
	 * @throws Exception
	 *             the exception
	 */
	public Metatag getMeta() throws Exception {
		return FicheUnivMgr.lireMeta(this);
	}

	/**
	 * Renvoie une chaine correspondant a la concatenation des champs texte de l'objet (utilise pour l'indexation Lucene).
	 *
	 * @return the full text string
	 *
	 */
	@Override
	public String getFullTextString() {
		return getSousTitre() + " " + getChapeau() + " " + getCorps() + " " + StringUtils.replace(getLibelleThematique(), ";", " ");
	}

	/**
	 * Traitement d'une requete sur l'objet.
	 *
	 * @param requete
	 *            the requete
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	public int traiterRequete(final String requete) throws Exception {
		// Recuperation des parametres de la requete
		final String code = RequeteUtil.renvoyerParametre(requete, "CODE");
		final String titre = RequeteUtil.renvoyerParametre(requete, "TITRE");
		String sDateDebut = RequeteUtil.renvoyerParametre(requete, "DATE_DEBUT");
		String sDateFin = RequeteUtil.renvoyerParametre(requete, "DATE_FIN");
		final String thematique = RequeteUtil.renvoyerParametre(requete, "THEMATIQUE");
		final String codeRubrique = RequeteUtil.renvoyerParametre(requete, "CODE_RUBRIQUE");
		final String codeRattachement = RequeteUtil.renvoyerParametre(requete, "CODE_RATTACHEMENT");
		final String langue = RequeteUtil.renvoyerParametre(requete, "LANGUE");
		final String selection = RequeteUtil.renvoyerParametre(requete, "SELECTION");
		final String tri = RequeteUtil.renvoyerParametre(requete, "TRI_DATE");
		final String nbJours = RequeteUtil.renvoyerParametre(requete, "JOUR");
		final String nombre = RequeteUtil.renvoyerParametre(requete, "NOMBRE");
		final String ids = RequeteUtil.renvoyerParametre(requete, "IDS");
		final String etatObjet = StringUtils.defaultIfBlank(RequeteUtil.renvoyerParametre(requete, "ETAT_OBJET"), EtatFiche.EN_LIGNE.toString());
		final String codeRedacteur = RequeteUtil.renvoyerParametre(requete, "CODE_REDACTEUR");
		// A SUPPRIMER dans une future version
		// (présent pour des raisons de compatibilité avec les contenus existants)
		if (sDateDebut.length() == 0) {
			sDateDebut = RequeteUtil.renvoyerParametre(requete, "DATE_ARTICLE_DEBUT");
		}
		if (sDateFin.length() == 0) {
			sDateFin = RequeteUtil.renvoyerParametre(requete, "DATE_ARTICLE_FIN");
		}
		Date dateDebut = null;
		Date dateFin = null;
		String orderDate = "";
		String orderTri = "";
		if (selection.length() > 0) {
			final Date today = new Date(System.currentTimeMillis());
			orderDate = "T1.DATE_ARTICLE";
			// Derniers articles parus
			if ("0003".equals(selection)) {
				dateFin = today; // aujourd'hui
				orderTri = " DESC";
			}
			// Articles du jour
			else if ("0004".equals(selection)) {
				// date début dans la journée ou date de fin dans la journée
				dateDebut = today; // aujourd'hui
				dateFin = today; // aujourd'hui
			}
			// Articles de la semaine
			else if ("0005".equals(selection)) {
				// date début dans la semaine ou date de fin dans la semaine
				dateDebut = DateUtil.getFirstDayOfWeek(today);
				dateFin = DateUtil.getLastDayOfWeek(today);
			}
			// Articles du mois
			else if ("0006".equals(selection)) {
				// date début dans le mois ou date de fin dans le mois
				dateDebut = DateUtil.getFirstDayOfMonth(today);
				dateFin = DateUtil.getLastDayOfMonth(today);
			}
			// Dernières articles publiés
			else if ("0007".equals(selection)) {
				orderDate = "META.META_DATE_MISE_EN_LIGNE";
				orderTri = "DESC";
			}
		} else if (StringUtils.isNotBlank(nbJours)) {
			dateDebut = DateUtil.addDays(new Date(System.currentTimeMillis()), -Integer.parseInt(nbJours)); // aujourd'hui - nbJours
			dateFin = new Date(System.currentTimeMillis());
			orderDate = "T1.DATE_ARTICLE";
			orderTri = "DESC";
		} else if (StringUtils.isNotBlank(sDateDebut) || StringUtils.isNotBlank(sDateFin)) {
			dateDebut = DateUtil.parseDate(sDateDebut);
			dateFin = DateUtil.parseDate(sDateFin);
			orderDate = "T1.DATE_ARTICLE";
			orderTri = "DESC";
		}
		if ("DATE_DESC".equals(tri)) {
			orderTri = "DESC";
		}
		if ("DATE_ASC".equals(tri)) {
			orderTri = "ASC";
		}
		String order = "T1.ORDRE, T1.TITRE";
		if (orderDate.length() > 0) {
			order = orderDate + " " + orderTri + ", " + order;
		}
		return select(titre, dateDebut, dateFin, thematique, codeRubrique, codeRattachement, etatObjet, langue, codeRedacteur, nombre, order, ids, code);
	}

	/**
	 * Renvoie la liste des references a cette fiche.
	 *
	 * @return the references
	 */
	@Override
	public String getReferences() {
		String res = FicheUnivMgr.getReferenceParPhoto(getIdVignette()) + FicheUnivMgr.getReferenceParTexte(getCorps()) + FicheUnivMgr.getReferenceParJointure("structure",
			getCodeRattachement());
		try {
			res += FicheUnivMgr.getReferenceParListePhoto(getListeFichiers()) + FicheUnivMgr.getReferenceParListePhoto(getListeFichiersIndices());
		} catch (final Exception ignored) {}
		return res;
	}

	/**
	 * Selection d'une instance de l'objet Article a partir de l'ensemble des criteres.
	 *
	 * @param titre
	 *            the titre
	 * @param dateDebut
	 *            the date debut
	 * @param dateFin
	 *            the date fin
	 * @param thematique
	 *            the thematique
	 * @param codeRubrique
	 *            the code rubrique
	 * @param codeRattachement
	 *            the code rattachement
	 * @param etatObjet
	 *            the etat objet
	 * @param langue
	 *            the langue
	 * @param codeRedacteur
	 *            the code redacteur
	 * @param nombre
	 *            the nombre
	 * @param ordre
	 *            the ordre
	 * @param ids
	 *            the ids
	 * @param code
	 *            the code
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 */
	public int select(final String titre, final Date dateDebut, final Date dateFin, final String thematique, final String codeRubrique, final String codeRattachement,
		final String etatObjet, final String langue, final String codeRedacteur, final String nombre, final String ordre, final String ids, final String code) throws Exception {
		currentSelect = ((ArticleDAO) commonDao).select(this, titre, dateDebut, dateFin, thematique, codeRubrique, codeRattachement, etatObjet, langue, codeRedacteur, nombre,
			ordre, ids, code);
		currentSelectIt = currentSelect.iterator();
		return currentSelect.size();
	}

	/**
	 * Récupération du libellé à afficher (spécifique au back-office).
	 *
	 * @return the libelle back office
	 *
	 */
	public String getLibelleBackOffice() {
		return getLibelleAffichable() + " [" + Rubrique.getIntitule(getCodeRubrique()) + "]";
	}

	/**
	 * renvoie le corps de l'article (utilise dans SaisieCommentaire) (non-Javadoc).
	 *
	 * @return the contenu
	 *
	 */
	public String getContenu() {
		return getCorps();
	}

	/* (non-Javadoc)
	 * @see com.univ.objetspartages.om.FicheExportableXML#exportNodeFiche(org.w3c.dom.Node, java.lang.String)
	 */
	@Override
	public Node exportNodeFiche(final Node liste, final String formatExport) throws Exception {
		final OMContext ctx = ContexteUtil.getContexteUniv();
		final org.w3c.dom.Document document = liste.getOwnerDocument();
		final Node fiche = document.createElement("ARTICLE");
		NodeUtil.addNode(fiche, "CODE", getCode());
		NodeUtil.addNode(fiche, "LANGUE", getLangue());
		NodeUtil.addNode(fiche, "RATTACHEMENT", Structure.getLibelleRattachement(ctx, getCodeRattachement(), getLangue(), true));
		NodeUtil.addNode(fiche, "LIBELLE", getLibelleAffichable());
		NodeUtil.addNode(fiche, "TITRE", getTitre());
		NodeUtil.addNode(fiche, "SOUS_TITRE", getTitre());
		NodeUtil.addNode(fiche, "DATE_ARTICLE", Formateur.formaterDate(getDateArticle(), ""));
		NodeUtil.addNode(fiche, "CHAPEAU", getChapeau());
		NodeUtil.addNode(fiche, "CORPS", getCorps());
		NodeUtil.addNode(fiche, "URL_VIGNETTE", Media.getMedia(getIdVignette()).getUrlAbsolue());
		NodeUtil.addNode(fiche, "CODE_RUBRIQUE", getCodeRubrique());
		NodeUtil.addNode(fiche, "RUBRIQUE", Rubrique.renvoyerItemRubrique(getCodeRubrique()).getIntitule());
		return fiche;
	}

	/**
	 * Gets the diffusion mode restriction.
	 *
	 * @return the diffusionModeRestriction
	 */
	@Override
	public String getDiffusionModeRestriction() {
		return persistenceBean.getDiffusionModeRestriction();
	}

	/**
	 * Sets the diffusion mode restriction.
	 *
	 * @param diffusionModeRestriction
	 *            the diffusionModeRestriction to set
	 */
	@Override
	public void setDiffusionModeRestriction(final String diffusionModeRestriction) {
		persistenceBean.setDiffusionModeRestriction(diffusionModeRestriction);
	}

	/**
	 * Gets the diffusion public vise.
	 *
	 * @return the diffusionPublicVise
	 */
	@Override
	public String getDiffusionPublicVise() {
		return persistenceBean.getDiffusionPublicVise();
	}

	/**
	 * Sets the diffusion public vise.
	 *
	 * @param diffusionPublicVise
	 *            the diffusionPublicVise to set
	 */
	@Override
	public void setDiffusionPublicVise(final String diffusionPublicVise) {
		persistenceBean.setDiffusionPublicVise(diffusionPublicVise);
	}

	/**
	 * Gets the diffusion public vise restriction.
	 *
	 * @return the diffusionPublicViseRestriction
	 */
	@Override
	public String getDiffusionPublicViseRestriction() {
		return persistenceBean.getDiffusionPublicViseRestriction();
	}

	/**
	 * Sets the diffusion public vise restriction.
	 *
	 * @param diffusionPublicViseRestriction
	 *            the diffusionPublicViseRestriction to set
	 */
	@Override
	public void setDiffusionPublicViseRestriction(final String diffusionPublicViseRestriction) {
		persistenceBean.setDiffusionPublicViseRestriction(diffusionPublicViseRestriction);
	}
}
