/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import com.univ.utils.ContexteDao;
import org.apache.commons.lang3.StringUtils;

import com.univ.objetspartages.sgbd.EncadreDB;
import com.univ.utils.ContexteUniv;
import com.univ.utils.sql.RequeteSQL;
import com.univ.utils.sql.clause.ClauseOrderBy;
import com.univ.utils.sql.clause.ClauseOrderBy.SensDeTri;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.condition.ConditionList;
import com.univ.utils.sql.criterespecifique.ConditionHelper;

/**
 * The Class Encadre.
 */
public class Encadre extends EncadreDB implements Cloneable {

	private static final long serialVersionUID = 1672741661121689614L;

	/**
	 * Instantiates a new encadre.
	 */
	public Encadre() {
		super();
	}

	/**
	 * implementation de la methode clone() (utilisée dans le module catalogue)
	 */
	@Override
	public Encadre clone() throws CloneNotSupportedException {
		return (Encadre) super.clone();
	}

	/**
	 * Inits the.
	 */
	public void init() {
		setIdEncadre(new Long(System.currentTimeMillis()));
		setIntitule("");
		setActif("1");
		setLangue("0");
		setPoids(new Integer(0));
		setContenu("");
		setObjets("");
		// JSS 20050713 : ajout code pour e-sup
		setCode("");
		setCodeRattachement("");
		setCodeRubrique("");
	}

	/* Renvoie la liste des objets sous forme de vecteur */
	/**
	 * Gets the vecteur objets.
	 *
	 * @return the vecteur objets
	 *
	 * @throws Exception
	 *             the exception
	 */
	public List<String> getVecteurObjets() {
		return Arrays.asList(StringUtils.split(getObjets(), ";"));
	}

	/* Valorise la liste des objets sous forme de vecteur */
	/**
	 * Sets the vecteur objets.
	 *
	 * @param v
	 *            the new vecteur objets
	 *
	 * @throws Exception
	 *             the exception
	 */
	public void setVecteurObjets(final Vector<String> v) {
		setObjets(StringUtils.defaultString(StringUtils.join(v, ";")));
	}

	/**
	 * Gets the liste encadres recherche.
	 *
	 * @return the liste encadres recherche
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static Hashtable<String, String> getListeEncadresRecherche() throws Exception {
		final Hashtable<String, String> listeEncadresRecherche = new Hashtable<>();
		for (final String codeObjet : ReferentielObjets.getListeCodesObjet()) {
			if (ReferentielObjets.gereEncadreRechercheEmbarquable(codeObjet)) {
				listeEncadresRecherche.put(codeObjet, ReferentielObjets.getLibelleObjet(codeObjet));
			}
		}
		return listeEncadresRecherche;
	}

	/**
	 * Récupération d'une liste d'encadrés triés par poids pour un type d'objet.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param ficheUniv
	 *            the fiche univ
	 * @param _langue
	 *            the _langue
	 *
	 * @return the liste objets encadres
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static ArrayList<Encadre> getListeObjetsEncadres(final ContexteUniv _ctx, final FicheUniv ficheUniv, final String _langue) throws Exception {
		final ArrayList<Encadre> encadres = new ArrayList<>();
		String codeObjet = "";
		if (ficheUniv != null) {
			codeObjet = ReferentielObjets.getCodeObjetParClasse(ficheUniv.getClass().getName());
			// JSS 20030114 : on remonte le test
			if ((codeObjet == null) || (codeObjet.length() == 0)) {
				return encadres;
			}
		}
		final Encadre encadre = new Encadre();
		encadre.init();
		encadre.setCtx(_ctx);
		final String codeRubrique = _ctx.getCodeRubriquePageCourante();
		String codeStructure = "";
		if (ficheUniv != null) {
			codeStructure = ficheUniv.getCodeRattachement();
		}
		final RequeteSQL requeteEncadre = construireRequeteObjetsEncadres(_langue, codeRubrique, codeStructure, codeObjet);
		final int count = encadre.select(requeteEncadre.formaterRequete());
		if (count > 0) {
			while (encadre.nextItem()) {
				encadres.add(encadre.clone());
			}
		}
		return encadres;
	}

	/**
	 * Récupération d'une liste d'encadrés triés par poids pour un type d'objet.
	 *
	 * @param codeRubrique
	 *            le code rubrique dont on souhaite récupérer les encadrés
	 * @param ficheUniv
	 *            la fiche dont on souhaite récupérer les encadrés
	 * @param langue
	 *            la langue des encadrés
	 *
	 * @return the liste objets encadres
	 *
	 * @throws Exception
	 *             lors de la récupération des encadrés en base
	 */
	public static ArrayList<Encadre> getListeObjetsEncadres(final String codeRubrique, final FicheUniv ficheUniv, final String langue) throws Exception {
		final ArrayList<Encadre> encadres = new ArrayList<>();
		String codeObjet = "";
		if (ficheUniv != null) {
			codeObjet = ReferentielObjets.getCodeObjetParClasse(ficheUniv.getClass().getName());
			// JSS 20030114 : on remonte le test
			if (StringUtils.isEmpty(codeObjet)) {
				return encadres;
			}
		}
		final Encadre encadre = new Encadre();
		encadre.init();
		try (ContexteDao ctxDao = new ContexteDao()) {
			encadre.setCtx(ctxDao);
			String codeStructure = "";
			if (ficheUniv != null) {
				codeStructure = ficheUniv.getCodeRattachement();
			}
			final RequeteSQL requeteEncadre = construireRequeteObjetsEncadres(langue, codeRubrique, codeStructure, codeObjet);
			final int count = encadre.select(requeteEncadre.formaterRequete());
			if (count > 0) {
				while (encadre.nextItem()) {
					encadres.add(encadre.clone());
				}
			}
		}
		return encadres;
	}

	/**
	 * Construit la requete pour retourner les objets encadré lié à la rubrique ou la structure ou l'objet dans la langue courante.
	 *
	 * @param langue
	 * @param codeRubrique
	 * @param codeStructure
	 * @param codeObjet
	 * @return
	 */
	private static RequeteSQL construireRequeteObjetsEncadres(String langue, final String codeRubrique, final String codeStructure, final String codeObjet) {
		final RequeteSQL requete = new RequeteSQL();
		final ClauseWhere where = new ClauseWhere();
		final ClauseOrderBy orderBy = new ClauseOrderBy("POIDS", SensDeTri.ASC);
		where.setPremiereCondition(ConditionHelper.egalVarchar("ACTIF", "1"));
		if (StringUtils.isEmpty(langue)) {
			langue = "0";
		}
		where.and(ConditionHelper.egalVarchar("LANGUE", langue));
		where.and(conditionRubriqueStructureEtObjet(codeRubrique, codeStructure, codeObjet));
		requete.where(where).orderBy(orderBy);
		return requete;
	}

	private static ConditionList conditionRubriqueStructureEtObjet(String codeRubrique, final String codeStructure, final String codeObjet) {
		final ConditionList listeConditionRubStructEtObjet = new ConditionList(ConditionHelper.egalVarchar("OBJETS", ""));
		if (StringUtils.isNotEmpty(codeRubrique)) {
			listeConditionRubStructEtObjet.or(ConditionHelper.likePourValeursMultiple("OBJETS", "RUB_" + codeRubrique));
			int niveau = Rubrique.renvoyerItemRubrique(codeRubrique).getNiveau();
			while (niveau > 1) {
				codeRubrique = Rubrique.renvoyerItemRubrique(codeRubrique).getCodeRubriqueMere();
				listeConditionRubStructEtObjet.or(ConditionHelper.likePourValeursMultiple("OBJETS", "RUB_" + codeRubrique));
				niveau--;
			}
		}
		listeConditionRubStructEtObjet.or(conditionStructure(codeStructure, codeObjet));
		listeConditionRubStructEtObjet.or(ConditionHelper.likePourValeursMultiple("OBJETS", codeObjet + "/"));
		return listeConditionRubStructEtObjet;
	}

	private static ConditionList conditionStructure(String codeStructure, final String codeObjet) {
		final ConditionList conditionStructure = new ConditionList();
		if (StringUtils.isNotEmpty(codeStructure)) {
			conditionStructure.setPremiereCondtion(ConditionHelper.likePourValeursMultiple("OBJETS", "/" + codeStructure));
			conditionStructure.or(ConditionHelper.likePourValeursMultiple("OBJETS", codeObjet + "/" + codeStructure));
			int niveau = Structure.renvoyerItemStructure(codeStructure).getNiveau();
			while (niveau > 1) {
				codeStructure = Structure.renvoyerItemStructure(codeStructure).getCodeRattachement();
				conditionStructure.or(ConditionHelper.likePourValeursMultiple("OBJETS", "/" + codeStructure));
				conditionStructure.or(ConditionHelper.likePourValeursMultiple("OBJETS", codeObjet + "/" + codeStructure));
				niveau--;
			}
		}
		return conditionStructure;
	}

	/**
	 * Récupération d'une liste d'encadrés triés par poids pour un type d'objet.
	 *
	 * @param ctx
	 *            Le contexte juste pour le code de rubrique de la page courante...
	 * @param ficheUniv
	 *            la fiche dont on souhaite récupérer les encadrés
	 * @param langue
	 *            la langue po
	 *
	 * @return the liste encadres
	 *
	 * @throws Exception
	 *             the exception
	 * @deprecated on se sert uniquement du contexte pour récupérer le code de rubrique de la page courante... utilisez {@link Encadre#getListeEncadres(String, FicheUniv, String)}
	 */
	@Deprecated
	public static ArrayList<String> getListeEncadres(final ContexteUniv ctx, final FicheUniv ficheUniv, final String langue) throws Exception {
		final ArrayList<String> a = new ArrayList<>();
		for (Encadre encadre : getListeObjetsEncadres(ctx.getCodeRubriquePageCourante(), ficheUniv, langue)) {
			a.add(encadre.getContenu());
		}
		return a;
	}

	/**
	 * Récupération d'une liste d'encadrés triés par poids pour un type d'objet.
	 *
	 * @param codeRubrique
	 *            Le code de la rubrique dont on souhaite récupérer les encadrés
	 * @param ficheUniv
	 *            la fiche dont on souhaite récupérer les encadrés
	 * @param langue
	 *            la langue po
	 *
	 * @return the liste encadres
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static ArrayList<String> getListeEncadres(final String codeRubrique, final FicheUniv ficheUniv, final String langue) throws Exception {
		final ArrayList<String> a = new ArrayList<>();
		for (Encadre encadre : getListeObjetsEncadres(codeRubrique, ficheUniv, langue)) {
			a.add(encadre.getContenu());
		}
		return a;
	}
}
