/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.InfoBean;
import com.jsbsoft.jtf.core.LangueUtil;
import com.univ.utils.ContexteDao;
import com.univ.utils.ContexteUniv;
import com.univ.utils.ContexteUtil;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;

/**
 * Cette classe permet ce construire des groupe DSI dynamiquement selon les structures. C'est à dire que dans ces groupe dynamique de structure nous allons préciser les codes
 * structures afin de fabriquer un groupe. Tous les utilisateur rattachés à ces structures feront parti du groupe.<br/>
 * De plus ce groupe est synchronisable cela signifie que qu'il sera possible de synchroniser toutes les structures afin de former des groupes.
 * 
 * @author Pierre Cosson
 */
public class GroupeDynamiqueDeStructure extends RequeteGroupeDynamiqueSynchronisable {

	/** Nom de la requete, on peut retrouver ce nom dans le JTF. */
	private static final String NOM_REQUETE = "req_structure";

	private static final Logger LOG = LoggerFactory.getLogger(GroupeDynamiqueDeStructure.class);

	public GroupeDynamiqueDeStructure() {
		super();
		setNomRequete(NOM_REQUETE);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * com.univ.objetspartages.om.RequeteGroupeDynamique#estGereParReqDyn(java
	 * .lang.String)
	 */
	@Override
	protected boolean estGereParReqDyn(final String requeteGroupe) {
		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * com.univ.objetspartages.om.RequeteGroupeDynamique#getGroupesUtilisateur
	 * (java.lang.String)
	 */
	@Override
	protected Vector<String> getGroupesUtilisateur(final String codeUtilisateur) throws Exception {
		LOG.debug("Début de requete dynamique de structure : getGroupesUtilisateur (" + codeUtilisateur + ")");
		final Vector<String> vGroupes = new Vector<>();
		try (ContexteDao ctx = new ContexteDao()) {
			final Utilisateur user = Utilisateur.getUtilisateur(codeUtilisateur);
			if (!"".equals(user.getCodeRattachement())) {
				//l'utilisateur a bien un code rattachement structure
				final Groupedsi groupeDsi = new Groupedsi();
				groupeDsi.setCtx(ctx);
				final HashSet<String> hsGroupes = new HashSet<>();
				final String codeRattachement = user.getCodeRattachement();
				final ClauseWhere whereRequeteLDAP = new ClauseWhere();
				whereRequeteLDAP.setPremiereCondition(ConditionHelper.likePourValeursMultiple("REQUETE_LDAP", codeRattachement));
				groupeDsi.select(whereRequeteLDAP.formaterSQL());
				while (groupeDsi.nextItem()) {
					hsGroupes.add(groupeDsi.getCode());
				}
				vGroupes.addAll(hsGroupes);
			}
		}
		return vGroupes;
	}

	/* (non-Javadoc)
	 * @see com.univ.objetspartages.om.RequeteGroupeDynamique#getVecteurUtilisateurs(java.lang.String)
	 */
	@Override
	protected Vector<String> getVecteurUtilisateurs(final String codeGroupe) throws Exception {
		LOG.debug("Debut de requete dynamique : getVecteurUtilisateur (" + codeGroupe + ")");
		final Vector<String> vUtilisateurs = new Vector<>();
		try (ContexteDao ctx = new ContexteDao()) {
			final Groupedsi groupeDsi = new Groupedsi();
			groupeDsi.setCtx(ctx);
			/* On récupère le groupedsi pour lequel on veut calculer les utilisateurs */
			if (groupeDsi.select(codeGroupe, "", "", "", "") > 0) {
				groupeDsi.nextItem();
				final HashSet<String> hsUtilisateurs = new HashSet<String>();
				/* dans ce groupe, on récupère la liste des structures
				 * sélectionnés via le contenu du champ REQUETE_LDAP */
				final StringTokenizer stCodesStructures = new StringTokenizer(groupeDsi.getRequeteLdap(), ";");
				final Utilisateur user = new Utilisateur();
				user.init();
				user.setCtx(ctx);
				//pour chaque structure selectionnée, on récupère la liste des membres directement rattachés à cette structure 
				while (stCodesStructures.hasMoreTokens()) {
					if (user.select("", "", "", "", "", "", stCodesStructures.nextToken(), "") > 0) {
						while (user.nextItem()) {
							hsUtilisateurs.add(user.getCode());
						}
					}
				}
				vUtilisateurs.addAll(hsUtilisateurs);
			}
		}
		LOG.debug("Fin de requete dynamique : getVecteurUtilisateur (" + codeGroupe + ")");
		return vUtilisateurs;
	}

	/* (non-Javadoc)
	 * @see com.univ.objetspartages.om.RequeteGroupeDynamiqueSynchronisable#getListeCodeObjetASynchroniser()
	 */
	@Override
	protected Map<String, SynchronisationGroupeDynamiqueData> getDataToSynchronize() throws Exception {
		SynchronisationGroupeDynamiqueData syncData;
		//reinitialisation du cache des structures
		Structure.forcerRechargement();
		final InfosStructure infosStructureRoot = Structure.getTopLevelStructure(getCurrentLanguage());
		//Collection<InfosStructure>
		final Collection<InfosStructure> listeStructure = infosStructureRoot.getListeSousStructuresTousNiveaux();
		if (listeStructure != null) {
			final Map<String, SynchronisationGroupeDynamiqueData> returnMap = new HashMap<>();
			final Iterator<InfosStructure> codesStructures = listeStructure.iterator();
			InfosStructure structure;
			String codeStructure;
			while (codesStructures.hasNext()) {
				structure = codesStructures.next();
				codeStructure = structure.getCode();
				if (codeStructure != null) {
					syncData = new SynchronisationGroupeDynamiqueData(codeStructure, structure.getLibelleAffichable());
					syncData.setCodeDuParent(structure.getCodeRattachement());
					returnMap.put(codeStructure, syncData);
				}
			}
			return returnMap;
		} else {
			return null;
		}
	}

	/* (non-Javadoc)
	 * @see com.univ.objetspartages.om.RequeteGroupeDynamique#preparerPRINCIPAL(com.jsbsoft.jtf.core.InfoBean, com.univ.objetspartages.om.Groupedsi)
	 */
	@Override
	public void preparerPRINCIPAL(final InfoBean infoBean, final Groupedsi groupe) throws Exception {
		if (groupe.getRequeteGroupe().length() > 0 && groupe.getRequeteGroupe().equals(this.getNomRequete())) {
			infoBean.set("REQUETE_LDAP_" + groupe.getRequeteGroupe(), groupe.getRequeteLdap());
			// Groupe de structure 
			infoBean.set("REQUETE_LDAP_" + this.getNomRequete(), groupe.getRequeteLdap());
			String libellesStructure = "";
			final StringTokenizer stCodesStructures = new StringTokenizer(groupe.getRequeteLdap(), ";");
			String langue = getCurrentLanguage();
			while (stCodesStructures.hasMoreTokens()) {
				libellesStructure += Structure.getLibelleAffichable(stCodesStructures.nextToken(), langue) + ";";
			}
			infoBean.set("LIBELLE_REQUETE_LDAP_" + this.getNomRequete(), libellesStructure);
			infoBean.set("INFOBULLE_REQUETE_LDAP_" + this.getNomRequete(), libellesStructure);
		}
	}

	private String getCurrentLanguage() {
		ContexteUniv ctx = ContexteUtil.getContexteUniv();
		Locale currentLocale = LangueUtil.getDefaultLocale();
		if (ctx != null) {
			currentLocale = ctx.getLocale();
		}
		return LangueUtil.getLangueLocale(currentLocale);
	}
}
