/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.util.Collection;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

import org.apache.commons.lang3.StringUtils;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.database.OMContext;
import com.kportal.core.config.MessageHelper;
import com.univ.objetspartages.cache.CacheGroupeDsiManager;
import com.univ.objetspartages.sgbd.GroupedsiDB;
import com.univ.utils.sql.RequeteSQL;
import com.univ.utils.sql.clause.ClauseOrderBy;
import com.univ.utils.sql.clause.ClauseOrderBy.SensDeTri;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.xhtml.JavascriptTreeGroupedsi;

/**
 * The Class Groupedsi.
 */
public class Groupedsi extends GroupedsiDB implements Cloneable {

	/** The Constant CODE_GROUPE_ROOT. */
	public final static String CODE_GROUPE_ROOT = "00";

	/**
	 * Inits the.
	 */
	public void init() {
		setIdGroupedsi((long) 0);
		setCode(String.valueOf(System.currentTimeMillis()));
		setLibelle("");
		setType("");
		setCodePageTete("");
		setCodeStructure("");
		setRoles("");
		setCodeGroupePere("");
		setSourceImport("");
		setRequeteGroupe("");
		setRequeteLdap("");
		setGestionCache("0");
		final Long tsInit = (long) 0;
		setDelaiExpirationCache(tsInit);
		setDerniereMajCache(tsInit);
		setSelectionnable("1");
	}

	@Deprecated
	public static String getIntitule(final OMContext _ctx, final String _code) {
		return getIntitule(_code);
	}

	/**
	 * Récupération de l'intitulé : gestion de plusieurs codes.
	 *
	 * @param _code
	 *            the _code
	 * 
	 * @return the intitule
	 *
	 */
	public static String getIntitule(final String _code) {
		String res = StringUtils.EMPTY;
		if ((_code == null) || (_code.length() == 0)) {
			return res;
		}
		final StringTokenizer st = new StringTokenizer(_code, ";");
		while (st.hasMoreTokens()) {
			final String code = st.nextToken();
			if (res.length() > 0) {
				res += ";";
			}
			String intitule = renvoyerItemGroupeDsi(code).getIntitule();
			if (intitule.length() == 0) {
				intitule = MessageHelper.getCoreMessage("BO_GROUPE_INEXISTANT");
			}
			res += intitule;
		}
		return res;
	}

	private static Map<String, InfosGroupeDsi> getListeGroupesDsi() {
		final CacheGroupeDsiManager cache = (CacheGroupeDsiManager) ApplicationContextManager.getCoreContextBean(CacheGroupeDsiManager.ID_BEAN);
		return cache.getListeGroupesDsi();
	}

	public static Map<String, InfosRequeteGroupe> getListeRequetesGroupes() {
		final CacheGroupeDsiManager cache = (CacheGroupeDsiManager) ApplicationContextManager.getCoreContextBean(CacheGroupeDsiManager.ID_BEAN);
		return cache.getListeRequetesGroupes();
	}

	/**
	 * Renvoie la liste des infos des groupes pour un type.
	 * 
	 * @param _type
	 *            the _type
	 * 
	 * @return the vector
	 *
	 */
	public static Vector<InfosGroupeDsi> renvoyerCodesGroupeDsiParType(final String _type) {
		final Vector<InfosGroupeDsi> res = new Vector<>();
		for (final InfosGroupeDsi info : getListeGroupesDsi().values()) {
			if (info.getType().equalsIgnoreCase(_type)) {
				res.add(info);
			}
		}
		return res;
	}

	/**
	 * Récupération d'une rubrique stockée en mémoire.
	 * 
	 * @param code
	 *            the code
	 * 
	 * @return the infos groupe dsi
	 *
	 */
	public static InfosGroupeDsi renvoyerItemGroupeDsi(final String code) {
		InfosGroupeDsi res = getListeGroupesDsi().get(code);
		/* groupe vide pour éviter les plantages */
		if (res == null) {
			res = new InfosGroupeDsi();
		}
		return res;
	}

	/**
	 * Renvoie la liste des {groupes,périmètres} associé à un role et pour un perimetre.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _role
	 *            the _role
	 * @param codesStructures
	 *            the codes structures
	 * @param _codeRubrique
	 *            the _code rubrique
	 * @param _publicsVises
	 *            the _publics vises
	 * @param _codeEspaceCollaboratif
	 *            the _code espace collaboratif
	 *
	 * @return the hashtable
	 *
	 * @throws Exception
	 *             the exception
	 * @deprecated Méthode plus utilisée
	 */
	@Deprecated
	public static Hashtable<String, Vector<Perimetre>> renvoyerGroupesEtPerimetres(final OMContext _ctx, final String _role, final List<String> codesStructures,
																				   final String _codeRubrique, final String _publicsVises, final String _codeEspaceCollaboratif) throws Exception {
		final Hashtable<String, Vector<Perimetre>> h = new Hashtable<>();
		for (final InfosGroupeDsi info : getListeGroupesDsi().values()) {
			final Vector<Perimetre> v = Role.renvoyerPerimetresAffectation(info.getRoles(), _role, codesStructures, _codeRubrique, _publicsVises, _codeEspaceCollaboratif);
			if (v.size() > 0) {
				h.put(info.getCode(), v);
			}
		}
		return h;
	}
	/**
	 * Renvoie la liste des {groupes,périmètres} associé à un role et pour un perimetre.
	 *
	 * @param _role
	 *            the _role
	 * @param codesStructures
	 *            the codes structures
	 * @param _codeRubrique
	 *            the _code rubrique
	 * @param _publicsVises
	 *            the _publics vises
	 * @param _codeEspaceCollaboratif
	 *            the _code espace collaboratif
	 *
	 * @return the hashtable
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static Hashtable<String, Vector<Perimetre>> renvoyerGroupesEtPerimetres(final String _role, final List<String> codesStructures,
																				   final String _codeRubrique, final String _publicsVises, final String _codeEspaceCollaboratif) throws Exception {
		final Hashtable<String, Vector<Perimetre>> h = new Hashtable<>();
		for (final InfosGroupeDsi info : getListeGroupesDsi().values()) {
			final Vector<Perimetre> v = Role.renvoyerPerimetresAffectation(info.getRoles(), _role, codesStructures, _codeRubrique, _publicsVises, _codeEspaceCollaboratif);
			if (v.size() > 0) {
				h.put(info.getCode(), v);
			}
		}
		return h;
	}

	/**
	 * Sélection d'un groupe DSI à partir de l'ensemble des critères combinés.
	 * 
	 * @param code
	 *            the _code
	 * @param type
	 *            the _type
	 * @param libelle
	 *            the _libelle
	 * @param codeStructure
	 *            the _code structure
	 * @param gestionCache
	 *            the _gestion cache
	 * 
	 * @return the int
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public int select(final String code, final String type, final String libelle, final String codeStructure, final String gestionCache) throws Exception {
		final RequeteSQL requeteSelect = new RequeteSQL();
		final ClauseWhere where = new ClauseWhere();
		final ClauseOrderBy orderBy = new ClauseOrderBy("LIBELLE", SensDeTri.ASC);
		if (StringUtils.isNotEmpty(code)) {
			where.setPremiereCondition(ConditionHelper.egalVarchar("CODE", code));
		}
		if (StringUtils.isNotEmpty(type) && !"0000".equals(type)) {
			where.and(ConditionHelper.egalVarchar("TYPE", type));
		}
		if (StringUtils.isNotEmpty(libelle)) {
			where.and(ConditionHelper.rechercheMots("LIBELLE", libelle));
		}
		if (StringUtils.isNotEmpty(codeStructure)) {
			where.and(ConditionHelper.getConditionStructure("CODE_STRUCTURE", codeStructure));
		}
		if (StringUtils.isNotEmpty(gestionCache)) {
			where.and(ConditionHelper.egalVarchar("GESTION_CACHE", gestionCache));
		}
		requeteSelect.where(where).orderBy(orderBy);
		return select(requeteSelect.formaterRequete());
	}

	@Deprecated
	public static Hashtable<String, String> getListeRequetesGroupesPourAffichage(final OMContext _ctx) {
		return getListeRequetesGroupesPourAffichage();
	}

	/**
	 * Récupération pour affichage liste des requêtes.
	 *
	 * @return the liste requetes groupes pour affichage
	 */
	public static Hashtable<String, String> getListeRequetesGroupesPourAffichage() {
		final Hashtable<String, String> res = new Hashtable<>();
		for (final InfosRequeteGroupe info : getListeRequetesGroupes().values()) {
			res.put(info.getAlias(), info.getIntitule());
		}
		return res;
	}

	/**
	 * Renvoie la liste des requetes.
	 * 
	 * @param _ctx
	 *            the _ctx
	 * 
	 * @return the infos requete
	 */
	@Deprecated
	public static Hashtable<String, String> getInfosRequete(final OMContext _ctx) {
		final Hashtable<String, String> res = new Hashtable<>();
		for (final InfosRequeteGroupe info : getListeRequetesGroupes().values()) {
			res.put(info.getAlias(), info.getIntitule());
		}
		return res;
	}

	/**
	 * Récupération d'une requête stockée en mémoire.
	 * 
	 * @param code
	 *            the code
	 * 
	 * @return the infos requete groupe
	 */
	public static InfosRequeteGroupe renvoyerItemRequeteGroupe(final String code) {
		InfosRequeteGroupe res = getListeRequetesGroupes().get(code);
		if (res == null) {
			res = new InfosRequeteGroupe();
		}
		return res;
	}

	@Deprecated
	public static Vector<InfosGroupeDsi> getListeGroupesDynamiques(final OMContext _ctx) {
		return getListeGroupesDynamiques();
	}

	/**
	 * Récupération des groupes dynamiques.
	 *
	 * @return the liste groupes dynamiques
	 *
	 */
	public static Vector<InfosGroupeDsi> getListeGroupesDynamiques() {
		final Vector<InfosGroupeDsi> res = new Vector<>();
		for (final InfosGroupeDsi info : getListeGroupesDsi().values()) {
			if (!info.getRequete().equals("")) {
				res.add(info);
			}
		}
		return res;
	}

	/**
	 * Renvoie la liste des groupes principaux (premier niveau).
	 * 
	 * @return Une liste d'InfosGroupeDsi
	 * @deprecated Méthode plus utilisée
	 */
	@Deprecated
	public static Collection<InfosGroupeDsi> getListeGroupesPrincipaux() {
		return getTopLevelGroupedsi().getListeSousGroupes();
	}

	/**
	 * Renvoie le groupe dsi de plus haut niveau (père des groupes de 1er niveau).
	 * 
	 * @return Un InfosGroupeDsi
	 *
	 */
	private static InfosGroupeDsi getTopLevelGroupedsi() {
		return getListeGroupesDsi().get(CODE_GROUPE_ROOT);
	}

	/**
	 * méthode qui permet de tester le périmètre de modification d'un objet sur un groupe dsi.
	 * 
	 * @param autorisations
	 *            the autorisations
	 * @param sPermission
	 *            the s permission
	 * @param codeGroupe
	 *            the code groupe
	 * 
	 * @return true, if controler permission
	 *
	 */
	public static boolean controlerPermission(final AutorisationBean autorisations, final String sPermission, final String codeGroupe) {
		final PermissionBean permission = new PermissionBean(sPermission);
		final Perimetre perimetre = new Perimetre("", "", "", codeGroupe, "");
		return autorisations.possedePermissionPartielleSurPerimetre(permission, perimetre);
	}

	/**
	 * Affichage dynamique de l'arbre javascript des rubriques.
	 * 
	 * @param code
	 *            the code
	 * @param autorisations
	 *            the autorisations
	 * @param sPermission
	 *            the s permission
	 * 
	 * @return the arbre java script
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public static String getArbreJavaScript(String code, final AutorisationBean autorisations, final String sPermission) throws Exception {
		if (code == null || code.length() == 0) {
			code = CODE_GROUPE_ROOT;
		}
		final JavascriptTreeGroupedsi tree = new JavascriptTreeGroupedsi(autorisations, sPermission);
		tree.load(getTopLevelGroupedsi(), renvoyerItemGroupeDsi(code));
		return tree.print();
	}

	/**
	 * Gets the fil ariane.
	 * 
	 * @param codeGroupe
	 *            the code groupe
	 * @param separateur
	 *            the separateur
	 * 
	 * @return the fil ariane
	 */
	public static String getFilAriane(final String codeGroupe, String separateur) {
		String filAriane = "";
		try {
			if (separateur.length() == 0) {
				separateur = "&gt;";
			}
			// On lit les structures dans le sens ascendant
			InfosGroupeDsi groupe = Groupedsi.renvoyerItemGroupeDsi(codeGroupe);
			while (groupe.getNiveau() > 0) {
				if (filAriane.length() > 0) {
					filAriane = " " + separateur + " " + filAriane;
				}
				filAriane = groupe.getIntitule() + filAriane;
				groupe = groupe.getGroupePere();
			}
		} catch (final Exception ignored) {}
		return filAriane;
	}

	@Override
	public Groupedsi clone() throws CloneNotSupportedException {
		return (Groupedsi) super.clone();
	}
}
