/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.Vector;

import org.apache.commons.lang3.StringUtils;

import com.jsbsoft.jtf.database.OMContext;
import com.univ.objetspartages.sgbd.GroupeutilisateurDB;
import com.univ.utils.ContexteDao;
import com.univ.utils.RequeteGroupeUtil;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;

/**
 * The Class Groupeutilisateur.
 */
public class Groupeutilisateur extends GroupeutilisateurDB {

	/**
	 * Inits the.
	 */
	public void init() {
		setIdGroupeutilisateur(0L);
		setCodeGroupe("");
		setCodeUtilisateur("");
		setSourceImport("");
	}

	/**
	 * Renvoie la liste des groupes d'un utilisateur.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _codeUtilisateur
	 *            the _code utilisateur
	 *
	 * @return @throws Exception
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Deprecated
	public static Vector<String> getVecteurGroupes(final OMContext _ctx, final String _codeUtilisateur) throws Exception {
		return getVecteurGroupes(_codeUtilisateur);
	}

	/**
	 * Renvoie la liste des groupes d'un utilisateur.
	 *
	 * @param _codeUtilisateur
	 *            the _code utilisateur
	 *
	 * @return @throws Exception
	 *
	 * @throws Exception
	 *             the exception
	 */
    public static Vector<String> getVecteurGroupes(final String _codeUtilisateur) throws Exception {
        final Set<String> setGroupes = new HashSet<>();
        if (StringUtils.isEmpty(_codeUtilisateur)) {
            return new Vector<>(0);
        }
        final Map<String, InfosRequeteGroupe> listeRequetes = Groupedsi.getListeRequetesGroupes();
        if (listeRequetes != null) {
            for (final InfosRequeteGroupe infosRequete : listeRequetes.values()) {
                final long ts = System.currentTimeMillis();
                if (infosRequete.getAlias().length() > 0) {
                    // appel du calcul des groupes de l'utilisateur
                    final Collection<String> listeGroupes = RequeteGroupeUtil.getGroupesUtilisateur(infosRequete, _codeUtilisateur, ts);
                    for (final String requeteGroupe : listeGroupes) {
                        setGroupes.add(requeteGroupe);
                    }
                }
            }
        }
        try (ContexteDao ctx = new ContexteDao()) {
            // groupes statiques : ajout apres rafraichissement du cache (si besoin) des groupes dynamiques
            final Groupeutilisateur groupeutilisateur = new Groupeutilisateur();
            groupeutilisateur.setCtx(ctx);
            groupeutilisateur.init();
            final ClauseWhere whereCodeUtilisateur = new ClauseWhere(ConditionHelper.egalVarchar("CODE_UTILISATEUR", _codeUtilisateur));
            groupeutilisateur.select(whereCodeUtilisateur.formaterSQL());
            while (groupeutilisateur.nextItem()) {
                setGroupes.add(groupeutilisateur.getCodeGroupe());
            }
        }
        return new Vector<>(setGroupes);
    }

	/**
	 * Renvoie la liste des groupes d'un utilisateur pour une source d'import.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _codeUtilisateur
	 *            the _code utilisateur
	 * @param sourceImport
	 *            the source import
	 *
	 * @return @throws Exception
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static Vector<String> getVecteurGroupesSourceImport(final OMContext _ctx, final String _codeUtilisateur, final String sourceImport) throws Exception {
		final Vector<String> res = new Vector<>();
		if (StringUtils.isEmpty(_codeUtilisateur)) {
			return res;
		}
		final Groupeutilisateur groupeutilisateur = new Groupeutilisateur();
		groupeutilisateur.setCtx(_ctx);
		groupeutilisateur.init();
		final ClauseWhere whereCodeUtilisateurEtSource = new ClauseWhere(ConditionHelper.egalVarchar("CODE_UTILISATEUR", _codeUtilisateur));
		whereCodeUtilisateurEtSource.and(ConditionHelper.egalVarchar("SOURCE_IMPORT", sourceImport));
		groupeutilisateur.select(whereCodeUtilisateurEtSource.formaterSQL());
		while (groupeutilisateur.nextItem()) {
			res.add(groupeutilisateur.getCodeGroupe());
		}
		return res;
	}

    @Deprecated
    public static Vector<String> getVecteurUtilisateursSourceImport(final OMContext _ctx, final String _codeGroupe, final String sourceImport) throws Exception {
        return getVecteurUtilisateursSourceImport(_codeGroupe, sourceImport);
    }

	/**
	 * Renvoie la liste des groupes d'un utilisateur pour une source d'import.
	 *
	 * @param _codeGroupe
	 *            the _code groupe
	 * @param sourceImport
	 *            the source import
	 *
	 * @return @throws Exception
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static Vector<String> getVecteurUtilisateursSourceImport(final String _codeGroupe, final String sourceImport) throws Exception {
		final Vector<String> res = new Vector<>();
		if (StringUtils.isEmpty(_codeGroupe)) {
			return res;
		}
		try (ContexteDao ctx = new ContexteDao()) {
			final Groupeutilisateur groupeutilisateur = new Groupeutilisateur();
			groupeutilisateur.setCtx(ctx);
			groupeutilisateur.init();
			final ClauseWhere whereCodeGroupe = new ClauseWhere(ConditionHelper.egalVarchar("CODE_GROUPE", _codeGroupe));
			whereCodeGroupe.and(ConditionHelper.egalVarchar("SOURCE_IMPORT", sourceImport));
			groupeutilisateur.select(whereCodeGroupe.formaterSQL());
			while (groupeutilisateur.nextItem()) {
				res.add(groupeutilisateur.getCodeUtilisateur());
			}
		}
		return res;
	}

	/**
	 * Valorise la liste des groupes d'un utilisateur.
	 *
	 * Cette méthode conserve les sources d'import pour les relations déjà présentes.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param codeUtilisateur
	 *            the _code utilisateur
	 * @param listeGroupes
	 *            the liste groupes
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static void setVecteurGroupes(final OMContext _ctx, final String codeUtilisateur, final Vector<String> listeGroupes) throws Exception {
		final Groupeutilisateur groupeutilisateur = new Groupeutilisateur();
		groupeutilisateur.setCtx(_ctx);
		groupeutilisateur.init();
		/* Ajout des nouvelles valeurs */
		for (String codeGroupe : listeGroupes) {
			if (!StringUtils.startsWith(codeGroupe, "#AUTO#")) {
				final InfosGroupeDsi infosGroupes = Groupedsi.renvoyerItemGroupeDsi(codeGroupe);
				/* JSS 20050510 : On ne stocke pas les groupes dynamiques */
				if (StringUtils.isEmpty(infosGroupes.getRequete())) {
					if (groupeutilisateur.select(whereCodeUtilisateurEtCodeGroupe(codeUtilisateur, codeGroupe).formaterSQL()) == 0) {
						groupeutilisateur.init();
						groupeutilisateur.setCodeUtilisateur(codeUtilisateur);
						groupeutilisateur.setCodeGroupe(codeGroupe);
						groupeutilisateur.add();
					}
				}
			}
		}
		/* Suppression des valeurs qui ne sont plus passées dans listeGroupe */
		final Vector<String> listeAPurger = Groupeutilisateur.getVecteurGroupes(codeUtilisateur);
		for (String codeGroupeAPurger : listeAPurger) {
			if (!listeGroupes.contains(codeGroupeAPurger)) {
				groupeutilisateur.init();
				if (groupeutilisateur.select(whereCodeUtilisateurEtCodeGroupe(codeUtilisateur, codeGroupeAPurger).formaterSQL()) > 0) {
					while (groupeutilisateur.nextItem()) {
						groupeutilisateur.delete();
					}
				}
			}
		}
	}

	/**
	 * Synchronise la liste des groupes d'un utilisateur pour une source d'import.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _codeUtilisateur
	 *            the _code utilisateur
	 * @param listeGroupes
	 *            the liste groupes
	 * @param sourceImport
	 *            the source import
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static void synchroniserGroupesUtilisateur(final OMContext _ctx, final String _codeUtilisateur, final Vector<String> listeGroupes, final String sourceImport)
		throws Exception {
		final Vector<String> oldGroupes = getVecteurGroupesSourceImport(_ctx, _codeUtilisateur, sourceImport);
		final Groupeutilisateur groupeutilisateur = new Groupeutilisateur();
		groupeutilisateur.setCtx(_ctx);
		groupeutilisateur.init();
		/* Ajout des nouvelles valeurs */
		for (String codeGroupe : listeGroupes) {
			if (groupeutilisateur.select(whereCodeUtilisateurEtCodeGroupe(_codeUtilisateur, codeGroupe).formaterSQL()) == 1) {
				groupeutilisateur.nextItem();
				groupeutilisateur.setSourceImport(sourceImport);
				groupeutilisateur.update();
			}
			else {
				groupeutilisateur.init();
				groupeutilisateur.setCodeUtilisateur(_codeUtilisateur);
				groupeutilisateur.setCodeGroupe(codeGroupe);
				groupeutilisateur.setSourceImport(sourceImport);
				groupeutilisateur.add();
			}
		}
		/* Suppression des valeurs qui ne sont plus passées dans listeGroupe */
		for (String codeGroupeAPurger : oldGroupes) {
			if (! listeGroupes.contains(codeGroupeAPurger)) {
				groupeutilisateur.init();
				if (groupeutilisateur.select(whereCodeUtilisateurEtCodeGroupe(_codeUtilisateur, codeGroupeAPurger).formaterSQL()) > 0) {
					while (groupeutilisateur.nextItem()) {
						groupeutilisateur.delete();
					}
				}
			}
		}
	}

	private static ClauseWhere whereCodeUtilisateurEtCodeGroupe(final String codeUtilisateur, final String codeGroupe) {
		final ClauseWhere whereCodeUserEtGroupe = new ClauseWhere(ConditionHelper.egalVarchar("CODE_UTILISATEUR", codeUtilisateur));
		whereCodeUserEtGroupe.and(ConditionHelper.egalVarchar("CODE_GROUPE", codeGroupe));
		return whereCodeUserEtGroupe;
	}

	/**
	 * Synchronise la liste des utilisateurs d'un groupe pour une source d'import.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _codeGroupe
	 *            the _code groupe
	 * @param listeUtilisateurs
	 *            the liste utilisateurs
	 * @param sourceImport
	 *            the source import
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static void synchroniserUtilisateursGroupe(final OMContext _ctx, final String _codeGroupe, final Vector<String> listeUtilisateurs, final String sourceImport)
		throws Exception {
		final Vector<String> oldUsers = getVecteurUtilisateursSourceImport(_ctx, _codeGroupe, sourceImport);
		final Groupeutilisateur groupeutilisateur = new Groupeutilisateur();
		groupeutilisateur.setCtx(_ctx);
		groupeutilisateur.init();
		/* Ajout des nouvelles valeurs */
		for (String codeUtilisateur : listeUtilisateurs) {
			if (groupeutilisateur.select(whereCodeUtilisateurEtCodeGroupe(codeUtilisateur, _codeGroupe).formaterSQL()) == 1) {
				groupeutilisateur.nextItem();
				groupeutilisateur.setSourceImport(sourceImport);
				groupeutilisateur.update();
			}
			else {
				groupeutilisateur.init();
				groupeutilisateur.setCodeUtilisateur(codeUtilisateur);
				groupeutilisateur.setCodeGroupe(_codeGroupe);
				groupeutilisateur.setSourceImport(sourceImport);
				groupeutilisateur.add();
			}
		}
		/* Suppression des valeurs qui ne sont plus passées dans listeUtilisateurs */
		for (String codeUtilisateurAPurger : oldUsers) {
			if (!listeUtilisateurs.contains(codeUtilisateurAPurger)) {
				groupeutilisateur.init();
				if (groupeutilisateur.select(whereCodeUtilisateurEtCodeGroupe(codeUtilisateurAPurger, _codeGroupe).formaterSQL()) > 0) {
					while (groupeutilisateur.nextItem()) {
						groupeutilisateur.delete();
					}
				}
			}
		}
	}

	/**
	 * Ajoute une liste de groupes à un utilisateur.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _codeUtilisateur
	 *            the _code utilisateur
	 * @param listeGroupes
	 *            the liste groupes
	 * @throws Exception
	 *             the exception
	 */
	public static void ajouterGroupesUtilisateur(final OMContext _ctx, final String _codeUtilisateur, final Vector<String> listeGroupes) throws Exception {
		final Groupeutilisateur groupeutilisateur = new Groupeutilisateur();
		groupeutilisateur.setCtx(_ctx);
		groupeutilisateur.init();
		/* Ajout des nouvelles valeurs */
		final Enumeration<String> eListeGroupes = listeGroupes.elements();
		while (eListeGroupes.hasMoreElements()) {
			final String codeGroupe = eListeGroupes.nextElement();
			if (groupeutilisateur.select(whereCodeUtilisateurEtCodeGroupe(_codeUtilisateur, codeGroupe).formaterSQL()) == 0) {
				groupeutilisateur.init();
				groupeutilisateur.setCodeUtilisateur(_codeUtilisateur);
				groupeutilisateur.setCodeGroupe(codeGroupe);
				groupeutilisateur.add();
			}
		}
	}

	/**
	 * Ajoute une liste d'utilisateurs à un groupe.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _codeGroupe
	 *            the _code groupe
	 * @param listeUtilisateurs
	 *            the liste utilisateurs
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static void ajouterUtilisateursGroupe(final OMContext _ctx, final String _codeGroupe, final Vector<String> listeUtilisateurs) throws Exception {
		final Groupeutilisateur groupeutilisateur = new Groupeutilisateur();
		groupeutilisateur.setCtx(_ctx);
		groupeutilisateur.init();
		/* Ajout des nouvelles valeurs */
		final Enumeration<String> eListeUtilisateurs = listeUtilisateurs.elements();
		while (eListeUtilisateurs.hasMoreElements()) {
			final String codeUtilisateur = eListeUtilisateurs.nextElement();
			if (groupeutilisateur.select(whereCodeUtilisateurEtCodeGroupe(codeUtilisateur, _codeGroupe).formaterSQL()) == 0) {
				groupeutilisateur.init();
				groupeutilisateur.setCodeUtilisateur(codeUtilisateur);
				groupeutilisateur.setCodeGroupe(_codeGroupe);
				groupeutilisateur.add();
			}
		}
	}

	/**
	 * Valorise la liste des utilisateurs d'un groupe.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _listeUtilisateurs
	 *            the _liste utilisateurs
	 * @param _codeGroupe
	 *            the _code groupe
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static void setVecteurUtilisateurs(final OMContext _ctx, final Vector<String> _listeUtilisateurs, final String _codeGroupe) throws Exception {
		/* On supprime les relations obsolètes */
		final Groupeutilisateur groupeutilisateur = new Groupeutilisateur();
		groupeutilisateur.setCtx(_ctx);
		groupeutilisateur.init();
		final ClauseWhere whereCodeGroupe = new ClauseWhere(ConditionHelper.egalVarchar("CODE_GROUPE", _codeGroupe));
		if (groupeutilisateur.select(whereCodeGroupe.formaterSQL()) >= 0) {
			while (groupeutilisateur.nextItem()) {
				final String codeUtilisateur = groupeutilisateur.getCodeUtilisateur();
				if (! _listeUtilisateurs.contains(codeUtilisateur)) {
					groupeutilisateur.delete();
				}
			}
		}
		/* Ajout des nouvelles valeurs */
		final Enumeration<String> eListeUtilisateurs = _listeUtilisateurs.elements();
		while (eListeUtilisateurs.hasMoreElements()) {
			final String codeUtilisateur = eListeUtilisateurs.nextElement();
			if (groupeutilisateur.select(whereCodeUtilisateurEtCodeGroupe(codeUtilisateur, _codeGroupe).formaterSQL()) == 0) {
				groupeutilisateur.init();
				groupeutilisateur.setCodeUtilisateur(codeUtilisateur);
				groupeutilisateur.setCodeGroupe(_codeGroupe);
				groupeutilisateur.add();
			}
		}
	}

	@Deprecated
	public static Vector<String> getVecteurUtilisateurs(final OMContext _ctx, final String _codesGroupe) throws Exception {
		return getVecteurUtilisateurs(_codesGroupe);
	}

	/**
	 * Renvoie la liste des utilisateurs d'un groupe (parcours hiérarchique).
	 *
	 * @param _codesGroupe
	 *            the _codes groupe
	 *
	 * @return @throws Exception
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static Vector<String> getVecteurUtilisateurs(final String _codesGroupe) throws Exception {
		// Le HashSet permet d'éviter les doublons
		final HashSet<String> set = new HashSet<>();
		final Vector<InfosGroupeDsi> listeGroupes = new Vector<>();
		final StringTokenizer st = new StringTokenizer(_codesGroupe, ";");
		while (st.hasMoreTokens()) {
			String codeGroupe = st.nextToken();
			InfosGroupeDsi infosGroupe = Groupedsi.renvoyerItemGroupeDsi(codeGroupe);
			// Pas d'héritage sur les groupes dynamiques
			if (infosGroupe.getRequete().equals("")) {
				listeGroupes.addAll(infosGroupe.getListeSousGroupesTousNiveaux());
			}
			listeGroupes.add(infosGroupe);
		}
		/* Requete sur tous les groupes */
		final List<String> codesGroupesPossible = new ArrayList<>();
		for (InfosGroupeDsi infosGroupe : listeGroupes) {
			if (StringUtils.isNotBlank(infosGroupe.getRequete())) {
				/* JSS 20050510 : groupes dynamiques */
				final Collection<String> codesUtilisateurs = RequeteGroupeUtil.getVecteurUtilisateurs(infosGroupe.getCode(), null);
				for (final String code : codesUtilisateurs) {
					if (!set.contains(code)) {
						set.add(code);
					}
				}
			} else {
				codesGroupesPossible.add(infosGroupe.getCode());
			}
		}
		/* Requete sur la table utilisateur */
		if (!codesGroupesPossible.isEmpty()) {
			final Groupeutilisateur groupe = new Groupeutilisateur();
			try (ContexteDao ctx = new ContexteDao()) {
				groupe.setCtx(ctx);
				groupe.init();
				final ClauseWhere whereCodeGroupe = new ClauseWhere(ConditionHelper.in("CODE_GROUPE", codesGroupesPossible));
				groupe.select(whereCodeGroupe.formaterSQL());
				while (groupe.nextItem()) {
					set.add(groupe.getCodeUtilisateur());
				}
			}
		}
		return new Vector<>(set);
	}

	/**
	 * Destruction des relations associées à un utilisateur.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _codeUtilisateur
	 *            the _code utilisateur
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static void deleteParUtilisateur(final OMContext _ctx, final String _codeUtilisateur) throws Exception {
		final Groupeutilisateur groupeutilisateur = new Groupeutilisateur();
		groupeutilisateur.setCtx(_ctx);
		groupeutilisateur.init();
		final ClauseWhere whereCodeUtilisateur = new ClauseWhere(ConditionHelper.egalVarchar("CODE_UTILISATEUR", _codeUtilisateur));
		groupeutilisateur.select(whereCodeUtilisateur.formaterSQL());
		while (groupeutilisateur.nextItem()) {
			groupeutilisateur.delete();
		}
	}

	/**
	 * Destruction des relations associées à un utilisateur et a une source d'import.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _codeUtilisateur
	 *            the _code utilisateur
	 * @param _sourceImport
	 *            the _source import
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static void deleteParUtilisateurSourceImport(final OMContext _ctx, final String _codeUtilisateur, final String _sourceImport) throws Exception {
		final Groupeutilisateur groupeutilisateur = new Groupeutilisateur();
		groupeutilisateur.setCtx(_ctx);
		groupeutilisateur.init();
		final ClauseWhere whereCodeUtilisateurEtSource = new ClauseWhere(ConditionHelper.egalVarchar("CODE_UTILISATEUR", _codeUtilisateur));
		whereCodeUtilisateurEtSource.and(ConditionHelper.egalVarchar("SOURCE_IMPORT", _sourceImport));
		groupeutilisateur.select(whereCodeUtilisateurEtSource.formaterSQL());
		while (groupeutilisateur.nextItem()) {
			groupeutilisateur.delete();
		}
	}

	/**
	 * Destruction des relations associées à un groupe.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _codeGroupe
	 *            the _code groupe
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static void deleteParGroupe(final OMContext _ctx, final String _codeGroupe) throws Exception {
		final Groupeutilisateur groupeutilisateur = new Groupeutilisateur();
		groupeutilisateur.setCtx(_ctx);
		groupeutilisateur.init();
		final ClauseWhere whereCodeGroupe = new ClauseWhere(ConditionHelper.egalVarchar("CODE_GROUPE", _codeGroupe));
		groupeutilisateur.select(whereCodeGroupe.formaterSQL());
		while (groupeutilisateur.nextItem()) {
			groupeutilisateur.delete();
		}
	}

	/**
	 * Destruction des relations associées à un groupe.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _codeGroupe
	 *            the _code groupe
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static void deleteUtilisateurDuGroupe(final OMContext _ctx, final String _codeUtilisateur, final String _codeGroupe) throws Exception {
		final Groupeutilisateur groupeutilisateur = new Groupeutilisateur();
		groupeutilisateur.setCtx(_ctx);
		groupeutilisateur.init();
		groupeutilisateur.select(whereCodeUtilisateurEtCodeGroupe(_codeUtilisateur, _codeGroupe).formaterSQL());
		while (groupeutilisateur.nextItem()) {
			groupeutilisateur.delete();
		}
	}

	@Deprecated
	public static void deleteParGroupeSourceImport(final OMContext _ctx, final String _codeGroupe, final String _sourceImport) throws Exception {
		deleteParGroupeSourceImport(_codeGroupe, _sourceImport);
	}

	/**
	 * Destruction des relations associées à un groupe et a une source d'import.
	 *
	 * @param _sourceImport
	 *            the _source import
	 * @param _codeGroupe
	 *            the _code groupe
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static void deleteParGroupeSourceImport(final String _codeGroupe, final String _sourceImport) throws Exception {
		try (ContexteDao ctx = new ContexteDao()) {
			final Groupeutilisateur groupeutilisateur = new Groupeutilisateur();
			groupeutilisateur.setCtx(ctx);
			groupeutilisateur.init();
			final ClauseWhere whereCodeGroupe = new ClauseWhere(ConditionHelper.egalVarchar("CODE_GROUPE", _codeGroupe));
			whereCodeGroupe.and(ConditionHelper.egalVarchar("SOURCE_IMPORT", _sourceImport));
			groupeutilisateur.select(whereCodeGroupe.formaterSQL());
			while (groupeutilisateur.nextItem()) {
				groupeutilisateur.delete();
			}
		}
	}

    /**
     * Renvoie la liste des codes de groupes d'un utilisateur (hors groupes dynamiques).
     * @param codeUtilisateur le code de l'utilisateur dont on souhaite récupérer les groupes.
     * @return La liste des groupes de l'utilisateur. Si le code de l'utilisateur est vide/null ou si l'utilisateur n'a aucun groupe, on retourne une liste vide.
     * @throws Exception A cause des espaces collaboratifs, à Supprimer lorsqu'ils auront étés migrés
     */
    public static Vector<String> getAllDatasourceGroupsCodesByUserCode(final String codeUtilisateur) throws Exception {
        final Set<String> setGroupes = new HashSet<>();
        if (StringUtils.isEmpty(codeUtilisateur)) {
            return new Vector<>(0);
        }
        try (ContexteDao ctx = new ContexteDao()) {
            // groupes statiques : ajout apres rafraichissement du cache (si besoin) des groupes dynamiques
            final Groupeutilisateur groupeutilisateur = new Groupeutilisateur();
            groupeutilisateur.setCtx(ctx);
            groupeutilisateur.init();
            final ClauseWhere whereCodeUtilisateur = new ClauseWhere(ConditionHelper.egalVarchar("CODE_UTILISATEUR", codeUtilisateur));
            groupeutilisateur.select(whereCodeUtilisateur.formaterSQL());
            while (groupeutilisateur.nextItem()) {
                setGroupes.add(groupeutilisateur.getCodeGroupe());
            }
        }
        return new Vector<>(setGroupes);
    }

    public static Collection<String> getAllDynamicGroupsCodesByUserCode(String _codeUtilisateur) throws Exception{
        final Set<String> setGroupes = new HashSet<>();
        if (StringUtils.isEmpty(_codeUtilisateur)) {
            return new Vector<>(0);
        }
        final Map<String, InfosRequeteGroupe> listeRequetes = Groupedsi.getListeRequetesGroupes();
        if (listeRequetes != null) {
            for (final InfosRequeteGroupe infosRequete : listeRequetes.values()) {
                final long ts = System.currentTimeMillis();
                if (infosRequete.getAlias().length() > 0) {
                    // appel du calcul des groupes de l'utilisateur
                    final Collection<String> listeGroupes = RequeteGroupeUtil.getGroupesUtilisateur(infosRequete, _codeUtilisateur, ts);
                    for (final String requeteGroupe : listeGroupes) {
                        setGroupes.add(requeteGroupe);
                    }
                }
            }
        }
        return new Vector<>(setGroupes);
    }
}
