/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.TreeMap;

/**
 * Informations stockées en mémoire pour un groupe DSI.
 */
public class InfosGroupeDsi implements Serializable {

	/**
	 *
	 */
	private static final long serialVersionUID = 7437361262139171837L;

	/** The id **/
	private Long id = 0L;

	/** The code. */
	private String code = "";

	/** The structure code */
	private String codeStructure = "";

	/** The intitule. */
	private String intitule = "";

	/** The code page tete. */
	private String codePageTete = "";

	/** The roles. */
	private String roles = "";

	/** The code groupe pere. */
	private String codeGroupePere = "";

	/** The requete. */
	private String requete = "";

	/** The selectionnable. */
	private String selectionnable = "";

	/** The type. */
	private String type = "";

	/** The niveau. */
	private int niveau = 0;

	/** The groupe pere. */
	private InfosGroupeDsi groupePere = null;

	/** The liste sous groupes. */
	private Collection<InfosGroupeDsi> listeSousGroupes = null;

	/** The liste sous groupes tous niveaux. */
	private Collection<InfosGroupeDsi> listeSousGroupesTousNiveaux = null;

	/**
	 * Constructeur vide.
	 */
	public InfosGroupeDsi() {
		this.listeSousGroupes = new ArrayList<InfosGroupeDsi>();
	}

	/**
	 * Constructeur de InfosGroupeDsi.
	 *
	 * @param groupe
	 *            the groupe
	 */
	public InfosGroupeDsi(final Groupedsi groupe) {
		this.id = groupe.getIdGroupedsi();
		this.code = groupe.getCode();
		this.codeStructure = groupe.getCodeStructure();
		this.intitule = groupe.getLibelle();
		this.type = groupe.getType();
		this.codeGroupePere = groupe.getCodeGroupePere();
		this.codePageTete = groupe.getCodePageTete();
		this.roles = groupe.getRoles();
		this.requete = groupe.getRequeteGroupe();
		this.selectionnable = groupe.getSelectionnable();
		this.listeSousGroupes = new ArrayList<InfosGroupeDsi>();
	}

	/**
	 * Get the id
	 *
	 * @return id as a <code>Long</code> value
	 */
	public Long getId() {
		return id;
	}

	/**
	 * Sets the code.
	 *
	 * @param code
	 *            The code to set.
	 */
	public void setCode(final String code) {
		this.code = code;
	}

	/**
	 * Get the structure's code
	 *
	 * @return the code as a <code>String</code> value
	 */
	public String getCodeStructure() {
		return codeStructure;
	}

	/**
	 * Gets the requete.
	 *
	 * @return Returns the requete.
	 */
	public String getRequete() {
		return requete;
	}

	/**
	 * Sets the requete.
	 *
	 * @param requete
	 *            The requete to set.
	 */
	public void setRequete(final String requete) {
		this.requete = requete;
	}

	/**
	 * Gets the code.
	 *
	 * @return the code
	 */
	public String getCode() {
		return code;
	}

	/**
	 * Gets the code page tete.
	 *
	 * @return the code page tete
	 */
	public String getCodePageTete() {
		return codePageTete;
	}

	/**
	 * Gets the intitule.
	 *
	 * @return the intitule
	 */
	public String getIntitule() {
		return intitule;
	}

	/**
	 * Insérez la description de la méthode à cet endroit. Date de création : (10/01/2003 13:50:46)
	 *
	 * @return String
	 */
	public String getType() {
		return type;
	}

	/**
	 * Gets the roles.
	 *
	 * @return Returns the roles.
	 */
	public String getRoles() {
		return roles;
	}

	/**
	 * Gets the liste sous groupes.
	 *
	 * @return Returns the sousGroupes.
	 */
	public Collection<InfosGroupeDsi> getListeSousGroupes() {
		return listeSousGroupes;
	}

	/**
	 * Trie les sous-structures sur leur libellé long.
	 *
	 * @return the liste sous groupes sorted by libelle
	 */
	public Collection<InfosGroupeDsi> getListeSousGroupesSortedByLibelle() {
		final Iterator<InfosGroupeDsi> listeSousGroupesIt = listeSousGroupes.iterator();
		final TreeMap<String, InfosGroupeDsi> sortedListeSousGroupes = new TreeMap<String, InfosGroupeDsi>();
		InfosGroupeDsi sousGroupe = null;
		while (listeSousGroupesIt.hasNext()) {
			sousGroupe = listeSousGroupesIt.next();
			sortedListeSousGroupes.put(sousGroupe.getIntitule() + "_" + sousGroupe.getCode(), sousGroupe);
		}
		return sortedListeSousGroupes.values();
	}

	/**
	 * Renvoie la liste des sous-groupes de ce groupe tous niveaux confondus.
	 *
	 * @return La liste des sous-groupes de ce groupe tous niveaux confondus
	 */
	public Collection<InfosGroupeDsi> getListeSousGroupesTousNiveaux() {
		if (listeSousGroupesTousNiveaux == null) {
			listeSousGroupesTousNiveaux = new ArrayList<InfosGroupeDsi>();
			final Iterator<InfosGroupeDsi> listeSousGroupesIt = getListeSousGroupes().iterator();
			InfosGroupeDsi sousGroupe = null;
			while (listeSousGroupesIt.hasNext()) {
				sousGroupe = listeSousGroupesIt.next();
				listeSousGroupesTousNiveaux.addAll(sousGroupe.getListeSousGroupesTousNiveaux()); // récursif
				listeSousGroupesTousNiveaux.add(sousGroupe);
			}
		}
		return listeSousGroupesTousNiveaux;
	}

	/**
	 * Gets the niveau.
	 *
	 * @return Returns the niveau.
	 */
	public int getNiveau() {
		return niveau;
	}

	/**
	 * Gets the code groupe pere.
	 *
	 * @return Returns the codeGroupePere.
	 */
	public String getCodeGroupePere() {
		return codeGroupePere;
	}

	/**
	 * Gets the groupe pere.
	 *
	 * @return Returns the codeGroupePere.
	 */
	public InfosGroupeDsi getGroupePere() {
		return groupePere;
	}

	/**
	 * Ajoute un sous-groupe à la liste des sous-groupes.
	 *
	 * @param sousGroupe
	 *            Un sous-groupe
	 */
	public void addSousGroupe(final InfosGroupeDsi sousGroupe) {
		listeSousGroupes.add(sousGroupe);
	}

	/**
	 * Rattache le groupe à son groupe père et met à jour son niveau et la liste des sous-groupes de son groupe père.
	 *
	 * @param groupe
	 *            Le groupe père
	 */
	public void rattacheA(final InfosGroupeDsi groupe) {
		this.groupePere = groupe;
		this.groupePere.addSousGroupe(this);
		updateNiveau();
	}

	/**
	 * Met a jour le niveau du groupe et de ses sous-groupes en fonction du niveau du groupe père.
	 */
	private void updateNiveau() {
		// Mise a jour du niveau du groupe
		niveau = this.groupePere.getNiveau() + 1;
		// Mise a jour du niveau des sous-groupes
		for (InfosGroupeDsi infosGroupeDsi : getListeSousGroupes()) {
			infosGroupeDsi.updateNiveau(); // récursif
		}
	}

	/**
	 * Vérifie si ce groupe contient le groupe passé en paramètre (on teste si le groupe passé en paramètre ou un de ses groupes père ne serait pas par hasard égal à ce groupe.
	 *
	 * @param groupe
	 *            the groupe
	 *
	 * @return true, if contains
	 *
	 */
	public boolean contains(InfosGroupeDsi groupe) {
		boolean inFamilia = false;
		while (groupe != null && !inFamilia) {
			if (groupe.equals(this)) {
				inFamilia = true;
			} else {
				groupe = groupe.getGroupePere(); // on remonte
			}
		}
		return inFamilia;
	}

	/**
	 * Gets the selectionnable.
	 *
	 * @return the selectionnable
	 */
	public String getSelectionnable() {
		return selectionnable;
	}

	/**
	 * Checks if is selectionnable.
	 *
	 * @return true, if is selectionnable
	 */
	public boolean isSelectionnable() {
		boolean res = false;
		if ("1".equals(selectionnable)) {
			res = true;
		}
		return res;
	}
}
