/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.TreeMap;

import org.apache.commons.lang3.StringUtils;

import com.univ.utils.Chaine;
import com.univ.utils.ContexteUniv;

/**
 * Classe permettant de stocker en mémoire les informations relatives à une rubrique.
 */
public class InfosRubriques implements Serializable {

	/**
	 *
	 */
	private static final long serialVersionUID = -8136833859411710834L;

	/** The id rubrique. */
	private Long idRubrique = (long) 0;

	/** The code. */
	private String code = StringUtils.EMPTY;

	/** The code rubrique mere. */
	private String codeRubriqueMere = StringUtils.EMPTY;

	/** The langue. */
	private String langue = "0";

	/** The type rubrique. */
	private String typeRubrique = StringUtils.EMPTY;

	/** The page accueil. */
	private String pageAccueil = StringUtils.EMPTY;

	/** The intitule. */
	private String intitule = StringUtils.EMPTY;

	/** The onglet. */
	private String onglet = StringUtils.EMPTY;

	/** The accroche. */
	private String accroche = StringUtils.EMPTY;

	/** The contact. */
	private String contact = StringUtils.EMPTY;

	/** The couleur fond. */
	private String couleurFond = StringUtils.EMPTY;

	/** The couleur titre. */
	private String couleurTitre = StringUtils.EMPTY;

	/** The encadre. */
	private String encadre = StringUtils.EMPTY;

	private long idMediaBandeau = 0l;

	/** The url bandeau. */
	private String urlBandeau = StringUtils.EMPTY;

	/** The ordre. */
	private String ordre = StringUtils.EMPTY;

	/** The gestion encadre. */
	private String gestionEncadre = StringUtils.EMPTY;

	/** The categorie. */
	private String categorie = StringUtils.EMPTY;

	/** The encadre sous rubrique. */
	private boolean encadreSousRubrique = false;

	/** The niveau. */
	private int niveau = 0;

	/** The groupes dsi. */
	private Set<String> groupesDsi = new HashSet<>();

	/** The rubrique mere. */
	private InfosRubriques rubriqueMere = null;

	/** The liste sous rubriques. */
	private TreeMap<String, InfosRubriques> listeSousRubriques = null;

	/** The liste sous rubriques tous niveaux. */
	private Collection<InfosRubriques> listeSousRubriquesTousNiveaux = null;

	private String urlPicto = StringUtils.EMPTY;

	private long idMediaPicto = 0l;

	/**
	 * Constructeur vide.
	 *
	 * @param code
	 *            the code
	 */
	public InfosRubriques(final String code) {
		this.code = code;
		this.listeSousRubriques = new TreeMap<>();
	}

	/**
	 * Constructeur.
	 *
	 * @param rubrique
	 *            La rubrique à partir de laquelle on construit l'objet InfosRubriques
	 */
	public InfosRubriques(final Rubrique rubrique) {
		this.idRubrique = rubrique.getIdRubrique();
		this.code = rubrique.getCode();
		this.codeRubriqueMere = rubrique.getCodeRubriqueMere();
		this.langue = rubrique.getLangue();
		this.typeRubrique = rubrique.getTypeRubrique();
		this.pageAccueil = rubrique.getPageAccueil();
		this.intitule = rubrique.getIntitule();
		this.onglet = rubrique.getNomOnglet();
		this.accroche = rubrique.getAccroche();
		this.contact = rubrique.getContact();
		this.couleurTitre = rubrique.getCouleurTitre();
		this.couleurFond = rubrique.getCouleurFond();
		this.encadre = rubrique.getEncadre();
		this.ordre = rubrique.getOrdre();
		this.groupesDsi = Chaine.getHashSetAccolades(rubrique.getGroupesDsi());
		this.gestionEncadre = rubrique.getGestionEncadre();
		this.encadreSousRubrique = "1".equals(rubrique.getEncadreSousRubrique());
		this.listeSousRubriques = new TreeMap<>();
		this.categorie = rubrique.getCategorie();
	}

	/**
	 * Gets the code.
	 *
	 * @return the code
	 */
	public String getCode() {
		return code;
	}

	/**
	 * Gets the code rubrique mere.
	 *
	 * @return the code rubrique mere
	 */
	public String getCodeRubriqueMere() {
		return codeRubriqueMere;
	}

	/**
	 * Gets the langue.
	 *
	 * @return the langue
	 */
	public String getLangue() {
		return langue;
	}

	/**
	 * Gets the gestion encadre.
	 *
	 * @return the gestion encadre
	 */
	public String getGestionEncadre() {
		return gestionEncadre;
	}

	/**
	 * Gets the intitule.
	 *
	 * @return the intitule
	 */
	public String getIntitule() {
		return intitule;
	}

	/**
	 * Traitement de l'ordre des rubriques avec numero d'ordre + separateur ex : 01#intitule.
	 *
	 * @param strSeparateur
	 *            : caractère séparateur
	 *
	 * @return the intitule
	 * @deprecated l'ordre des rubriques est géré via un champ dédié depuis la v5
	 */
	@Deprecated
	public String getIntitule(final String strSeparateur) {
		String strRes = StringUtils.EMPTY;
		if (intitule.contains(strSeparateur)) {
			strRes = intitule.substring(intitule.indexOf(strSeparateur) + 1);
		} else {
			strRes = intitule;
		}
		return strRes;
	}

	/**
	 * Gets the niveau.
	 *
	 * @return the niveau
	 */
	public int getNiveau() {
		return niveau;
	}

	/**
	 * Gets the onglet.
	 *
	 * @return the onglet
	 */
	public String getOnglet() {
		return onglet;
	}

	/**
	 * Gets the contact.
	 *
	 * @return the contact
	 */
	public String getContact() {
		return contact;
	}

	/**
	 * Renvoie la rubrique mère de cette rubrique.
	 *
	 * @return La rubrique mère de cette rubrique
	 */
	public InfosRubriques getRubriqueMere() {
		return rubriqueMere;
	}

	/**
	 * Gets the type rubrique.
	 *
	 * @return the type rubrique
	 */
	public String getTypeRubrique() {
		return typeRubrique;
	}

	/**
	 * Insérez la description de la méthode à cet endroit. Date de création : (16/12/2001 15:13:49)
	 *
	 * @return String
	 */
	public String getCouleurFond() {
		return couleurFond;
	}

	/**
	 * Insérez la description de la méthode à cet endroit. Date de création : (16/12/2001 15:13:28)
	 *
	 * @return String
	 */
	public String getCouleurTitre() {
		return couleurTitre;
	}

	/**
	 * Gets the encadre.
	 *
	 * @return the encadre
	 */
	public String getEncadre() {
		return encadre;
	}

	/**
	 * Renvoie true si l'encadré est affichable dans les sous-rubriques.
	 *
	 * @return true, if checks if is encadre sous rubrique
	 */
	public boolean isEncadreSousRubrique() {
		return encadreSousRubrique;
	}

	/**
	 * Gets the id rubrique.
	 *
	 * @return the id rubrique
	 */
	public Long getIdRubrique() {
		return idRubrique;
	}

	/**
	 * Insérez la description de la méthode à cet endroit. Date de création : (16/12/2001 15:12:34)
	 *
	 * @return String
	 * @deprecated ne se met pas à jour lors de la modification d'un média
	 */
	@Deprecated
	public String getUrlBandeau() {
		return urlBandeau;
	}

	/**
	 * Insérez la description de la méthode à cet endroit. Date de création : (16/12/2001 15:12:34)
	 *
	 * @param urlBandeau
	 *            the url bandeau
	 * @deprecated ne se met pas à jour lors de la modification d'un média
	 */
	@Deprecated
	public void setUrlBandeau(final String urlBandeau) {
		this.urlBandeau = urlBandeau;
	}

	/**
	 * Gets the accroche.
	 *
	 * @return Returns the accroche.
	 */
	public String getAccroche() {
		return accroche;
	}

	/**
	 * Gets the ordre.
	 *
	 * @return Returns the ordre.
	 */
	public String getOrdre() {
		return ordre;
	}

	/**
	 * Gets the groupes dsi.
	 *
	 * @return the groupes dsi
	 */
	public Set<String> getGroupesDsi() {
		return groupesDsi;
	}

	/**
	 * Gets the libelle affichable.
	 *
	 * @return the libelle affichable
	 */
	public String getLibelleAffichable() {
		return intitule;
	}

	/**
	 * Renvoie la liste triée sur l'ordre des sous-rubriques de cette rubrique.
	 *
	 * @return La liste triée sur l'ordre des sous-rubriques de cette rubrique
	 */
	public Collection<InfosRubriques> getListeSousRubriques() {
		return listeSousRubriques.values();
	}

	/**
	 * Renvoie la liste des sous-rubriques de cette rubrique tous niveaux confondus.
	 *
	 * @return La liste des sous-rubriques de cette rubrique tous niveaux confondus
	 */
	public Collection<InfosRubriques> getListeSousRubriquesTousNiveaux() {
		if (listeSousRubriquesTousNiveaux == null) {
			listeSousRubriquesTousNiveaux = new ArrayList<InfosRubriques>();
			final Iterator<InfosRubriques> listeSousRubriquesIt = getListeSousRubriques().iterator();
			InfosRubriques sousRubrique = null;
			while (listeSousRubriquesIt.hasNext()) {
				sousRubrique = listeSousRubriquesIt.next();
				listeSousRubriquesTousNiveaux.addAll(sousRubrique.getListeSousRubriquesTousNiveaux()); // récursif
				listeSousRubriquesTousNiveaux.add(sousRubrique);
			}
		}
		// JB & RP 20060131 : on ne doit pas partager la meme collection pour l'ensemble des threads
		return new ArrayList<InfosRubriques>(listeSousRubriquesTousNiveaux);
	}

	/**
	 * Renvoie la liste des sous-rubriques (sous forme d'instances de rubriques) d'une rubrique, ordonnée par l'ordre des rubriques + application de la DSI Si InfoRubrique vide
	 * (new InfosRubrique()) on va chercher les rubriques principales.
	 *
	 * @param _ctx
	 *            the _ctx
	 *
	 * @return : Vector : liste des rubriques filles. Dans l'instance, <b>seul le code et l'intitule sont renseignes.</b>
	 *
	 *
	 */
	public Collection<InfosRubriques> getListeSousRubriquesFront(final ContexteUniv _ctx) {
		final ArrayList<InfosRubriques> lSousRubriquesFront = new ArrayList<InfosRubriques>();
		final Collection<InfosRubriques> lSousRubriques = getListeSousRubriques();
		final Iterator<InfosRubriques> itSousRubriques = lSousRubriques.iterator();
		InfosRubriques sousRubrique = null;
		while (itSousRubriques.hasNext()) {
			sousRubrique = itSousRubriques.next();
			if (Rubrique.controlerRestrictionRubrique(_ctx, sousRubrique.getCode()) && !sousRubrique.getCategorie().toUpperCase().equals("HIDDEN")) {
				lSousRubriquesFront.add(sousRubrique);
			}
		}
		return lSousRubriquesFront;
	}

	/**
	 * Ajoute une sous-rubrique à la liste des sous-rubriques (triée sur l'ordre).
	 *
	 * @param sousRubrique
	 *            Une sous-rubrique
	 */
	public void addSousRubrique(final InfosRubriques sousRubrique) {
		String ordreSousRub = sousRubrique.getOrdre();
		for (int i = ordreSousRub.length(); i < 4; i++) {
			ordreSousRub = "0" + ordreSousRub;
		}
		ordreSousRub += "_" + sousRubrique.getOnglet() + "_" + sousRubrique.getCode(); // au cas ou plusieurs rubs ont le meme code
		listeSousRubriques.put(ordreSousRub, sousRubrique);
	}

	/**
	 * Rattache la rubrique à sa rubrique mère et met à jour son niveau et la liste des sous-rubriques de la rubrique mère.
	 *
	 * @param rubrique
	 *            La rubrique mère
	 */
	public void rattacheA(final InfosRubriques rubrique) {
		this.rubriqueMere = rubrique;
		this.rubriqueMere.addSousRubrique(this);
		updateNiveau();
	}

	/**
	 * Met a jour le niveau de la rubrique et de ses sous-rubriques en fonction du niveau de la rubrique mère.
	 */
	private void updateNiveau() {
		// Mise a jour du niveau de la rubrique
		niveau = this.rubriqueMere.getNiveau() + 1;
		// Mise a jour du niveau des sous-rubriques
		for (InfosRubriques infosRubriques : getListeSousRubriques()) {
			infosRubriques.updateNiveau(); // récursif
		}
	}

	/**
	 * Vérifie si cette rubrique contient la rubrique passée en paramètre (on teste si la rubrique passée en paramètre ou une de ses rubriques mère ne serait pas par hasard égale à
	 * cette rubrique).
	 *
	 * @param rubrique
	 *            the rubrique
	 *
	 * @return true, if contains
	 */
	public boolean contains(InfosRubriques rubrique) {
		boolean inFamilia = false;
		while (rubrique != null && !inFamilia) {
			if (rubrique.equals(this)) {
				inFamilia = true;
			} else {
				rubrique = rubrique.getRubriqueMere(); // on remonte
			}
		}
		return inFamilia;
	}

	/**
	 * Gets the categorie.
	 *
	 * @return Returns the categorie.
	 */
	public String getCategorie() {
		return categorie;
	}

	public String getPageAccueil() {
		return pageAccueil;
	}

	/**
	 * @deprecated l'url n'est pas mise à jour lors de la modification du média
	 */
	@Deprecated
	public String getUrlPicto() {
		return urlPicto;
	}

	/**
	 * @deprecated l'url n'est pas mise à jour lors de la modification du média
	 * @param urlPicto
	 */
	@Deprecated
	public void setUrlPicto(final String urlPicto) {
		this.urlPicto = urlPicto;
	}

	public long getIdMediaBandeau() {
		return idMediaBandeau;
	}

	public void setIdMediaBandeau(long idMediaBandeau) {
		this.idMediaBandeau = idMediaBandeau;
	}

	public long getIdMediaPicto() {
		return idMediaPicto;
	}

	public void setIdMediaPicto(long idMediaPicto) {
		this.idMediaPicto = idMediaPicto;
	}
}
