/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.List;
import java.util.Locale;
import java.util.StringTokenizer;
import java.util.Vector;

import org.apache.commons.lang3.StringUtils;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.core.LangueUtil;
import com.jsbsoft.jtf.database.OMContext;
import com.kportal.core.config.MessageHelper;
import com.univ.objetspartages.cache.CacheLibelleManager;
import com.univ.objetspartages.sgbd.LibelleDB;
import com.univ.utils.ContexteDao;
import com.univ.utils.ContexteUniv;
import com.univ.utils.ContexteUtil;
import com.univ.utils.sql.RequeteSQL;
import com.univ.utils.sql.clause.ClauseOrderBy;
import com.univ.utils.sql.clause.ClauseOrderBy.SensDeTri;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;

/**
 * The Class Libelle.
 */
public class Libelle extends LibelleDB implements Cloneable {

	public void init() {
		setIdLibelle((long) 0);
		setType("");
		setCode("");
		setLibelle("");
		setLangue("");
	}

	private Collection<String> languesPossiblesPourTraduction = new ArrayList<>();

	private boolean isLibelleSupprimable = Boolean.TRUE;

	/**
	 * Gets the libelle affichable.
	 *
	 * @return the libelle affichable
	 */
	public String getLibelleAffichable() {
		if (getLibelle().startsWith("[")) {
			final int i = getLibelle().indexOf("]") + 1;
			if (i != 0 && i != getLibelle().length()) {
				return getLibelle().substring(i);
			}
		}
		return getLibelle();
	}

	/**
	 * Gets the code site.
	 *
	 * @return the code site
	 */
	public String getCodeSiteLibelle() {
		if (getLibelle().startsWith("[")) {
			final int i = getLibelle().indexOf("]") + 1;
			if (i != 0 && i != getLibelle().length()) {
				return getLibelle().substring(1, i - 1);
			}
		}
		return "";
	}

	/**
	 * Ajoute (ou modifie) un libellé
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _typeLibelle
	 *            the _type libelle
	 * @param _code
	 *            the _code
	 * @param _libelle
	 *            the _libelle
	 * @param _langue
	 *            the _langue
	 *
	 * @return the string
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String addLibelle(final OMContext _ctx, final String _typeLibelle, final String _code, final String _libelle, final String _langue) throws Exception {
		final Libelle libelle = new Libelle();
		libelle.setCtx(_ctx);
		final ClauseWhere typeCodeLangue = new ClauseWhere(ConditionHelper.egalVarchar("TYPE", _typeLibelle));
		typeCodeLangue.and(ConditionHelper.egalVarchar("CODE", _code));
		typeCodeLangue.and(ConditionHelper.egalVarchar("LANGUE", _langue));
		if (libelle.select(typeCodeLangue.formaterSQL()) > 0) {
			/* Modification */
			while (libelle.nextItem()) {
				libelle.setLibelle(_libelle);
				libelle.update();
			}
		} else {
			/* Ajout */
			libelle.init();
			libelle.setCode(_code);
			libelle.setLibelle(_libelle);
			libelle.setLangue(_langue);
			libelle.setType(_typeLibelle);
			libelle.add();
		}
		return StringUtils.EMPTY;
	}

	@Deprecated
	public static List<String> getLibelleSousFormeDeListe(final OMContext _ctx, final String _typeLibelle, final String _codes, final Locale _locale) {
		return getLibelleSousFormeDeListe(_typeLibelle, _codes, _locale);
	}

	/**
	 * Renvoie les libellés sous forme de liste pour pouvoir les gérer de manière plus souple.
	 *
	 * @param _codes
	 * @param _typeLibelle
	 * @param _locale
	 */
	public static List<String> getLibelleSousFormeDeListe(final String _typeLibelle, final String _codes, final Locale _locale) {
		String code = "";
		final List<String> libelles = new ArrayList<>();
		if (_codes != null && !_codes.equals("") && !_codes.equals("0000")) {
			final StringTokenizer st = new StringTokenizer(_codes, ";");
			int i = 0;
			while (st.hasMoreTokens()) {
				code = st.nextToken();
				InfosLibelle infosLibelle = renvoyerItemLibelle(_typeLibelle, code, Integer.toString(LangueUtil.getIndiceLocale(_locale)));
				if (infosLibelle != null) {
					if (ContexteUtil.getContexteUniv() != null) {
						libelles.add(i, infosLibelle.getLibelleAffichable());
					} else {
						libelles.add(i, infosLibelle.getLibelle());
					}
				} else {
					libelles.add(i, String.format("&lt;%s&gt;", MessageHelper.getCoreMessage("LIBELLE.INCONNU")));
				}
				++i;
			}
		}
		return libelles;
	}

	@Deprecated
	public static String getLibelle(final OMContext _ctx, final String _typeLibelle, final String _codes, final Locale _locale) {
		return getLibelle(_typeLibelle, _codes, _locale);
	}

	@Deprecated
	public static String getLibelle(final OMContext _ctx, final String _typeLibelle, final String _codes, final Locale _locale, final String _separateur) {
		return getLibelle(_typeLibelle, _codes, _locale);
	}

	/**
	 * Gets the libelle. Renvoie un libelle (ou une liste de libellés) Si plusieurs libellés, et ctx = ContexteUniv (front-office), le séparateur est ", " Si plusieurs libellés et
	 * ctx != ContexteUniv, le séparateur est ";" Peut être affiché directement dans une Combo
	 *
	 * @param _typeLibelle
	 *            the _type libelle
	 * @param _codes
	 *            the _codes
	 * @param _locale
	 *            the _locale
	 *
	 * @return the libelle
	 *
	 */
	public static String getLibelle(final String _typeLibelle, final String _codes, final Locale _locale) {
		String res = "";
		if (_codes != null && !_codes.equals("") && !_codes.equals("0000")) {
			final StringTokenizer st = new StringTokenizer(_codes, ";");
			ContexteUniv ctx = ContexteUtil.getContexteUniv();
			while (st.hasMoreTokens()) {
				String code = st.nextToken();
				if (res.length() > 0) {
					if (ctx != null) {
						res += ", ";
					} else {
						res += ";";
					}
				}
				InfosLibelle infosLibelle = renvoyerItemLibelle(_typeLibelle, code, Integer.toString(LangueUtil.getIndiceLocale(_locale)));
				if (infosLibelle != null) {
					if (ctx != null) {
						res += infosLibelle.getLibelleAffichable();
					} else {
						res += infosLibelle.getLibelle();
					}
				} else {
					res += String.format("&lt;%s&gt;", MessageHelper.getCoreMessage("LIBELLE.INCONNU"));
				}
			}
		}
		return res;
	}

	@Deprecated
	public static Hashtable<String, String> getListe(final OMContext _ctx, final String _typeLibelle, Locale _locale) throws Exception {
		return getListe(_typeLibelle, _locale);
	}

	/**
	 * Renvoie la liste de tous les libelles Peut être affiché directement dans une Combo
	 *
	 * @param _typeLibelle
	 *            the _type libelle
	 * @param _locale
	 *            the _locale
	 *
	 * @return the liste
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static Hashtable<String, String> getListe(final String _typeLibelle, Locale _locale) throws Exception {
		final Hashtable<String, String> h = new Hashtable<>();
		if (_locale == null) {
			_locale = LangueUtil.getDefaultLocale();
		}
		try (ContexteDao ctx = new ContexteDao()) {
			final Libelle libelle = new Libelle();
			libelle.setCtx(ctx);
			final ClauseWhere typeLangue = new ClauseWhere(ConditionHelper.egalVarchar("TYPE", _typeLibelle));
			typeLangue.and(ConditionHelper.egalVarchar("LANGUE", String.valueOf(LangueUtil.getIndiceLocale(_locale))));
			if (libelle.select(typeLangue.formaterSQL()) > 0) {
				while (libelle.nextItem()) {
					if (ContexteUtil.getContexteUniv() != null) {
						final String codeSite = ContexteUtil.getContexteUniv().getInfosSite().getAlias();
						final String codeSiteLibelle = libelle.getCodeSiteLibelle();
						if (codeSiteLibelle.equals("") || codeSiteLibelle.equalsIgnoreCase(codeSite)) {
							h.put(libelle.getCode(), libelle.getLibelleAffichable());
						}
					} else {
						h.put(libelle.getCode(), libelle.getLibelle());
					}
				}
			}
		}
		return h;
	}

	/**
	 * Renvoie une liste triée de libellés
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _typeLibelle
	 *            the _type libelle
	 * @param _locale
	 *            the _locale
	 *
	 * @return the liste triee par libelle
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static Vector<Libelle> getListeTrieeParLibelle(final OMContext _ctx, final String _typeLibelle, final Locale _locale) throws Exception {
		final Libelle libelle = new Libelle();
		libelle.setCtx(_ctx);
		final Vector<Libelle> v = new Vector<>();
		final ClauseWhere typeLangue = new ClauseWhere(ConditionHelper.egalVarchar("TYPE", _typeLibelle));
		typeLangue.and(ConditionHelper.egalVarchar("LANGUE", String.valueOf(LangueUtil.getIndiceLocale(_locale))));
		final ClauseOrderBy orderByLibelle = new ClauseOrderBy("LIBELLE", SensDeTri.ASC);
		final RequeteSQL requeteLibelle = new RequeteSQL();
		requeteLibelle.where(typeLangue).orderBy(orderByLibelle);
		if (libelle.select(requeteLibelle.formaterRequete()) > 0) {
			while (libelle.nextItem()) {
				v.add(libelle.clone());
			}
		}
		return v;
	}

	/**
	 * Renvoie la liste de tous les libelles matchant un token SQL (ex :%math%) Peut être affiché directement dans une Combo
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _typeLibelle
	 *            the _type libelle
	 * @param token
	 *            the token
	 *
	 * @return the liste par token
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static Hashtable<String, String> getListeParToken(final OMContext _ctx, final String _typeLibelle, final String token) throws Exception {
		final Libelle libelle = new Libelle();
		libelle.setCtx(_ctx);
		final Hashtable<String, String> h = new Hashtable<>();
		//    FIXME les token % etc sont présent dans l'attribut on peut pas échapper le champ...
		if (libelle.select("WHERE TYPE='" + _typeLibelle + "' AND LIBELLE  LIKE '" + token + "'") > 0) {
			while (libelle.nextItem()) {
				h.put(libelle.getCode(), libelle.getLibelle());
			}
		}
		return h;
	}

	public static Hashtable<String, String> getListeTypesLibelles() {
		final CacheLibelleManager cache = (CacheLibelleManager) ApplicationContextManager.getCoreContextBean(CacheLibelleManager.ID_BEAN);
		return cache.getListeTypesLibelles();
	}

	/**
	 * Récupération d'un libelle stocké en mémoire.
	 *
	 * @param type
	 *            the type
	 * @param code
	 *            the code
	 * @param langue
	 *            the langue
	 *
	 * @return the infos libelle
	 *
	 */
	public static InfosLibelle renvoyerItemLibelle(final String type, final String code, final String langue) {
		final CacheLibelleManager cache = (CacheLibelleManager) ApplicationContextManager.getCoreContextBean(CacheLibelleManager.ID_BEAN);
		return cache.getListeInfosLibelles().get(type + code.toLowerCase() + langue);
	}

	@Override
	public Libelle clone() throws CloneNotSupportedException {
		return (Libelle) super.clone();
	}

	public Collection<String> getLanguesPossiblesPourTraduction() {
		return languesPossiblesPourTraduction;
	}

	public void setLanguesPossiblesPourTraduction(final Collection<String> languesPossiblesPourTraduction) {
		this.languesPossiblesPourTraduction = languesPossiblesPourTraduction;
	}

	public boolean isLibelleSupprimable() {
		return isLibelleSupprimable;
	}

	public void setLibelleSupprimable(final boolean isLibelleSupprimable) {
		this.isLibelleSupprimable = isLibelleSupprimable;
	}
}
