/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import org.apache.commons.lang3.StringUtils;

import com.univ.objetspartages.sgbd.MailingDB;
import com.univ.utils.sql.Operateur;
import com.univ.utils.sql.RequeteSQL;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.utils.sql.operande.TypeOperande;

/**
 * The Class Mailing.
 */
public class Mailing extends MailingDB {

	/**
	 * Instantiates a new mailing.
	 */
	public Mailing() {
		super();
	}

	/**
	 * Inits the.
	 */
	public void init() {
		setIdMailing(new Long(0));
		setEmail("");
		setNomUtilisateur("");
		setReplyTo("");
		setObjet("");
		setTypeMime("");
		setMessage("");
		setFichier("");
		setNomFichier("");
		setEtat("");
		setErrorMessage("");
		setIdNewsletter("");
		setCodeUtilisateur("");
		setDatePurge(new java.sql.Date(0));
		setDateCreation(new java.sql.Timestamp(System.currentTimeMillis()));
		setType("");
	}

	/**
	 * Sélectionne le mailing envoyé a un utilisateur.
	 * 
	 * @param emailUtilisateur
	 *            email de l'utilisateur
	 * @param idNewsletter
	 *            identifiant de la newsletter
	 * 
	 * @return the int
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public int select(final String emailUtilisateur, final String idNewsletter) throws Exception {
		final ClauseWhere whereMailAndIdNewsletter = new ClauseWhere();
		if (StringUtils.isNotBlank(emailUtilisateur)) {
			whereMailAndIdNewsletter.setPremiereCondition(ConditionHelper.egalVarchar("EMAIL", emailUtilisateur));
		}
		whereMailAndIdNewsletter.and(ConditionHelper.egalVarchar("ID_NEWSLETTER", idNewsletter));
		return select(whereMailAndIdNewsletter.formaterSQL());
	}

	/**
	 * Sélectionne les mailings dont la date de purge est dépassée.
	 * 
	 * @param date
	 *            the date
	 * 
	 * @return the int
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public int selectParDate(final java.sql.Date date) throws Exception {
		final ClauseWhere whereDate = new ClauseWhere(ConditionHelper.genericConditionSurColonne("DATE_PURGE", date, TypeOperande.DATE, Operateur.GREATER_EQUALS));
		return select(whereDate.formaterSQL());
	}

	/**
	 * Sélectionne les mailing par titre,type et date de création
	 * 
	 * @param objet
	 *            L' objet du mailing recherché
	 * @param type
	 *            Le type du mailing rechderché
	 * @param dateDebut
	 *            La date minimum de création du mailing
	 * @param dateFin
	 *            La date maximum de création du mailing
	 * @param publicInterne
	 *            vrai le public est interne, faux si il est externe NE SERT PLUS
	 * @return Les mailings correspondant aux critères
	 * @throws Exception
	 */
	public int selectParObjetTypeDateCreationEtPublic(final String objet, final String type, final java.sql.Date dateDebut, final java.sql.Date dateFin, final boolean publicInterne)
		throws Exception {
		final RequeteSQL requeteSelect = new RequeteSQL();
		final ClauseWhere whereTypeDatePublic = new ClauseWhere();
		if (StringUtils.isNotEmpty(objet)) {
			whereTypeDatePublic.setPremiereCondition(ConditionHelper.like("T1.OBJET", objet, "%", "%"));
		}
		if (StringUtils.isNotEmpty(type) && !"0000".equals(type)) {
			whereTypeDatePublic.and(ConditionHelper.egalVarchar("T1.TYPE", type));
		}
		if (dateDebut != null) {
			whereTypeDatePublic.and(ConditionHelper.greaterEquals("T1.DATE_CREATION", dateDebut, TypeOperande.DATE));
		}
		if (dateFin != null) {
			whereTypeDatePublic.and(ConditionHelper.lessEquals("T1.DATE_CREATION", "('" + dateFin + "' + INTERVAL 1 DAY - INTERVAL 1 SECOND)", TypeOperande.NON_ECHAPABLE));
		}
		// on ne requete que les envois bien envoyés
		whereTypeDatePublic.and(ConditionHelper.egalVarchar("T1.ETAT", "SENT"));
		// les envois html
		whereTypeDatePublic.and(ConditionHelper.egalVarchar("T1.TYPE_MIME", "0"));
		// les archives publiques, le boolean publicInterne ne sert plus
		whereTypeDatePublic.and(ConditionHelper.like("T1.EMAIL", "[", "", "%"));
		requeteSelect.where(whereTypeDatePublic);
		return select(requeteSelect.formaterRequete());
	}
}
