/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 *
 */
package com.univ.objetspartages.om;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.sql.Date;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Vector;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.Formateur;
import com.jsbsoft.jtf.core.InfoBean;
import com.jsbsoft.jtf.core.LangueUtil;
import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.kportal.core.webapp.WebAppUtil;
import com.univ.mediatheque.CacheMediaManager;
import com.univ.mediatheque.Mediatheque;
import com.univ.mediatheque.utils.MediathequeHelper;
import com.univ.objetspartages.bean.MediaBean;
import com.univ.objetspartages.processus.SaisieMedia;
import com.univ.objetspartages.sgbd.MediaDB;
import com.univ.utils.Chaine;
import com.univ.utils.ContexteDao;
import com.univ.utils.FileUtil;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.condition.Condition;
import com.univ.utils.sql.condition.ConditionList;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.utils.sql.operande.TypeOperande;

/**
 * Classe representant un objet media.
 */
public class Media extends MediaDB implements RessourceUniv, Cloneable {

    private static final long serialVersionUID = 4562627951365613065L;

    private static final Logger LOG = LoggerFactory.getLogger(Media.class);

    /** The langue. */
    private int langue = 0;

    private String generateName = "";

    /**
     * @return the generateName
     */
    @Override
    public String getGenerateName() {
        return generateName;
    }

    /**
     * @param generateName
     *            the generateName to set
     */
    @Override
    public void setGenerateName(final String generateName) {
        this.generateName = generateName;
    }

    /**
     * Initialise l'objet metier.
     */
    public void init() {
        setIdMedia((long) 0);
        setTitre("");
        setLegende("");
        setDescription("");
        setFormat("");
        setSource("");
        setIsMutualise("");
        setAuteur("");
        setCopyright("");
        setTypeRessource("");
        setTypeMedia("");
        setCodeRubrique("");
        setCodeRattachement("");
        setUrl("");
        setUrlVignette("");
        setPoids(0);
        setCodeRedacteur("");
        setDateCreation(new Date(0));
        setThematique("");
        setMetaKeywords("");
        setSpecificData("");
        setTraductionData("");
        setAccessibilityData("");
    }

    /**
     * Renvoie le libelle a afficher.
     *
     * @return the libelle affichable
     */
    public String getLibelleAffichable() {
        String res = getTitre();
        if (res.length() == 0) {
            res = getSource();
        }
        return res;
    }

	/* (non-Javadoc)
     * @see com.univ.objetspartages.om.RessourceUniv#getUrlAbsolue()
	 */

    /**
     * Renvoie le libelle pour l'attribut typeRessource.
     *
     * @return the libelle type ressource
     *
     * @throws Exception
     *             the exception
     */
    public String getLibelleTypeRessource() throws Exception {
        return Mediatheque.getInstance().getRessourceByExtension(getExtension()).getLibelle();
    }

    /**
     * Renvoie le libelle pour l'attribut typeMedia.
     *
     * @return the libelle type media
     *
     * @throws Exception
     *             the exception
     */
    public String getLibelleTypeMedia() throws Exception {
        return Libelle.getLibelle("201", getTypeMedia(), LangueUtil.getDefaultLocale());
    }

    /**
     * Renvoie le libelle pour l'attribut thematique.
     *
     * @return the libelle thematique
     *
     * @throws Exception
     *             the exception
     */
    public String getLibelleThematique() throws Exception {
        return Libelle.getLibelle("04", getThematique(), LangueUtil.getDefaultLocale());
    }

    /**
     * Selection d'une instance de l'objet Media a partir de l'ensemble des criteres.
     *
     * @param idMedia
     *            the id media
     * @param titre
     *            the titre
     * @param legende
     *            the legende
     * @param description
     *            the description
     * @param auteur
     *            the auteur
     * @param typeRessource
     *            the type ressource
     * @param typeMedia
     *            the type media
     * @param codeRattachement
     *            the code rattachement
     * @param codeRubrique
     *            the code rubrique
     * @param codeRedacteur
     *            the code redacteur
     * @param metaKeywords
     *            the meta keywords
     * @param dateCreation
     *            the date creation
     * @param _mode
     *            the _mode
     * @param copyright
     *            the copyright
     * @param thematique
     *            the thematique
     *
     * @return String la requête
     *
     * @throws Exception
     *             the exception
     */
    private ClauseWhere preparerRequete(final Long idMedia, final String titre, final String legende, final String description, final String copyright, final String auteur, final String typeRessource, final String typeMedia, final String thematique, final String codeRattachement, final String codeRubrique, final String codeRedacteur, final String metaKeywords, final Date dateCreation, final String _mode) throws Exception {
        final ClauseWhere where = new ClauseWhere();
        //CQCB? -> TODO : type de donnees non pris en compte, controler le parametre
        if (!idMedia.equals(Long.valueOf(0))) {
            where.setPremiereCondition(ConditionHelper.egal("T1.ID_MEDIA", idMedia, TypeOperande.LONG));
        }
        if (StringUtils.isNotEmpty(titre)) {
            final ConditionList titreEtSource = new ConditionList(ConditionHelper.rechercheMots("T1.TITRE", titre));
            titreEtSource.or(ConditionHelper.rechercheMots("T1.SOURCE", titre));
            where.and(titreEtSource);
        }
        if (StringUtils.isNotEmpty(legende)) {
            where.and(ConditionHelper.rechercheMots("T1.LEGENDE", legende));
        }
        if (StringUtils.isNotEmpty(description)) {
            where.and(ConditionHelper.rechercheMots("T1.DESCRIPTION", description));
        }
        if (StringUtils.isNotEmpty(copyright)) {
            where.and(ConditionHelper.rechercheMots("T1.COPYRIGHT", copyright));
        }
        if (StringUtils.isNotEmpty(auteur)) {
            where.and(ConditionHelper.rechercheMots("T1.AUTEUR", auteur));
        }
        if (StringUtils.isNotEmpty(typeRessource) && !"0000".equals(typeRessource)) {
            where.and(ConditionHelper.egalVarchar("T1.TYPE_RESSOURCE", typeRessource));
        } else {
            where.and(ConditionHelper.notEgal("T1.TYPE_RESSOURCE", "", TypeOperande.TEXTE));
        }
        if (StringUtils.isNotEmpty(typeMedia) && !"0000".equals(typeMedia)) {
            where.and(ConditionHelper.egalVarchar("T1.TYPE_MEDIA", typeMedia));
        }
        if (StringUtils.isNotEmpty(thematique) && !"0000".equals(thematique)) {
            where.and(ConditionHelper.egalVarchar("T1.THEMATIQUE", thematique));
        }
        if (StringUtils.isNotEmpty(codeRattachement)) {
            where.and(ConditionHelper.egalVarchar("T1.CODE_RATTACHEMENT", codeRattachement));
        }
        if (StringUtils.isNotEmpty(codeRubrique)) {
            where.and(ConditionHelper.getConditionRubrique("CODE_RUBRIQUE", codeRubrique));
        }
        if (StringUtils.isNotEmpty(codeRedacteur)) {
            where.and(ConditionHelper.egalVarchar("T1.CODE_REDACTEUR", codeRedacteur));
        }
        if (StringUtils.isNotEmpty(metaKeywords)) {
            where.and(ConditionHelper.rechercheMots("META_KEYWORDS", metaKeywords));
        }
        if (Formateur.estSaisie(dateCreation)) {
            where.and(ConditionHelper.greaterEquals("T1.DATE_CREATION", dateCreation, TypeOperande.DATE_ET_HEURE));
        }
        where.and(ConditionHelper.egalVarchar("T1.IS_MUTUALISE", "0"));
        where.and(conditionPerimetreMedia(ctx, _mode));
        return where;
    }

    /**
     * Preparer requete.
     *
     * @param idMedia
     *            the id media
     * @param titre
     *            the titre
     * @param legende
     *            the legende
     * @param description
     *            the description
     * @param copyright
     *            the copyright
     * @param auteur
     *            the auteur
     * @param typeRessource
     *            the type ressource
     * @param typeMedia
     *            the type media
     * @param thematique
     *            the thematique
     * @param codeRattachement
     *            the code rattachement
     * @param codeRubrique
     *            the code rubrique
     * @param codeRedacteur
     *            the code redacteur
     * @param metaKeywords
     *            the meta keywords
     * @param dateCreation
     *            the date creation
     * @param _mode
     *            the _mode
     * @param poidsMinimum
     *            the poids minimum
     * @param poidsMaximum
     *            the poids maximum
     *
     * @return the string
     *
     * @throws Exception
     *             the exception
     */
    public ClauseWhere preparerRequete(final Long idMedia, final String titre, final String legende, final String description, final String copyright, final String auteur, final String typeRessource, final String typeMedia, final String thematique, final String codeRattachement, final String codeRubrique, final String codeRedacteur, final String metaKeywords, final Date dateCreation, final String _mode, final String poidsMinimum, final String poidsMaximum) throws Exception {
        final ClauseWhere where = preparerRequete(idMedia, titre, legende, description, copyright, auteur, typeRessource, typeMedia, thematique, codeRattachement, codeRubrique, codeRedacteur, metaKeywords, dateCreation, _mode);
        if (StringUtils.isNotBlank(poidsMinimum)) {
            where.and(ConditionHelper.greaterEquals("T1.POIDS", poidsMinimum, TypeOperande.VARCHAR));
        }
        if (StringUtils.isNotBlank(poidsMaximum)) {
            where.and(ConditionHelper.lessEquals("T1.POIDS", poidsMaximum, TypeOperande.VARCHAR));
        }
        return where;
    }

    public static ConditionList conditionPerimetreMedia(final OMContext ctx, final String mode) throws Exception {
        final ConditionList conditionBOMedia = new ConditionList();
        if (possedeDesAutorisations(ctx)) {
            Boolean accesTousPerimetre = Boolean.FALSE;
            final AutorisationBean autorisations = (AutorisationBean) ctx.getDatas().get("AUTORISATIONS");
            final Hashtable<String, Vector<Perimetre>> permisssions = autorisations.getListePermissions();
            for (final Entry<String, Vector<Perimetre>> clePermission : permisssions.entrySet()) {
                final PermissionBean permission = new PermissionBean(clePermission.getKey());
                if (isValidePermission(permission, mode)) {
                    final Vector<Perimetre> lesPerimetres = clePermission.getValue();
                    for (final Perimetre perimetre : lesPerimetres) {
                        final ConditionList conditionPerimetre = new ConditionList();
                        accesTousPerimetre = isAccesTousPerimetre(perimetre);
                        if (!accesTousPerimetre) {
                            conditionPerimetre.setPremiereCondtion(traiterConditionRubrique(perimetre));
                            conditionPerimetre.and(traiterConditionStructure(perimetre));
                        }
                        conditionBOMedia.or(conditionPerimetre);
                    }
                }
            }
            if (!accesTousPerimetre) {
                conditionBOMedia.or(ConditionHelper.egalVarchar("CODE_REDACTEUR", autorisations.getCode()));
            }
        }
        return conditionBOMedia;
    }

    private static Condition traiterConditionStructure(final Perimetre perimetre) throws Exception {
        Condition conditionsStructures = new ConditionList();
        final String codeStructure = perimetre.getCodeStructure();
        if (StringUtils.isNotEmpty(codeStructure)) {
            conditionsStructures = ConditionHelper.getConditionStructure("CODE_RATTACHEMENT", codeStructure);
        }
        return conditionsStructures;
    }

    private static Condition traiterConditionRubrique(final Perimetre perimetre) throws Exception {
        Condition conditionsRubriques = new ConditionList();
        final String codeRubrique = perimetre.getCodeRubrique();
        if (StringUtils.isNotEmpty(codeRubrique)) {
            conditionsRubriques = ConditionHelper.getConditionRubrique("CODE_RUBRIQUE", codeRubrique);
        }
        return conditionsRubriques;
    }

    private static boolean isAccesTousPerimetre(final Perimetre perimetre) {
        return perimetre.getCodeRubrique().length() == 0 && perimetre.getCodeStructure().length() == 0;
    }

    private static boolean possedeDesAutorisations(final OMContext ctx) {
        return ctx.getDatas() != null && ctx.getDatas().get("AUTORISATIONS") != null;
    }

    private static boolean isValidePermission(final PermissionBean permission, final String mode) {
        return (permission.getType().equals("TECH")) && (((SaisieMedia.MODE_ADMINISTRATION.equals(mode)) && (permission.getObjet().equals("pho")) && (permission.getAction().equals("M") || permission.getAction().equals("S"))) || (!(SaisieMedia.MODE_ADMINISTRATION.equals(mode)) && (permission.getObjet().equals("acp"))));
    }

    /* (non-Javadoc)
     * @see com.univ.objetspartages.om.RessourceUniv#getUrlAbsolue()
     */
    @Override
    public String getUrlAbsolue() {
        if (this.getIdMedia()!=null && !this.getIdMedia().equals(0L)) {
            String sUrl = this.getUrl();
            if (StringUtils.isNotBlank(sUrl) && isLocal()) {
                sUrl = MediathequeHelper.getAbsolutePath();
                String sServerPath = new File(WebAppUtil.getWebInfPath()).getAbsolutePath();
                if (!sServerPath.endsWith(File.separator)) {
                    sServerPath += File.separator;
                }
                if (sUrl.indexOf(sServerPath) == 0) {
                    if (sServerPath.endsWith("/")) {
                        sUrl = sUrl.substring(sServerPath.length());
                    } else {
                        sUrl = "/" + sUrl.substring(sServerPath.length());
                    }
                } else {
                    sUrl = MediathequeHelper.getDefaultRelativePath();
                    if (!StringUtils.startsWith(sUrl, "/")) {
                        sUrl = "/" + sUrl;
                    }
                }
                if (StringUtils.isNotBlank(this.getTypeRessource())) {
                    sUrl += "/" + this.getTypeRessource().toLowerCase();
                }
                sUrl += "/" + this.getUrl();
            }
            if (StringUtils.startsWith(sUrl, "/") || !isLocal()) {
                return sUrl;
            } else {
                return "/" + sUrl;
            }
        }
        return StringUtils.EMPTY;
    }

    public static String getPathAbsoluFichierTmp(final String urlTmp) {
        if (StringUtils.isNotEmpty(urlTmp)) {
            return MediathequeHelper.getAbsolutePath() + urlTmp;
        } else {
            return StringUtils.EMPTY;
        }
    }

    public boolean isPublic() {
        return !getIsMutualise().equals(Mediatheque.ETAT_NON_MUTUALISE_NON_PUBLIC);
    }

    /* (non-Javadoc)
     * @see com.univ.objetspartages.om.RessourceUniv#getUrlAbsolue()
     */
    @Override
    public String getPathAbsolu() {
        String sUrl = getUrl();
        if (sUrl.length() > 0 && isLocal()) {
            sUrl = MediathequeHelper.getAbsolutePath();
            if (StringUtils.isNotBlank(getTypeRessource())) {
                sUrl += "/" + getTypeRessource().toLowerCase();
            }
            sUrl += "/" + getUrl();
        }
        return sUrl;
    }

    /* (non-Javadoc)
     * @see com.univ.objetspartages.om.RessourceUniv#getUrlAbsolue()
     */
    @Override
    public String getPathVignetteAbsolu() {
        String sUrl = getUrlVignette();
        if (sUrl.length() > 0) {
            sUrl = MediathequeHelper.getAbsolutePath();
            if (getTypeRessource().length() > 0) {
                sUrl += "/" + getTypeRessource().toLowerCase();
            }
            sUrl += "/" + getUrlVignette();
        }
        return sUrl;
    }

	/* (non-Javadoc)
     * @see com.univ.objetspartages.om.RessourceUniv#getUrlAbsolue()
	 */

    /**
     * Gets the url vignette absolue.
     *
     * @return the url vignette absolue
     */
    public String getUrlVignetteAbsolue() {
        String sUrl = getUrlVignette();
        if (sUrl.length() > 0) {
            sUrl = MediathequeHelper.getAbsolutePath();
            final String sServerPath = WebAppUtil.getAbsolutePath();
            if (sUrl.indexOf(sServerPath) == 0) {
                if (sServerPath.endsWith("/")) {
                    sUrl = "/" + sUrl.substring(sServerPath.length());
                } else {
                    sUrl = "/" + sUrl.substring(sServerPath.length() + 1);
                }
            } else {
                sUrl = MediathequeHelper.getDefaultRelativePath();
            }
            if (getTypeRessource().length() > 0) {
                sUrl += "/" + getTypeRessource().toLowerCase();
            }
            sUrl += "/" + getUrlVignette();
        }
        return sUrl;
    }

    /**
     * Checks if is local.
     *
     * @return true, if is local
     */
    public boolean isLocal() {
        return !getUrl().contains("/");
    }

    /**
     * Gets the specific data.
     *
     * @param propertie
     *            the propertie
     *
     * @return the specific data
     */
    public String getSpecificData(final String propertie) {
        String sRes = "";
        final Properties properties = new Properties();
        try {
            properties.load(new ByteArrayInputStream(getSpecificData().getBytes()));
        } catch (final IOException e) {
            LOG.error("erreur de la lecture des properties du media", e);
        }
        if (properties.get(propertie) != null) {
            sRes = (String) properties.get(propertie);
        }
        return sRes;
    }

    /**
     * Gets the traduction data.
     *
     * @param propertie
     *            the propertie
     *
     * @return the specific data
     */
    public String getTraductionData(final String propertie) {
        String sRes = "";
        final Properties properties = new Properties();
        try {
            properties.load(new ByteArrayInputStream(getTraductionData().getBytes()));
        } catch (final IOException e) {
            LOG.error("erreur de la lecture des properties du media", e);
        }
        if (properties.get(propertie) != null) {
            sRes = (String) properties.get(propertie);
        }
        return sRes;
    }

    /**
     * Gets the specific data.
     *
     * @return the specific data
     */
    public String getSpecificDataAsString() {
        String sRes = "";
        final Properties properties = new Properties();
        try {
            properties.load(new ByteArrayInputStream(getSpecificData().getBytes()));
        } catch (final IOException e) {
            LOG.error("erreur de la lecture des properties du media", e);
        }
        final Enumeration<Object> e = properties.keys();
        while (e.hasMoreElements()) {
            final String propertie = (String) e.nextElement();
            final String value = properties.getProperty(propertie);
            if (value.length() > 0 && !value.equals("0")) {
                if (sRes.length() > 0) {
                    sRes += '\n';
                }
                sRes += propertie.toLowerCase() + " : " + value;
            }
        }
        return sRes;
    }

	/* (non-Javadoc)
     * @see com.univ.objetspartages.processus.MediaSpecific#generateName(java.lang.String)
	 */

    /**
     * Save.
     *
     * @param _ctx
     *            the _ctx
     * @param sFileName
     *            the s file name
     * @param sAbsolutePathFile
     *            the s absolute path file
     * @param sAbsolutePathVignette
     *            the s absolute path vignette
     * @param sVignetteName
     *            the s vignette name
     *
     * @throws Exception
     *             the exception
     */
    public void save(final OMContext _ctx, final String sAbsolutePathFile, final String sAbsolutePathVignette, final String sFileName, final String sVignetteName) throws Exception {
        final String sOldAbsolutePathFile = getPathAbsolu();
        final String sOldAbsolutePathVignette = getPathVignetteAbsolu();
        boolean suppressionFile = false;
        boolean suppressionVignette = false;
        // média stocké en local
        if (isLocal()) {
            // nouveau fichier source
            if (!sAbsolutePathFile.equals(sOldAbsolutePathFile)) {
                suppressionFile = true;
                final File tmpFile = new File(sAbsolutePathFile);
                if (tmpFile.exists()) {
                    /* Création du répertoire si nécessaire */
                    String repertoire = "";
                    repertoire = MediathequeHelper.getAbsolutePath();
                    if (getTypeRessource().length() > 0) {
                        repertoire += File.separator + getTypeRessource().toLowerCase();
                    }
                    final File fRepertoire = new File(repertoire);
                    boolean okDirs = true;
                    if (!fRepertoire.exists()) {
                        okDirs = fRepertoire.mkdirs();
                    }
                    if (!okDirs) {
                        LOG.error("Impossible de creer le repertoire des medias :" + fRepertoire + " (verifier les droits d'ecriture sur le repertoire)");
                    }
                    /* Valorisation du path et de l'url */
                    final String newPath = repertoire + File.separator + sFileName;
                    // pas de suppression de source s'il s'agit d'une image qui n'a pas été redimmensionnée (les deux urls et donc la source sont alors identiques)
                    try {
                        FileUtil.primitiveCopierFichier(tmpFile, new File(newPath), !sAbsolutePathFile.equals(sAbsolutePathVignette));
                    } catch (final IOException e) {
                        throw new ErreurApplicative(String.format("Une erreur est survenue lors de l'enregistrement du média : %s", e.getMessage()));
                    }
                    setUrl(sFileName);
                }
            }
        }
        // nouvelle vignette
        if (!sAbsolutePathVignette.equals(sOldAbsolutePathVignette)) {
            suppressionVignette = true;
            final File tmpFile = new File(sAbsolutePathVignette);
            if (tmpFile.exists()) {
                /* Création du répertoire si nécessaire */
                String repertoire = MediathequeHelper.getAbsolutePath();
                if (getTypeRessource().length() > 0) {
                    repertoire += File.separator + getTypeRessource().toLowerCase();
                }
                //if (getTypeMedia().length()>0)
                //	repertoire += File.separator + getTypeMedia().toLowerCase();
                final File fRepertoire = new File(repertoire);
                boolean okDirs = true;
                if (!fRepertoire.exists()) {
                    okDirs = fRepertoire.mkdirs();
                }
                if (!okDirs) {
                    LOG.error("Impossible de creer le repertoire des medias :" + fRepertoire + " (verifier les droits d'ecriture sur le repertoire)");
                }
                /* Valorisation du path et de l'url */
                String newPath = repertoire + File.separator + "v_";
                // si vignette auto on garde le meme nom
                if (sVignetteName.length() > 0) {
                    newPath += sVignetteName;
                } else {
                    newPath += sFileName;
                }
                FileUtil.copierFichier(tmpFile, new File(newPath), true);
                setUrlVignette("v_" + ((sVignetteName.length() > 0) ? sVignetteName : sFileName));
            }
        }
        if (getIdMedia().equals(new Long(0))) {
            add();
        } else {
            update();
            // Suppression de l'ancien fichier physique
            if (suppressionFile) {
                if (sOldAbsolutePathFile.length() > 0) {
                    new File(sOldAbsolutePathFile).delete();
                }
            }
            if (suppressionVignette) {
                if (sOldAbsolutePathVignette.length() > 0) {
                    new File(sOldAbsolutePathVignette).delete();
                }
            }
        }
        // flush cache media
        CacheMediaManager.getInstance().refresh(this, true);
    }

    /**
     * Delete by id.
     *
     * @param ctx
     *            the ctx
     * @param id
     *            the id
     * @deprecated see {@link Media#deleteById(Long)}
     */
    @Deprecated
    public static void deleteById(final OMContext ctx, final Long id) {
        deleteById(ctx, id, true);
    }

    /**
     * Delete by id.
     *
     * @param ctx
     *            the ctx
     * @param id
     *            the id
     * @param onlyNotMutualise
     *            the only not mutualise
     * @deprecated see {@link Media#deleteById(Long, boolean)}
     */
    @Deprecated
    public static void deleteById(final OMContext ctx, final Long id, final boolean onlyNotMutualise) {
        try {
            final Media media = new Media();
            media.setCtx(ctx);
            media.setIdMedia(id);
            media.retrieve();
            if (!onlyNotMutualise || (onlyNotMutualise && !media.getIsMutualise().equals(Mediatheque.ETAT_MUTUALISE))) {
                media.delete();
                new File(media.getPathAbsolu()).delete();
                if (media.getUrlVignette().length() > 0) {
                    new File(media.getUrlVignetteAbsolue()).delete();
                }
            }
        } catch (final Exception e) {
            LOG.error("erreur lors de la requete en BDD", e);
        }
    }

    /**
     * Delete by id.
     *
     * @param id
     *            the id
     */
    public static void deleteById(final Long id) {
        deleteById(id, true);
    }

    /**
     * Delete a media by id.
     *
     * @param id
     *            the id
     * @param onlyNotMutualise
     *            should we delete the file if it's shared between different content?
     */
    public static void deleteById(final Long id, final boolean onlyNotMutualise) {
        try (ContexteDao ctx = new ContexteDao()) {
            final Media media = new Media();
            media.setCtx(ctx);
            media.setIdMedia(id);
            media.retrieve();
            if (!onlyNotMutualise || (onlyNotMutualise && !media.getIsMutualise().equals(Mediatheque.ETAT_MUTUALISE))) {
                media.delete();
                new File(media.getPathAbsolu()).delete();
                if (media.getUrlVignette().length() > 0) {
                    new File(media.getUrlVignetteAbsolue()).delete();
                }
            }
        } catch (final Exception e) {
            LOG.error("An error occured while deleting the media", e);
        }
    }

    /**
     * Gets the media.
     *
     * @param ctx
     *            the ctx
     * @param idMedia
     *            the id media
     *
     * @return the media
     * @deprecated le contexte n'est pas vraiment nécessaire. Utilisez {@link Media#getMedia(Long)}. L'unique différence c'est que la langue n'est pas setter.
     */
    @Deprecated
    public static Media getMedia(final OMContext ctx, final Long idMedia) {
        final Media media = new Media();
        media.setCtx(ctx);
        media.init();
        if (idMedia != null && !idMedia.equals(new Long(0))) {
            media.setIdMedia(idMedia);
            try {
                media.retrieve();
                if (ctx.getLocale() != null && !ctx.getLocale().equals(LangueUtil.getDefaultLocale())) {
                    media.setLangue(LangueUtil.getIndiceLocale(ctx.getLocale()));
                }
            } catch (final Exception e) {
                LOG.warn("Media id=" + idMedia + " inexistant");
            }
        }
        return media;
    }

    /**
     * Récupère un Media depuis l'id fourni en paramètre
     *
     * @param idMedia
     *            l'id du média à récupéerer
     * @return le média ayant cet id ou un Media vide sinon...
     */
    public static Media getMedia(final Long idMedia) {
        final Media media = new Media();
        if (idMedia != null && !idMedia.equals(new Long(0))) {
            media.init();
            media.setIdMedia(idMedia);
            try (ContexteDao ctx = new ContexteDao()) {
                media.setCtx(ctx);
                media.retrieve();
            } catch (final Exception e) {
                LOG.warn("Media id=" + idMedia + " inexistant");
            }
        }
        return media;
    }

    /**
     * Récupère un média à partir de son url ou de son url vignette
     * @param _ctx
     * @param sMedia
     * @return
     * @throws Exception
     */
    public static Media getMediaParUrlOuUrlVignette(final OMContext _ctx, final String sMedia) {
        final Media media = new Media();
        media.setCtx(_ctx);
        final Hashtable<String, String> h = new Hashtable<>();
        final ClauseWhere whereMedia = new ClauseWhere(ConditionHelper.egalVarchar("URL", sMedia));
        whereMedia.or(new ConditionHelper().egalVarchar("URL_VIGNETTE", sMedia));
        try {
            if (media.select(whereMedia.formaterSQL()) > 0) {
                if (media.nextItem()) {
                    return media;
                }
            }
        } catch (final Exception e) {
            LOG.debug("Erreur lors de la récupération du média par son Url", e);
        }
        return null;
    }

    /**
     * Renvoie la liste de tous les medias photos d'un type donné Peut être affiché directement dans une Combo.
     *
     * @param _ctx
     *            the _ctx
     * @param type
     *            the type
     *
     * @return the liste par type
     *
     * @throws Exception
     *             the exception
     */
    public static Hashtable<String, String> getListePhotoParType(final OMContext _ctx, final String type) throws Exception {
        final Media media = new Media();
        media.setCtx(_ctx);
        final Hashtable<String, String> h = new Hashtable<>();
        final ClauseWhere whereMedia = new ClauseWhere(ConditionHelper.egalVarchar("TYPE_RESSOURCE", "photo"));
        whereMedia.and(ConditionHelper.egalVarchar("TYPE_MEDIA", type));
        whereMedia.and(Media.conditionPerimetreMedia(_ctx, SaisieMedia.MODE_SELECTION));
        if (media.select(whereMedia.formaterSQL()) > 0) {
            while (media.nextItem()) {
                h.put(media.getIdMedia().toString(), media.getLibelleAffichable());
            }
        }
        return h;
    }

    /**
     * Modifie le code rubrique des médias non associées à une fiche (ex : encadres, newsletter)
     * @param infoBean
     * @param codeRubrique : Le nouveau code rubrique
     */
    public static void modifierCodeRubriqueParInfobean(final InfoBean infoBean, final String codeRubrique) {
        if (CollectionUtils.isNotEmpty(infoBean.getHashSet("contentNewRessources"))) {
            // contentNewRessources est valorisé dans la méthode ParserRequete#recueillirDonneesSaisies, on récupère tous les médias saisies lors de la requête
            Collection<String> listRessource = infoBean.getHashSet("contentNewRessources");
            for (String ressource : listRessource) {
                // les ressources sont de type ID#img
                String[] tabRessource = ressource.split("#");
                if (ArrayUtils.isNotEmpty(tabRessource) && StringUtils.isNotEmpty(tabRessource[0])) {
                    Media media = getMedia(Long.valueOf(tabRessource[0]));
                    if (media != null) {
                        media.setCodeRubrique(codeRubrique);
                        try {
                            media.update();
                        } catch (Exception e) {
                            LOG.error("Impossible de mettre à jour le média");
                        }
                    }
                }
            }
        }
    }

	/* (non-Javadoc)
     * @see com.univ.objetspartages.processus.MediaSpecific#generateName(java.lang.String)
	 */

    /**
     * Generate name.
     *
     * @param extension
     *            the extension
     *
     * @return the string
     */
    public synchronized String generateName(String extension) {
        if (generateName.equals("")) {
            if (extension.length() == 0) {
                extension = getExtension();
            }
            final String nomFichierSansExtension = StringUtils.defaultString(StringUtils.substringBeforeLast(getSource(), "."));
            generateName = Chaine.formatString(nomFichierSansExtension) + "_" + System.currentTimeMillis() + MediathequeHelper.PRE_EXTENSION_CHARACTER + extension;
        }
        return generateName;
    }

    /**
     * Retourne si possible l'extension du média
     *
     * @return l'extension du media, si le nom du mÃ©dia se termine par un . ou un -
     */
    public String getExtension() {
        String res = "";
        int i = getSource().lastIndexOf(".");
        if (i != -1) {
            res = getSource().substring(i + 1);
        } else {
            i = getFormat().lastIndexOf("-");
            if (i != -1) {
                res = getFormat().substring(i + 1);
            }
        }
        return res;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#clone()
     */
    @Override
    public Media clone() throws CloneNotSupportedException {
        return (Media) super.clone();
    }

    /**
     * Gets the langue.
     *
     * @return the langue
     */
    public int getLangue() {
        return langue;
    }

    /**
     * Sets the langue.
     *
     * @param langue
     *            the new langue
     */
    public void setLangue(final int langue) {
        this.langue = langue;
    }

    /* (non-Javadoc)
     * @see com.univ.objetspartages.sgbd.MediaDB#getDescription()
     */
    @Override
    public String getDescription() {
        if (langue != 0 && Mediatheque.ETAT_MUTUALISE.equals(getIsMutualise())) {
            return getTraductionData("DESCRIPTION_" + langue);
        } else {
            return super.getDescription();
        }
    }

    /* (non-Javadoc)
     * @see com.univ.objetspartages.sgbd.MediaDB#getTitre()
     */
    @Override
    public String getTitre() {
        if (langue != 0 && Mediatheque.ETAT_MUTUALISE.equals(getIsMutualise())) {
            return getTraductionData("TITRE_" + langue);
        } else {
            return super.getTitre();
        }
    }

    /* (non-Javadoc)
     * @see com.univ.objetspartages.sgbd.MediaDB#getAuteur()
     */
    @Override
    public String getAuteur() {
        if (langue != 0 && Mediatheque.ETAT_MUTUALISE.equals(getIsMutualise())) {
            return getTraductionData("AUTEUR_" + langue);
        } else {
            return super.getAuteur();
        }
    }

    /**
     * Gets the.
     *
     * @return the string
     */
    @Override
    public String getCopyright() {
        if ((langue != 0) && Mediatheque.ETAT_MUTUALISE.equals(getIsMutualise())) {
            return getTraductionData("COPYRIGHT_" + langue);
        } else {
            return super.getCopyright();
        }
    }

    /* (non-Javadoc)
     * @see com.univ.objetspartages.sgbd.MediaDB#getLegende()
     */
    @Override
    public String getLegende() {
        if (langue != 0 && Mediatheque.ETAT_MUTUALISE.equals(getIsMutualise())) {
            return getTraductionData("LEGENDE_" + langue);
        } else {
            return super.getLegende();
        }
    }

    /* (non-Javadoc)
     * @see com.univ.objetspartages.sgbd.MediaDB#getMetaKeywords()
     */
    @Override
    public String getMetaKeywords() {
        if (langue != 0 && Mediatheque.ETAT_MUTUALISE.equals(getIsMutualise())) {
            return getTraductionData("LEGENDE_" + "META_KEYWORDS_" + langue);
        } else {
            return super.getMetaKeywords();
        }
    }
    
}
