/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Enumeration;
import java.util.GregorianCalendar;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.Formateur;
import com.kportal.core.config.PropertyHelper;
import com.kportal.extension.module.plugin.toolbox.PluginTagHelper;
import com.univ.objetspartages.sgbd.MetatagDB;
import com.univ.utils.Chaine;
import com.univ.utils.ContexteDao;
import com.univ.utils.FicheUnivMgr;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.utils.sql.operande.TypeOperande;

/**
 * The Class Metatag.
 */
public class Metatag extends MetatagDB implements Cloneable {

	/**
	 *
	 */
	private static final long serialVersionUID = 5615825273534294569L;

    // REGEX pour capturer tous les liens compris dans les balises <a> et l'attribut href"
    public static final String REGEX_CAPTURE_LIEN = ".*?<a.*?href=\"([^\\\"]*)\".*?<\\/a>.*?";

    public static final Pattern PATTERN_CAPTURE_LIEN = Pattern.compile(REGEX_CAPTURE_LIEN);

    //  On vérifie que l'URL commence bien par un / et on récupère l'extension (kjsp,htm, etc.. )
    public static final String REGEX_CAPTURE_EXTENSION = "^\\/.*\\.([^?#]*)[?#]?.*";

    public static final Pattern PATTERN_CAPTURE_EXTENSION = Pattern.compile(REGEX_CAPTURE_EXTENSION);

    public static final String REGEX_ID_METATAG = "^.*(?:\\D(\\d+))\\.[^.]*$";

    public static final Pattern PATTERN_ID_METATAG = Pattern.compile(REGEX_ID_METATAG);

    private static final Logger LOGGER = LoggerFactory.getLogger(Metatag.class);

	/** The HISTORIQU e_ annulatio n_ demand e_ validation. */
	public static String HISTORIQUE_ANNULATION_DEMANDE_VALIDATION = "ANNULATION_DEMANDE_VALIDATION";

	/** The HISTORIQU e_ annulatio n_ suppression. */
	public static String HISTORIQUE_ANNULATION_SUPPRESSION = "ANNULATION_SUPPRESSION";

	/** The HISTORIQU e_ archivage. */
	public static String HISTORIQUE_ARCHIVAGE = "ARCHIVAGE";

	/** The HISTORIQU e_ archivage. */
	public static String HISTORIQUE_ARCHIVAGE_AUTO = "ARCHIVAGE_AUTO";

	/** The HISTORIQU e_ rubriquage. */
	public static String HISTORIQUE_RUBRIQUAGE = "RUBRIQUAGE";

	/** The HISTORIQU e_ rubriquage. */
	public static String HISTORIQUE_RUBRIQUAGE_AUTO = "RUBRIQUAGE_AUTO";

	/** The HISTORIQU e_ creation. */
	public static String HISTORIQUE_CREATION = "CREATION";

	/** The HISTORIQU e_ demand e_ validation. */
	public static String HISTORIQUE_DEMANDE_VALIDATION = "DEMANDE_VALIDATION";

	/** The HISTORIQU e_ duplication. */
	public static String HISTORIQUE_DUPLICATION = "DUPLICATION";

	/** The HISTORIQU e_ traduction. */
	public static String HISTORIQUE_TRADUCTION = "TRADUCTION";

	/** The HISTORIQU e_ modification. */
	public static String HISTORIQUE_MODIFICATION = "MODIFICATION";

	/** The HISTORIQU e_ restauratio n_ archivage. */
	public static String HISTORIQUE_RESTAURATION_ARCHIVAGE = "RESTAURATION_ARCHIVAGE";

	/** The HISTORIQU e_ restauratio n_ sauvegarde. */
	public static String HISTORIQUE_RESTAURATION_SAUVEGARDE = "RESTAURATION_SAUVEGARDE";

	/** The HISTORIQU e_ retou r_ auteur. */
	public static String HISTORIQUE_RETOUR_AUTEUR = "RETOUR_AUTEUR";

	/** The HISTORIQU e_ suppression. */
	public static String HISTORIQUE_SUPPRESSION = "SUPPRESSION";

	/** The HISTORIQU e_ validation. */
	public static String HISTORIQUE_VALIDATION = "VALIDATION";

	/**
	 * Instantiates a new metatag.
	 */
	public Metatag() {
		super();
	}

	/**
	 * Inits the.
	 */
	public void init() {
		setIdMetatag((long) 0);
		setMetaIdFiche((long) 0);
		setMetaCodeObjet("");
		setMetaLibelleObjet("");
		setMetaHistorique("");
		setMetaDateArchivage(new Date(0));
		setMetaDateOperation(new Date(System.currentTimeMillis()));
		setMetaListeReferences("");
		setMetaForum("");
		setMetaForumAno("");
		setMetaMailAnonyme("");
		setMetaSaisieFront("0");
		setMetaInTree("0");
		setMetaNotificationMail("0");
		setMetaDocumentFichiergw("0");
		setMetaRubriquesPublication("");
		setMetaLibelleFiche("");
		setMetaCode("");
		setMetaCodeRattachement("");
		setMetaCodeRattachementAutres("");
		setMetaCodeRubrique("");
		setMetaMetaKeywords("");
		setMetaMetaDescription("");
		setMetaDateCreation(new Date(0));
		setMetaDateProposition(new Date(0));
		setMetaDateValidation(new Date(0));
		setMetaDateModification(new Date(0));
		setMetaCodeRedacteur("");
		setMetaCodeValidation("");
		setMetaLangue("");
		setMetaEtatObjet("");
		setMetaNbHits((long) 0);
		setMetaDiffusionPublicVise("");
		setMetaDiffusionModeRestriction("");
		setMetaDiffusionPublicViseRestriction("");
		setMetaNiveauApprobation("");
		setMetaSourceImport("");
		setMetaDateMiseEnLigne(new Date(0));
		setMetaDateSuppression(new Date(0));
		setMetaDateRubriquage(new Date(0));
		setMetaCodeRubriquage("");
	}

	/**
	 * Ajoute un élément dans l'historique.
	 *
	 * @param evenement
	 *            the evenement
	 * @param code_auteur
	 *            the code_auteur
	 * @param etat_objet
	 *            the etat_objet
	 *
	 */
	public void addHistorique(final String evenement, final String code_auteur, final String etat_objet) {
		final Vector<String> v = getVecteurHistorique();
		// Suppression dernier élément
		if (v.size() > 19) {
			v.removeElementAt(v.size() - 1);
		}
		final java.util.GregorianCalendar cal = new java.util.GregorianCalendar();
		final Object[] arguments = {
				cal.get(Calendar.DAY_OF_MONTH), cal.get(Calendar.MONTH) + 1, cal.get(Calendar.YEAR), cal.get(Calendar.HOUR_OF_DAY), cal.get(Calendar.MINUTE), cal.get(Calendar.SECOND)
		};
		final String dateModif = java.text.MessageFormat.format("{2,number,0000}{1,number,00}{0,number,00}{3,number,00}{4,number,00}{5,number,00}", arguments);
		final String item = "[" + evenement + "/" + dateModif + "/" + code_auteur + "/" + etat_objet + "]";
		v.insertElementAt(item, 0);
		setVecteurHistorique(v);
	}

	/**
	 * Etablit la liste des références à partir de la chaine renvoyée par FicheUniv.getReferences().
	 *
	 * @param ficheUniv
	 *            the fiche univ
	 */
	public void calculerReferences(final FicheUniv ficheUniv) {
		String references = ficheUniv.getReferences();
		references += FicheUnivMgr.getReferenceParTexte(ficheUniv.getContenuEncadre());
		final String langue = ficheUniv.getLangue();
		String newRef = "";
		try {
			newRef += PluginTagHelper.getReferencesTags(references);
		} catch (final Exception e) {
			LOGGER.error("impossible de calculer les références des tags", e);
		}
		int indexTexte = 0;
		int indexDebutMotCle = -1;
		int indexFinMotCle = 0;
		/********************************************************
		 * LIENS INTER-FICHES *
		 *********************************************************/
		/* Boucle sur chaque mot clé */
		while ((indexDebutMotCle = references.indexOf("[id-fiche]", indexTexte)) != -1) {
			// Extraction de la chaine
			indexFinMotCle = references.indexOf("[/id-fiche]", indexDebutMotCle);
			// Fin du traitement si  tag incomplet
			if (indexFinMotCle == -1) {
				break;
			}
			final int indexDebutMotCleSuivant = references.indexOf("[id-fiche]", indexDebutMotCle + 1);
			if ((indexDebutMotCleSuivant != -1) && (indexDebutMotCleSuivant < indexFinMotCle)) {
				break;
			}
			final String chaine = references.substring(indexDebutMotCle + 10, indexFinMotCle);
			// Analyse de la chaine
			String codeFiche = "", objetReference = "";
			int indiceToken = 0;
			final StringTokenizer st = new StringTokenizer(chaine, ";");
			while (st.hasMoreTokens()) {
				if (indiceToken == 0) {
					objetReference = st.nextToken();
				} else if (indiceToken == 1) {
					codeFiche = st.nextToken();
				} else {
					st.nextToken();
				}
				indiceToken++;
			}
			/* si le code est vide, ou contient le caractère = il s'agit d'une requete
			   sinon, il s'agit d'une fiche
			 */
			boolean traitementRequete = false;
			if (codeFiche.length() == 0) {
				traitementRequete = true;
			}
			if (codeFiche.contains("=")) {
				if (!codeFiche.contains(",LANGUE=")) {
					traitementRequete = true;
				}
			}
			if (!traitementRequete) {
				/* Analyse de la langue */
				String codeReference = "";
				String langueReference = "";
				if (!codeFiche.contains(",LANGUE=")) {
					codeReference = codeFiche;
					langueReference = langue;
				} else {
					codeReference = codeFiche.substring(0, codeFiche.indexOf(",LANGUE="));
					langueReference = codeFiche.substring(codeFiche.indexOf(",LANGUE=") + 8);
				}
				/* Ajout de la référence */
				newRef += "[" + objetReference + ";" + codeReference + ";" + langueReference + "]";
			}
			indexTexte = indexFinMotCle + 11;
		}
        /********************************************************
         * LIENS RELATIFS *
         *********************************************************/
        Set<String> listeLiens = getListesLiensParReferences(references);
        if (CollectionUtils.isNotEmpty(listeLiens)) {
            final String extension = PropertyHelper.getCoreProperty("application.content_extension");
            for (String lien : listeLiens) {
                if (!this.validerUrlRelative(lien, extension)) {
                    continue;
                }
                String newReferencesMeta = getReferenceParMeta(lien);
                if (StringUtils.isNotEmpty(newReferencesMeta)) {
                    newRef += newReferencesMeta;
                }
            }
        }
		/********************************************************
		 * LIENS DIRECT DOCUMENTS *
		 *********************************************************/
		indexTexte = 0;
		indexDebutMotCle = -1;
		indexFinMotCle = 0;
		while ((indexDebutMotCle = references.indexOf("[id-document]", indexTexte)) != -1) {
			// Extraction de la chaine
			indexFinMotCle = references.indexOf("[/id-document]", indexDebutMotCle);
			// JSS 20020910-001 : Fin du traitement si  tag incomplet
			if (indexFinMotCle == -1) {
				break;
			}
			final int indexDebutMotCleSuivant = references.indexOf("[id-document]", indexDebutMotCle + 1);
			if ((indexDebutMotCleSuivant != -1) && (indexDebutMotCleSuivant < indexFinMotCle)) {
				break;
			}
			final String document = references.substring(indexDebutMotCle + 14, indexFinMotCle);
			String codeReference = "";
			String langueReference = "";
			if (!document.contains(",LANGUE=")) {
				codeReference = document;
				langueReference = langue;
			} else {
				codeReference = document.substring(0, document.indexOf(",LANGUE="));
				langueReference = document.substring(document.indexOf(",LANGUE=") + 8);
			}
			/* Ajout de la référence */
			newRef += "[document;" + codeReference + ";" + langueReference + "]";
			indexTexte = indexFinMotCle + 14;
		}
		setMetaListeReferences(newRef);
	}

	/**
	 * Renvoie la date de dernière modif à partir de l'historique.
	 *
	 * @return the date derniere modification
	 *
	 * @throws Exception
	 *             the exception
	 */
	public GregorianCalendar getDateDerniereModification() throws Exception {
		GregorianCalendar cal = null;
		if (getVecteurHistorique().size() > 0) {
			final StringTokenizer st = new StringTokenizer(getVecteurHistorique().elementAt(0), "[/]");
			// on retient le 2ème token pour la date*/
			st.nextToken();
			final String date = st.nextToken();
			// Conversion en Calendar
			int annee, mois, jour, heure, minute, secondes;
			annee = Integer.parseInt(date.substring(0, 4));
			mois = Integer.parseInt(date.substring(4, 6));
			jour = Integer.parseInt(date.substring(6, 8));
			heure = Integer.parseInt(date.substring(8, 10));
			minute = Integer.parseInt(date.substring(10, 12));
			secondes = Integer.parseInt(date.substring(12, 14));
			cal = new java.util.GregorianCalendar(annee, mois - 1, jour, heure, minute, secondes);
		}
		return cal;
	}

	/**
	 * Renvoie un libellé d'action (à partir du code stocké dans l'historique).
	 *
	 * @param action
	 *            the action
	 *
	 * @return the intitule action
	 */
	public static String getIntituleAction(final String action) {
		String res = "";
		final Hashtable<String, String> actions = new Hashtable<String, String>();
		actions.put(HISTORIQUE_SUPPRESSION, "Suppression");
		actions.put(HISTORIQUE_ANNULATION_SUPPRESSION, "Annulation suppression");
		actions.put(HISTORIQUE_RESTAURATION_SAUVEGARDE, "Restauration");
		actions.put(HISTORIQUE_DUPLICATION, "Duplication");
		actions.put(HISTORIQUE_TRADUCTION, "Traduction");
		actions.put(HISTORIQUE_CREATION, "Création");
		actions.put(HISTORIQUE_MODIFICATION, "Modification");
		actions.put(HISTORIQUE_VALIDATION, "Validation");
		actions.put(HISTORIQUE_DEMANDE_VALIDATION, "Demande de validation");
		actions.put(HISTORIQUE_ANNULATION_DEMANDE_VALIDATION, "Annulation demande ");
		actions.put(HISTORIQUE_RETOUR_AUTEUR, "Retour au rédacteur");
		actions.put(HISTORIQUE_RUBRIQUAGE, "Rubriquage ");
		actions.put(HISTORIQUE_RUBRIQUAGE_AUTO, "Rubriquage automatique ");
		actions.put(HISTORIQUE_ARCHIVAGE, "Archivage ");
		actions.put(HISTORIQUE_ARCHIVAGE_AUTO, "Archivage automatique ");
		actions.put(HISTORIQUE_RESTAURATION_ARCHIVAGE, "Restauration archive");
		if (actions.get(action) != null) {
			res = actions.get(action);
		}
		return res;
	}

	/**
	 * Renvoie la liste de de l'historique.
	 *
	 * @return the vecteur historique
	 *
	 */
	public Vector<String> getVecteurHistorique() {
		final Vector<String> v = new Vector<String>();
		final StringTokenizer st = new StringTokenizer(getMetaHistorique(), ";");
		while (st.hasMoreTokens()) {
			final String val = st.nextToken();
			v.add(val);
		}
		return v;
	}

	/**
	 * Sélection d'une Article à partir de l'ensemble des critères combinés.
	 *
	 * @param codeObjet
	 *            the _code objet
	 * @param idFiche
	 *            the id fiche
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 */
	public int select(final String codeObjet, final Long idFiche) throws Exception {
		final ClauseWhere whereCodeObjetIdFiche = new ClauseWhere();
		if (StringUtils.isNotEmpty(codeObjet)) {
			whereCodeObjetIdFiche.setPremiereCondition(ConditionHelper.egalVarchar("META_CODE_OBJET", codeObjet));
		}
		if (idFiche != null) {
			whereCodeObjetIdFiche.and(ConditionHelper.egal("META_ID_FICHE", idFiche, TypeOperande.LONG));
		}
		return select(whereCodeObjetIdFiche.formaterSQL());
	}

	/**
	 * Sélection de Metatag par type d'objet sur une liste de valeur
	 *
	 * @param codeObjet
	 *            the _code objet
	 * @param idsFiches
	 *            the id fiche
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 */
	public int selectParCodeIds(final String codeObjet, final Collection<String> idsFiches) throws Exception {
		final ClauseWhere whereCodeObjetIdFiche = new ClauseWhere();
		if (StringUtils.isNotEmpty(codeObjet)) {
			whereCodeObjetIdFiche.setPremiereCondition(ConditionHelper.egalVarchar("META_CODE_OBJET", codeObjet));
		}
		if (CollectionUtils.isNotEmpty(idsFiches)) {
			whereCodeObjetIdFiche.and(ConditionHelper.in("META_ID_FICHE", idsFiches));
		}
		return select(whereCodeObjetIdFiche.formaterSQL());
	}

	/**
	 * Sélection d'une liste de Méta-tags des fiches qui référencent une fiche ou une photo déterminée .
	 *
	 * @param codeObjet
	 *            the _code objet
	 * @param code
	 *            the code
	 * @param langue
	 *            the langue
	 * @param idPhoto
	 *            the id photo
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 */
	public int selectParReferences(final String codeObjet, final String code, final String langue, final Long idPhoto) throws Exception {
		final ClauseWhere whereReferences = new ClauseWhere();
		if (StringUtils.isNotEmpty(codeObjet)) {
			final String nomObjet = ReferentielObjets.getNomObjet(codeObjet);
			// On peut avoir des references du type [annuaire;1493737150098;0]
			whereReferences.setPremiereCondition(ConditionHelper.like("META_LISTE_REFERENCES", nomObjet + ";" + code + ";" + langue, "%[", "]%"));
			// ou [annuaire;1493737150098,0,0000;0]
			whereReferences.or(ConditionHelper.like("META_LISTE_REFERENCES", nomObjet + ";" + code + ",%;" + langue, "%[", "]%"));
		}
		if (!idPhoto.equals(new Long(0))) {
			whereReferences.and(ConditionHelper.like("META_LISTE_REFERENCES", String.valueOf(idPhoto), "%[photo;", "]%"));
		}
		return select(whereReferences.formaterSQL());
	}

	/**
	 * Valorise la liste de l'historique sous forme de vecteur.
	 *
	 * @param v
	 *            the new vecteur historique
	 *
	 */
	public void setVecteurHistorique(final Vector<String> v) {
		String liste = "";
		final Enumeration<String> en = v.elements();
		while (en.hasMoreElements()) {
			liste = liste + (en.nextElement()) + ";";
		}
		setMetaHistorique(liste);
	}

	/**
	 * Select count.
	 *
	 * @param sqlSuffix
	 *            the sql suffix
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 */
	public int selectCount(String sqlSuffix) throws Exception {
		int count = 0;
		PreparedStatement _stmt = null;
		try {
			if (sqlSuffix == null) {
				sqlSuffix = "";
			}
			final String query = "SELECT COUNT( DISTINCT ID_METATAG ) FROM " + qualifier + "METATAG T1 " + sqlSuffix;
			/* rÃ©cupÃ©ration nombre de lignes */
			_stmt = ctx.getConnection().prepareStatement(query);
			final ResultSet _rs = _stmt.executeQuery(query);
			_rs.next();
			count = _rs.getInt(1);
		} catch (final SQLException exc) {
			throw new Exception("SELECT_FAILED " + exc);
		} finally {
			_stmt = null;
		}
		return count;
	}

	/**
	 * Gets the liste objets redacteur.
	 *
	 * @param codeRedacteur
	 *            the code redacteur
	 *
	 * @return the liste objets redacteur
	 *
	 * @throws Exception
	 *             the exception
	 */
	public Hashtable<String, String> getListeObjetsRedacteur(final String codeRedacteur) throws Exception {
		final Hashtable<String, String> hTypeFiche = new Hashtable<String, String>();
		PreparedStatement _stmt = null;
		try {
			final ClauseWhere whereCodeRedacteur = new ClauseWhere(ConditionHelper.egalVarchar("META_CODE_REDACTEUR", codeRedacteur));
			final String query = "SELECT DISTINCT ( META_CODE_OBJET ) FROM " + qualifier + " METATAG T1 " + whereCodeRedacteur.formaterSQL();
			_stmt = ctx.getConnection().prepareStatement(query);
			final ResultSet _rs = _stmt.executeQuery();
			while (_rs.next()) {
				hTypeFiche.put(_rs.getString(1), "");
			}
		} catch (final Exception exc) {
			throw new Exception("Erreur lors de la récupération des objets utilisateurs ");
		} finally {
			_stmt = null;
		}
		return hTypeFiche;
	}

	/**
	 * Test de cohérence sur un champ.
	 *
	 * @param differences
	 *            the differences
	 * @param nomChamp
	 *            the nom champ
	 * @param valeurMeta
	 *            the valeur meta
	 * @param valeurFiche
	 *            the valeur fiche
	 */
	private void testerCoherenceChamp(final List<String> differences, final String nomChamp, final String valeurMeta, final String valeurFiche) {
		if (valeurMeta == null || !valeurMeta.equals(valeurFiche)) {
			differences.add("champ: " + nomChamp + " fiche: " + valeurFiche + " meta:" + valeurMeta);
		}
	}

	/**
	 * Teste si les données dupliquées dans les métas sont cohérentesavec celles de la la fiche d'origine.
	 *
	 * @param ficheRef
	 *            the fiche ref
	 *
	 * @return the list
	 */
	public List<String> controlerCoherenceAvecFiche(final FicheUniv ficheRef) {
		final List<String> differences = new ArrayList<String>();
		testerCoherenceChamp(differences, "code", getMetaCode(), ficheRef.getCode());
		testerCoherenceChamp(differences, "code_rattachement", getMetaCodeRattachement(), ficheRef.getCodeRattachement());
		final String codeRubriqueFiche = ficheRef.getCodeRubrique();
		testerCoherenceChamp(differences, "code_rubrique", getMetaCodeRubrique(), codeRubriqueFiche);
		String dateMeta = Formateur.formater(getMetaDateCreation());
		String dateFiche = Formateur.formater(ficheRef.getDateCreation());
		testerCoherenceChamp(differences, "date_creation", dateMeta, dateFiche);
		dateMeta = Formateur.formater(getMetaDateProposition());
		dateFiche = Formateur.formater(ficheRef.getDateProposition());
		testerCoherenceChamp(differences, "date_proposition", dateMeta, dateFiche);
		dateMeta = Formateur.formater(getMetaDateValidation());
		dateFiche = Formateur.formater(ficheRef.getDateValidation());
		testerCoherenceChamp(differences, "date_validation", dateMeta, dateFiche);
		dateMeta = Formateur.formater(getMetaDateModification());
		dateFiche = Formateur.formater(ficheRef.getDateModification());
		testerCoherenceChamp(differences, "date_modification", dateMeta, dateFiche);
		dateMeta = Formateur.formater(getMetaDateModification());
		dateFiche = Formateur.formater(ficheRef.getDateModification());
		testerCoherenceChamp(differences, "date_modification", dateMeta, dateFiche);
		dateMeta = Formateur.formater(getMetaDateModification());
		dateFiche = Formateur.formater(ficheRef.getDateModification());
		testerCoherenceChamp(differences, "date_modification", dateMeta, dateFiche);
		testerCoherenceChamp(differences, "code_redacteur", getMetaCodeRedacteur(), ficheRef.getCodeRedacteur());
		testerCoherenceChamp(differences, "code_validation", getMetaCodeValidation(), ficheRef.getCodeValidation());
		testerCoherenceChamp(differences, "langue", getMetaLangue(), ficheRef.getLangue());
		testerCoherenceChamp(differences, "etat_objet", getMetaEtatObjet(), ficheRef.getEtatObjet());
		String codeRattachementAutres = "";
		if (ficheRef instanceof FicheRattachementsSecondaires) {
			codeRattachementAutres = Chaine.convertirPointsVirgulesEnAccolades(((FicheRattachementsSecondaires) ficheRef).getCodeRattachementAutres());
			testerCoherenceChamp(differences, "code_rattachement_autres", getMetaCodeRattachementAutres(), codeRattachementAutres);
		}
		String diffusionPublicVise = "";
		if (ficheRef instanceof DiffusionSelective) {
			diffusionPublicVise = ((DiffusionSelective) ficheRef).getDiffusionPublicVise();
			testerCoherenceChamp(differences, "diffusion_public_vise", getMetaDiffusionPublicVise(), diffusionPublicVise);
		}
		String diffusionModeRestriction = "";
		if (ficheRef instanceof DiffusionSelective) {
			diffusionModeRestriction = ((DiffusionSelective) ficheRef).getDiffusionModeRestriction();
			testerCoherenceChamp(differences, "diffusion_public_vise", getMetaDiffusionModeRestriction(), diffusionModeRestriction);
		}
		String diffusionPublicViseRestriction = "";
		if (ficheRef instanceof DiffusionSelective) {
			diffusionPublicViseRestriction = ((DiffusionSelective) ficheRef).getDiffusionPublicViseRestriction();
			testerCoherenceChamp(differences, "diffusion_public_vise", getMetaDiffusionPublicViseRestriction(), diffusionPublicViseRestriction);
		}
		return differences;
	}

	/**
	 * Synchroniser.
	 *
	 * @param ficheUniv
	 *            the fiche univ
	 * @param calculerReferences
	 *            the calculer references
	 *
	 * @throws Exception
	 *             the exception
	 */
	public void synchroniser(final FicheUniv ficheUniv, final boolean calculerReferences) throws Exception {
		if (calculerReferences) {
			calculerReferences(ficheUniv);
		}
		setMetaIdFiche(ficheUniv.getIdFiche());
		setMetaCodeRattachement(ficheUniv.getCodeRattachement());
		setMetaCodeRubrique(ficheUniv.getCodeRubrique());
		if (ficheUniv instanceof FicheRattachementsSecondaires) {
			setMetaCodeRattachementAutres(Chaine.convertirPointsVirgulesEnAccolades(((FicheRattachementsSecondaires) ficheUniv).getCodeRattachementAutres()));
		} else {
			setMetaCodeRattachementAutres("");
		}
		setMetaLibelleObjet(ReferentielObjets.getLibelleObjet(getMetaCodeObjet()));
		setMetaLibelleFiche(ficheUniv.getLibelleAffichable());
		setMetaMetaKeywords(ficheUniv.getMetaKeywords());
		setMetaMetaDescription(ficheUniv.getMetaDescription());
		setMetaDateCreation(ficheUniv.getDateCreation());
		setMetaDateProposition(ficheUniv.getDateProposition());
		setMetaDateValidation(ficheUniv.getDateValidation());
		setMetaDateModification(ficheUniv.getDateModification());
		setMetaCodeRedacteur(ficheUniv.getCodeRedacteur());
		setMetaCodeValidation(ficheUniv.getCodeValidation());
		setMetaCode(ficheUniv.getCode());
		setMetaLangue(ficheUniv.getLangue());
		setMetaEtatObjet(ficheUniv.getEtatObjet());
		setMetaNbHits(ficheUniv.getNbHits());
		if (ficheUniv instanceof DiffusionSelective) {
			setMetaDiffusionPublicVise(((DiffusionSelective) ficheUniv).getDiffusionPublicVise());
			setMetaDiffusionModeRestriction(((DiffusionSelective) ficheUniv).getDiffusionModeRestriction());
			if (((DiffusionSelective) ficheUniv).getDiffusionModeRestriction().equals("4")) {
				setMetaForumAno("0");
			}
			setMetaDiffusionPublicViseRestriction(((DiffusionSelective) ficheUniv).getDiffusionPublicViseRestriction());
		} else {
			setMetaDiffusionPublicVise("");
			setMetaDiffusionModeRestriction("0");
			setMetaDiffusionPublicViseRestriction("");
		}
		if (getMetaForumAno() == null) {
			setMetaForumAno("");
		}
		if (!Formateur.estSaisie(getMetaDateMiseEnLigne())) {
			setMetaDateMiseEnLigne(ficheUniv.getDateCreation());
		}
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#clone()
	 */
	@Override
	public Metatag clone() throws CloneNotSupportedException {
		return (Metatag) super.clone();
	}

    /**
     * Renvoi tous les liens d'une réference
     * @param references
     * @return
     */
	protected Set<String> getListesLiensParReferences(final String references) {
        Set<String> listeReferences = new HashSet<>();
        if (StringUtils.isNotEmpty(references)) {
            // On récupère tous les liens relatifs dans les réferences
            final Matcher matcher = PATTERN_CAPTURE_LIEN.matcher(references);
            while (matcher.find()) {
                for (int i = 1; i <= matcher.groupCount(); i++) {
                    String url = matcher.group(1);
                    if(StringUtils.isNotEmpty(url)) {
                        if (LOGGER.isDebugEnabled()) {
                            LOGGER.debug(String.format("URL detectée : %s", url));
                        }
                        listeReferences.add(url);
                    }
                }
            }
        }
        return listeReferences;
    }

    /**
     * Renvoi la réference comprise dans uine URL
     * @param url
     * @return
     */
    public String getReferenceParMeta(final String url) {
        // Pour chaque URL récuperée on regarde si l'ID méta correspond à une fiche
        String idMetatag = this.getIdMetaParUrl(url);
        if (StringUtils.isNotEmpty(idMetatag)) {
            try (final ContexteDao ctxConnection = new ContexteDao()) {
                final Metatag meta = new Metatag();
                meta.setCtx(ctxConnection);
                try {
                    meta.setIdMetatag(Long.valueOf(idMetatag));
                } catch (NumberFormatException e1) {
                    LOGGER.error("Le metatag récuperé n'est pas numérique", e1);
                    return StringUtils.EMPTY;
                }
                try {
                    meta.retrieve();
                } catch (Exception e) {
                    LOGGER.error("Impossible de retrouver le metatag", e);
                    return StringUtils.EMPTY;
                }
                String objet = ReferentielObjets.getNomObjet(meta.getMetaCodeObjet());
                String langueObjet = meta.getMetaLangue();
                String codeFiche = meta.getMetaCode();
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Ajout de la réference, objet : %s , code : %s , langue : %s", objet, codeFiche, langueObjet);
                }
                return "[" + objet + ";" + codeFiche + ";" + langueObjet + "]";
            }
        }
        return StringUtils.EMPTY;
    }

    /**
     * Renvoi l'id métatag contenu par l'url
     * @param url
     * @return
     */
    protected String getIdMetaParUrl(String url) {
        if (StringUtils.isNotEmpty(url)) {
            final Matcher matcherIdMeta = PATTERN_ID_METATAG.matcher(url);
            if (matcherIdMeta.find() && StringUtils.isNotEmpty(matcherIdMeta.group(1))) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug(String.format("ID Métatag trouvé : %s", matcherIdMeta.group(1)));
                }
                return matcherIdMeta.group(1);
            }
        }
        return StringUtils.EMPTY;
    }

    /**
     * Méthode qui valide que l'URL fournie est bien une URL relative et qui a pour extension, l'extension 
     * définie dans les properties
     * @param url
     * @param extension
     * @return
     */
    protected boolean validerUrlRelative(final String url, final String extension) {
        if (StringUtils.isNotEmpty(url) && StringUtils.isNotEmpty(extension)) {
            final Matcher matcherValidationUrl = PATTERN_CAPTURE_EXTENSION.matcher(url);
            if (matcherValidationUrl.find() && StringUtils.isNotEmpty(matcherValidationUrl.group(1))) {
                return matcherValidationUrl.group(1).equals(extension);
            }
        }
        return false;
    }
}
