/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.util.Iterator;

import org.apache.commons.lang3.StringUtils;

/**
 * The Class Perimetre.
 * 
 * @author jean-sébastien steux
 * 
 *         stockage, formatage et recherche d'un périmètre de délégation Un périmètre est une combinaison (ET logique) d'une structure d'une rubrique et d'un groupe
 */
public class Perimetre {

	/** The code structure. */
	private String codeStructure = "";

	/** The code rubrique. */
	private String codeRubrique = "";

	/** The code profil. */
	private String codeProfil = "";

	/** The code groupe. */
	private String codeGroupe = "";

	/** The code espace collaboratif. */
	private String codeEspaceCollaboratif = "";

	/** The identifiant niveau. */
	private String identifiantNiveau = "";

	/**
	 * Intialisation à partir de la chaine en base de donnée (sous la forme STRUCTURE/RUBRIQUE/PROFIL/GROUPE/ESPACE).
	 * 
	 * @param chaineSerialisee
	 *            the chaine serialisee
	 */
	public Perimetre(final String chaineSerialisee) {
		super();
		codeStructure = "";
		codeRubrique = "";
		codeProfil = "";
		codeGroupe = "";
		codeEspaceCollaboratif = "";
		String chaineTmp = chaineSerialisee;
		int indice = 0;
		while (chaineTmp.length() > 0) {
			String item = "";
			// Extraction du premier item
			final int iSlash = chaineTmp.indexOf('/');
			if (iSlash != -1) {
				item = chaineTmp.substring(0, iSlash);
				chaineTmp = chaineTmp.substring(iSlash + 1);
			} else {
				item = chaineTmp;
				chaineTmp = "";
			}
			switch (indice) {
				case 0:
					codeStructure = item;
					break;
				case 1:
					codeRubrique = item;
					break;
				case 2:
					codeProfil = item;
					break;
				case 3:
					codeGroupe = item;
					break;
				case 4:
					codeEspaceCollaboratif = item;
					break;
			}
			indice++;
		}
		/*
		 String[] params = new String[5];
		 String chaineTmp = chaineSerialisee;
		 int indice = 0;
		 int iDeb = 0;
		 int iFin = 0;
		 for (int i = 0; i < 5; i++) {
		 iFin = chaineTmp.indexOf('/', iDeb);
		 if (iFin == -1) {
		 iFin = chaineTmp.length() - 1;
		 }
		 params[i] = chaineTmp.substring(iDeb, iFin);
		 iDeb = iFin+1;
		 }
		 super(param[0], param[1], param[2], param[3], param[4]);
		 */
		identifiantNiveau = codeStructure + "/" + codeRubrique + "/" + codeGroupe + "/" + codeEspaceCollaboratif;
	}

	/**
	 * Intialisation à partir des valeurs saisies.
	 * 
	 * @param _codeStructure
	 *            the _code structure
	 * @param _codeRubrique
	 *            the _code rubrique
	 * @param _codeProfil
	 *            the _code profil
	 * @param _codeGroupe
	 *            the _code groupe
	 * @param _codeEspaceCollaboratif
	 *            the _code espace collaboratif
	 */
	public Perimetre(final String _codeStructure, final String _codeRubrique, final String _codeProfil, final String _codeGroupe, final String _codeEspaceCollaboratif) {
		codeStructure = _codeStructure;
		codeRubrique = _codeRubrique;
		codeProfil = _codeProfil;
		codeGroupe = _codeGroupe;
		codeEspaceCollaboratif = _codeEspaceCollaboratif;
		identifiantNiveau = codeStructure + "/" + codeRubrique + "/" + codeGroupe + "/" + codeEspaceCollaboratif;
	}

	/**
	 * Formatage de la chaine sérialisée.
	 * 
	 * @return the chaine serialisee
	 */
	public String getChaineSerialisee() {
		String res = "";
		if (getCodeStructure().length() > 0) {
			res += getCodeStructure();
		}
		res += "/";
		if (getCodeRubrique().length() > 0) {
			res += getCodeRubrique();
		}
		res += "/";
		if (getCodeProfil().length() > 0) {
			res += getCodeProfil();
		}
		res += "/";
		if (getCodeGroupe().length() > 0) {
			res += getCodeGroupe();
		}
		res += "/";
		if (getCodeEspaceCollaboratif().length() > 0) {
			res += getCodeEspaceCollaboratif();
		}
		return res;
	}

	/**
	 * Sets the code groupe.
	 * 
	 * @param codeGroupe
	 *            The codeGroupe to set.
	 */
	public void setCodeGroupe(final String codeGroupe) {
		this.codeGroupe = codeGroupe;
	}

	/**
	 * Gets the code groupe.
	 * 
	 * @return Returns the codeGroupe.
	 */
	public String getCodeGroupe() {
		return codeGroupe;
	}

	/**
	 * Gets the code rubrique.
	 * 
	 * @return Returns the codeRubrique.
	 */
	public String getCodeRubrique() {
		return codeRubrique;
	}

	/**
	 * Gets the code structure.
	 * 
	 * @return Returns the codeStructure.
	 */
	public String getCodeStructure() {
		return codeStructure;
	}

	/**
	 * Gets the code profil.
	 * 
	 * @return Returns the codeProfil.
	 */
	public String getCodeProfil() {
		return codeProfil;
	}

	/**
	 * Controle si le périmètre courant est conforme à un périmètre de rôle
	 * 
	 * périmètre courant :
	 * 
	 * "" signifie vide (pas de valeur) "*" signifie 'champ à ignorer'
	 * 
	 * 
	 * périmètre du role
	 * 
	 * "" signifie tous (toutes valeurs autorisées) "-" signifie aucune valeur autorisée.
	 * 
	 * @param perimetre
	 *            the _perimetre
	 * 
	 * @return true, if est conforme au perimetre
	 *
	 */
	public boolean estConformeAuPerimetre(final Perimetre perimetre) {
		boolean estUnSousPerimetre = true;
		if (!"*".equals(getCodeStructure())) {
			if (StringUtils.isNotBlank(perimetre.getCodeStructure())) {
				estUnSousPerimetre = false;
				if ("-".equals(perimetre.getCodeStructure())) {
					estUnSousPerimetre = StringUtils.isEmpty(getCodeStructure());
				} else if (perimetre.getCodeStructure().equals(getCodeStructure())) {
					estUnSousPerimetre = true;
				} else {
					// Parcours des sous-structures
					final Iterator<InfosStructure> listeStructures = Structure.renvoyerItemStructure(perimetre.getCodeStructure()).getListeSousStructuresTousNiveaux().iterator();
					while (listeStructures.hasNext() && !estUnSousPerimetre) {
						if (listeStructures.next().getCode().equals(getCodeStructure())) {
							estUnSousPerimetre = true;
						}
					}
				}
			}
			if (!estUnSousPerimetre) {
				return false;
			}
		}
		if (!"*".equals(getCodeRubrique())) {
			if (StringUtils.isNotBlank(perimetre.getCodeRubrique())) {
				estUnSousPerimetre = false;
				if ("-".equals(perimetre.getCodeRubrique())) {
					estUnSousPerimetre = StringUtils.isEmpty(getCodeRubrique());
				} else {
					if (perimetre.getCodeRubrique().equals(getCodeRubrique())) {
						estUnSousPerimetre = true;
					} else {
						// Parcours des sous-rubriques
						final InfosRubriques rubriquePerimetre = Rubrique.renvoyerItemRubrique(perimetre.getCodeRubrique());
						final InfosRubriques rubriqueThis = Rubrique.renvoyerItemRubrique(getCodeRubrique());
						estUnSousPerimetre = rubriquePerimetre.contains(rubriqueThis);
					}
				}
			}
			if (!estUnSousPerimetre) {
				return false;
			}
		}
		if (!"*".equals(getCodeGroupe())) {
			if (StringUtils.isNotBlank(perimetre.getCodeGroupe())) {
				estUnSousPerimetre = false;
				if ("-".equals(perimetre.getCodeGroupe())) {
					estUnSousPerimetre = StringUtils.isEmpty(getCodeGroupe());
				} else {
					if (perimetre.getCodeGroupe().equals(getCodeGroupe())) {
						estUnSousPerimetre = true;
					} else {
						// Parcours des sous-groupes
						final InfosGroupeDsi groupePerimetre = Groupedsi.renvoyerItemGroupeDsi(perimetre.getCodeGroupe());
						final InfosGroupeDsi groupeThis = Groupedsi.renvoyerItemGroupeDsi(getCodeGroupe());
						estUnSousPerimetre = groupePerimetre.contains(groupeThis);
					}
				}
				if (!estUnSousPerimetre) {
					return false;
				}
			}
		}
		if (!"*".equals(getCodeEspaceCollaboratif())) {
			if (StringUtils.isNotBlank(perimetre.getCodeEspaceCollaboratif())) {
				if (!perimetre.getCodeEspaceCollaboratif().equals(getCodeEspaceCollaboratif())) {
					estUnSousPerimetre = false;
				}
			}
		}
		return estUnSousPerimetre;
	}

	/**
	 * Controle si le périmètre courant est un sous-périmètre du périmètre en paramètre
	 * 
	 * périmètre courant :
	 * 
	 * "" signifie tous (toutes valeurs autorisées) "-" signifie aucune valeur autorisée
	 * 
	 * 
	 * périmètre du role
	 * 
	 * "" signifie tous (toutes valeurs autorisées) "-" signifie aucune valeur autorisée.
	 * 
	 * @param _perimetre
	 *            the _perimetre
	 * @param _masque
	 *            the _masque
	 * 
	 * @return true, if est un sous perimetre de
	 *
	 */
	public boolean estUnSousPerimetreDe(final Perimetre _perimetre, final boolean _masque) {
		boolean estUnSousPerimetre = true;
		if ((!_masque) || (codeStructure.length() > 0)) {
			final String codeStructurePerimetre = _perimetre.getCodeStructure();
			if (codeStructurePerimetre.length() > 0) {
				estUnSousPerimetre = false;
				if (codeStructurePerimetre.equals("-")) {
					if (codeStructure.equals("-")) {
						estUnSousPerimetre = true;
					}
				} else if (codeStructure.equals(codeStructurePerimetre)) {
					estUnSousPerimetre = true;
				} else {
					// Parcours des sous-structures
					/* Iterator listeStructures = Structure.renvoyerItemStructure(_perimetre.getCodeStructure())
					 .getListeSousStructuresTousNiveaux().iterator();
					 while (listeStructures.hasNext()) {
					 if (((InfosStructure)listeStructures.next()).getCode().equals(getCodeStructure()))
					 estUnSousPerimetre = true;
					 }*/
					if (Structure.renvoyerItemStructure(codeStructurePerimetre).getListeSousStructuresTousNiveaux().contains(Structure.renvoyerItemStructure(codeStructure))) {
						estUnSousPerimetre = true;
					}
				}
			}
			if (!estUnSousPerimetre) {
				return false;
			}
		}
		if ((!_masque) || (codeRubrique.length() > 0)) {
			final String codeRubriquePerimetre = _perimetre.getCodeRubrique();
			if (codeRubriquePerimetre.length() > 0) {
				estUnSousPerimetre = false;
				if (codeRubriquePerimetre.equals("-")) {
					if (codeRubrique.equals("-")) {
						estUnSousPerimetre = true;
					}
				} else {
					if (codeRubrique.equals(codeRubriquePerimetre)) {
						estUnSousPerimetre = true;
					} else {
						// Parcours des sous-rubriques
						final InfosRubriques rubriquePerimetre = Rubrique.renvoyerItemRubrique(codeRubriquePerimetre);
						final InfosRubriques rubriqueThis = Rubrique.renvoyerItemRubrique(codeRubrique);
						estUnSousPerimetre = rubriquePerimetre.contains(rubriqueThis);
					}
				}
			}
			if (!estUnSousPerimetre) {
				return false;
			}
		}
		if ((!_masque) || (codeGroupe.length() > 0)) {
			final String codeGroupePerimetre = _perimetre.getCodeGroupe();
			if (codeGroupePerimetre.length() > 0) {
				estUnSousPerimetre = false;
				if (codeGroupePerimetre.equals("-")) {
					if (codeGroupe.equals("-")) {
						estUnSousPerimetre = true;
					}
				} else {
					if (codeGroupe.equals(codeGroupePerimetre)) {
						estUnSousPerimetre = true;
					} else {
						// Parcours des sous-groupes
						final InfosGroupeDsi groupePerimetre = Groupedsi.renvoyerItemGroupeDsi(codeGroupePerimetre);
						final InfosGroupeDsi groupeThis = Groupedsi.renvoyerItemGroupeDsi(codeGroupe);
						estUnSousPerimetre = groupePerimetre.contains(groupeThis);
					}
				}
			}
		}
		if ((!_masque) || (getCodeEspaceCollaboratif().length() > 0)) {
			if (_perimetre.getCodeEspaceCollaboratif().length() > 0) {
				if (!getCodeEspaceCollaboratif().equals(_perimetre.getCodeEspaceCollaboratif())) {
					estUnSousPerimetre = false;
				}
			}
		}
		return estUnSousPerimetre;
	}

	/**
	 * Controle si le périmètre courant est conforme à un périmètre de rôle
	 * 
	 * périmètre courant :
	 * 
	 * "" signifie vide (pas de valeur)
	 * 
	 * 
	 * périmètre du role
	 * 
	 * "" signifie tous (toutes valeurs autorisées) "-" signifie aucune valeur autorisée.
	 * 
	 * @param _perimetre
	 *            the _perimetre
	 * 
	 * @return true, if est partiellement conforme au perimetre
	 *
	 */
	public boolean estPartiellementConformeAuPerimetre(final Perimetre _perimetre) {
		boolean estUnSousPerimetre = false;
		if (getCodeStructure().length() > 0) {
			if (_perimetre.getCodeStructure().length() > 0) {
				// JSS 20051104 : gestion des périmètres
				if (!_perimetre.getCodeStructure().equals("-")) {
					if (getCodeStructure().equals(_perimetre.getCodeStructure())) {
						estUnSousPerimetre = true;
					} else {
						// Parcours des sous-structures
						for (InfosStructure infosStructure : Structure.renvoyerItemStructure(_perimetre.getCodeStructure()).getListeSousStructuresTousNiveaux()) {
							if (infosStructure.getCode().equals(getCodeStructure())) {
								estUnSousPerimetre = true;
							}
						}
					}
				}
			} else {
				estUnSousPerimetre = true;
			}
		}
		if (estUnSousPerimetre) {
			return true;
		}
		if (getCodeRubrique().length() > 0) {
			if (_perimetre.getCodeRubrique().length() > 0) {
				// JSS 20051104 : gestion des périmètres
				if (!_perimetre.getCodeRubrique().equals("-")) {
					if (getCodeRubrique().equals(_perimetre.getCodeRubrique())) {
						estUnSousPerimetre = true;
					} else {
						// Parcours des sous-rubriques
						final InfosRubriques rubriquePerimetre = Rubrique.renvoyerItemRubrique(_perimetre.getCodeRubrique());
						final InfosRubriques rubriqueThis = Rubrique.renvoyerItemRubrique(getCodeRubrique());
						estUnSousPerimetre = rubriquePerimetre.contains(rubriqueThis);
					}
				}
			} else {
				estUnSousPerimetre = true;
			}
		}
		if (estUnSousPerimetre) {
			return true;
		}
		if (getCodeGroupe().length() > 0) {
			if (_perimetre.getCodeGroupe().length() > 0) {
				// JSS 20051104 : gestion des périmètres
				if (!_perimetre.getCodeGroupe().equals("-")) {
					if (getCodeGroupe().equals(_perimetre.getCodeGroupe())) {
						estUnSousPerimetre = true;
					} else {
						// Parcours des sous-groupes
						final InfosGroupeDsi groupePerimetre = Groupedsi.renvoyerItemGroupeDsi(_perimetre.getCodeGroupe());
						final InfosGroupeDsi groupeThis = Groupedsi.renvoyerItemGroupeDsi(getCodeGroupe());
						estUnSousPerimetre = groupePerimetre.contains(groupeThis);
					}
				}
			} else {
				estUnSousPerimetre = true;
			}
		}
		if (getCodeEspaceCollaboratif().length() > 0) {
			if (_perimetre.getCodeEspaceCollaboratif().length() > 0) {
				if (getCodeEspaceCollaboratif().equals(_perimetre.getCodeEspaceCollaboratif())) {
					estUnSousPerimetre = true;
				}
			} else {
				estUnSousPerimetre = true;
			}
		}
		return estUnSousPerimetre;
	}

	/**
	 * Calcul du périmètre d'une permission . COnsiste à appliquer le masque de l'affectation sur le périmètre du role
	 * 
	 * @param _perimetreAffectation
	 *            the _perimetre affectation
	 * @param _perimetreRole
	 *            the _perimetre role
	 * 
	 * @return the perimetre
	 *
	 */
	public static Perimetre calculerPerimetrePermission(final Perimetre _perimetreAffectation, final Perimetre _perimetreRole) {
		boolean isRubOk = false;
		boolean isStructOk = false;
		String codeStructure = _perimetreRole.getCodeStructure();
		if (_perimetreAffectation.getCodeStructure().length() > 0 && !_perimetreAffectation.getCodeStructure().equals("-")) {
			codeStructure = _perimetreAffectation.getCodeStructure();
			// RP20070707 controle de l'existance des données du périmètre
			// si une des données n'existe pas on ignore le périmètre et donc le rôle associé
			final InfosStructure infosStruct = Structure.renvoyerItemStructure(codeStructure);
			if (infosStruct.getCode().length() > 0) {
				isStructOk = true;
			}
		}
		String codeRubrique = _perimetreRole.getCodeRubrique();
		if (_perimetreAffectation.getCodeRubrique().length() > 0 && !_perimetreAffectation.getCodeRubrique().equals("-")) {
			codeRubrique = _perimetreAffectation.getCodeRubrique();
			// RP20070707 controle de l'existance des données du périmètre
			// si une des données n'existe pas on ignore le périmètre et donc le rôle associé
			final InfosRubriques infosRub = Rubrique.renvoyerItemRubrique(codeRubrique);
			if (infosRub.getCode().length() > 0) {
				isRubOk = true;
			}
		}
		String codeGroupe = _perimetreRole.getCodeGroupe();
		if (_perimetreAffectation.getCodeGroupe().length() > 0) {
			codeGroupe = _perimetreAffectation.getCodeGroupe();
		}
		String codeProfil = _perimetreRole.getCodeProfil();
		if (_perimetreAffectation.getCodeProfil().length() > 0) {
			codeProfil = _perimetreAffectation.getCodeProfil();
		}
		String codeEspaceCollabortif = _perimetreRole.getCodeEspaceCollaboratif();
		if (_perimetreAffectation.getCodeEspaceCollaboratif().length() > 0) {
			codeEspaceCollabortif = _perimetreAffectation.getCodeEspaceCollaboratif();
		}
		// RP20070707 controle de l'existance des données du périmètre
		// si une des données n'existe pas on ignore le périmètre et donc le rôle associé
		if (!isRubOk) {
			final InfosRubriques infosRub = Rubrique.renvoyerItemRubrique(codeRubrique);
			if (codeRubrique.length() > 0 && infosRub.getCode().length() == 0 && !codeRubrique.equals("-")) {
				return null;
			}
		}
		if (!isStructOk) {
			final InfosStructure infosStruct = Structure.renvoyerItemStructure(codeStructure);
			if (codeStructure.length() > 0 && infosStruct.getCode().length() == 0 && !codeStructure.equals("-")) {
				return null;
			}
		}
		return new Perimetre(codeStructure, codeRubrique, codeProfil, codeGroupe, codeEspaceCollabortif);
	}

	/**
	 * Sets the code profil.
	 * 
	 * @param codeProfil
	 *            The codeProfil to set.
	 */
	public void setCodeProfil(final String codeProfil) {
		this.codeProfil = codeProfil;
	}

	/**
	 * Gets the code espace collaboratif.
	 * 
	 * @return Returns the codeEspaceCollaboratif.
	 */
	public String getCodeEspaceCollaboratif() {
		return codeEspaceCollaboratif;
	}

	/**
	 * Sets the code espace collaboratif.
	 * 
	 * @param codeEspaceCollaboratif
	 *            The codeEspaceCollaboratif to set.
	 */
	public void setCodeEspaceCollaboratif(final String codeEspaceCollaboratif) {
		this.codeEspaceCollaboratif = codeEspaceCollaboratif;
	}

	/**
	 * Equals.
	 * 
	 * @param perimetre
	 *            the perimetre
	 * 
	 * @return true, if successful
	 */
	public boolean equals(final Perimetre perimetre) {
		return getChaineSerialisee().equals(perimetre.getChaineSerialisee());
	}

	/**
	 * Gets the identifiant niveau.
	 * 
	 * @return Returns the identifiantNiveau.
	 */
	public String getIdentifiantNiveau() {
		return identifiantNiveau;
	}

	/**
	 * Méthode toString
	 * 
	 * @return toString
	 */
	@Override
	public String toString() {
		return getChaineSerialisee();
	}
}
