/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;
import java.util.Vector;

import org.apache.commons.lang3.StringUtils;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.database.OMContext;
import com.kportal.core.config.MessageHelper;
import com.univ.objetspartages.cache.CacheProfilDsiManager;
import com.univ.objetspartages.sgbd.ProfildsiDB;
import com.univ.utils.Chaine;
import com.univ.utils.sql.RequeteSQL;
import com.univ.utils.sql.clause.ClauseOrderBy;
import com.univ.utils.sql.clause.ClauseOrderBy.SensDeTri;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;

/**
 * The Class Profildsi.
 */
public class Profildsi extends ProfildsiDB implements Cloneable {

	/**
	 * Inits the.
	 */
	public void init() {
		setIdProfildsi((long) 0);
		setCode(String.valueOf(System.currentTimeMillis()));
		setLibelle("");
		setCodeRubriqueAccueil("");
		setRoles("");
		setGroupes("");
		setCodeRattachement("");
	}

	public static Map<String, InfosProfilDsi> getListeInfosProfilDsi() {
		CacheProfilDsiManager cache = (CacheProfilDsiManager) ApplicationContextManager.getCoreContextBean(CacheProfilDsiManager.ID_BEAN);
		return cache.getListeProfilsDsi();
	}

	@Deprecated
	public static String getIntitule(OMContext _ctx, String _code) {
		return getIntitule(_code);
	}

	/**
	 * Récupération de l'intitulé.
	 *
	 * @param code
	 *            the code
	 * 
	 * @return the intitule
	 *
	 */
	public static String getIntitule(final String code) {
		if (code == null || code.length() == 0) {
			return MessageHelper.getCoreMessage("BO_PROFIL_INEXISTANT");
		}
		return StringUtils.defaultIfBlank(renvoyerItemProfilDsi(code).getIntitule(), MessageHelper.getCoreMessage("BO_PROFIL_INEXISTANT"));
	}

	/**
	 * Renvoie la liste de tous les profils DSI (id+libellé) Peut être affiché directement dans une Combo
	 * 
	 * @param _ctx
	 *            the _ctx
	 * 
	 * @return the liste id profils dsi
	 * 
	 * @throws Exception
	 *             the exception
	 * @deprecated Méthode plus utilisée
	 */
	@Deprecated
	public static Hashtable<String, String> getListeIDProfilsDSI(OMContext _ctx) throws Exception {
		Profildsi profildsi = new Profildsi();
		profildsi.setCtx(_ctx);
		profildsi.init();
		Hashtable<String, String> h = new Hashtable<>();
		if (profildsi.select("") > 0) {
			while (profildsi.nextItem()) {
				h.put(profildsi.getIdProfildsi().toString(), profildsi.getLibelle());
			}
		}
		return h;
	}

	/**
	 * Renvoie la liste de tous les profils DSI (id+libellé) Peut être affiché directement dans une Combo
	 * 
	 * @param ctx
	 *            the _ctx
	 * 
	 * @return the liste id profils dsi
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public static Collection<Profildsi> getListeProfilsDSI(OMContext ctx) throws Exception {
		Profildsi profildsi = new Profildsi();
		profildsi.setCtx(ctx);
		profildsi.init();
		Collection<Profildsi> toutLesProfils = new ArrayList<>();
		if (profildsi.select("") > 0) {
			while (profildsi.nextItem()) {
				toutLesProfils.add(profildsi.clone());
			}
		}
		return toutLesProfils;
	}

	/**
	 * Renvoie la liste de tous les profils DSI (code+libellé) Peut être affiché directement dans une Combo.
	 * 
	 * @return the liste profils dsi
	 *
	 */
	public static Hashtable<String, String> getListeProfilsDSI() {
		Map<String, InfosProfilDsi> infosProfils = getListeInfosProfilDsi();
		Hashtable<String, String> res = new Hashtable<>(infosProfils.size());
		for (InfosProfilDsi info : infosProfils.values()) {
			res.put(info.getCode(), info.getIntitule());
		}
		return res;
	}

	/**
	 * Renvoyer profils groupes.
	 * 
	 * @param _groupesUtilisateur
	 *            the _groupes utilisateur
	 * 
	 * @return the vector
	 *
	 */
	public static Vector<String> renvoyerProfilsGroupes(Vector<String> _groupesUtilisateur) {
		TreeSet<String> listeProfilsUser = new TreeSet<>();
		Enumeration<String> eProfil = Profildsi.getListeProfilsDSI().keys();
		while (eProfil.hasMoreElements()) {
			String codeProfil = eProfil.nextElement();
			InfosProfilDsi infosProfil = Profildsi.renvoyerItemProfilDsi(codeProfil);
			Vector<String> vGroupesProfil = infosProfil.getGroupes();
			Enumeration<String> eGroupeProfil = vGroupesProfil.elements();
			// On regarde si l'utilisateur appartient à un des groupes du profil
			boolean profil = false;
			while (eGroupeProfil.hasMoreElements() && !profil) {
				InfosGroupeDsi infosGroupeProfil = Groupedsi.renvoyerItemGroupeDsi(eGroupeProfil.nextElement());
				Enumeration<String> eGroupeUser = _groupesUtilisateur.elements();
				while (eGroupeUser.hasMoreElements() && !profil) {
					InfosGroupeDsi groupeUser = Groupedsi.renvoyerItemGroupeDsi(eGroupeUser.nextElement());
					profil = infosGroupeProfil.contains(groupeUser);
				}
			}
			if (profil) {
				listeProfilsUser.add(infosProfil.getCode());
			}
		}
		return new Vector<>(listeProfilsUser);
	}

	/**
	 * Renvoie la liste de tous les profils pour une liste de groupes.
	 * 
	 * @param _groupes
	 *            the _groupes
	 * 
	 * @return the liste profils dsi par groupes
	 *
	*/
	public static Hashtable<String, String> getListeProfilsDSIParGroupes(Vector<String> _groupes) {
		Vector<String> listeProfils = renvoyerProfilsGroupes(_groupes);
		Hashtable<String, String> hListeProfils = new Hashtable<>();
		Enumeration<String> e = listeProfils.elements();
		while (e.hasMoreElements()) {
			String codeProfil = e.nextElement();
			hListeProfils.put(codeProfil, Profildsi.renvoyerItemProfilDsi(codeProfil).getIntitule());
		}
		return hListeProfils;
	}

	/**
	 * Récupération d'un profil stocké en mémoire.
	 * 
	 * @param code
	 *            the code
	 * 
	 * @return the infos profil dsi
	 *
	 */
	public static InfosProfilDsi renvoyerItemProfilDsi(String code) {
		InfosProfilDsi res = getListeInfosProfilDsi().get(code);
		/* groupe vide pour éviter les plantages */
		if (res == null) {
			res = new InfosProfilDsi();
		}
		return res;
	}

	/**
	 * Renvoie la liste des {profils/périmètres} associé à un role sous forme de Hashtable.
	 * 
	 * @param _ctx
	 *            the _ctx
	 * @param _role
	 *            the _role
	 * @param _codesStructures
	 *            the _codes structures
	 * @param _codeRubrique
	 *            the _code rubrique
	 * @param _publicsVises
	 *            the _publics vises
	 * @param _codeEspaceCollaboratif
	 *            the _code espace collaboratif
	 * 
	 * @return the hashtable
	 * 
	 * @throws Exception
	 *             the exception
	 * @deprecated Méthode plus utilisée
	 */
	@Deprecated
	public static Hashtable<String, Vector<Perimetre>> renvoyerProfilsEtPerimetres(OMContext _ctx, String _role, List<String> _codesStructures, String _codeRubrique,
		String _publicsVises, String _codeEspaceCollaboratif) throws Exception {
		Hashtable<String, Vector<Perimetre>> h = new Hashtable<>();
		for (InfosProfilDsi info : getListeInfosProfilDsi().values()) {
			Vector<Perimetre> v = Role.renvoyerPerimetresAffectation(info.getRoles(), _role, _codesStructures, _codeRubrique, _publicsVises, _codeEspaceCollaboratif);
			if (v.size() > 0) {
				h.put(info.getCode(), v);
			}
		}
		return h;
	}

	/**
	 * Select.
	 * 
	 * @param code
	 *            the _code
	 * @param libelle
	 *            the _libelle
	 * 
	 * @return the int
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public int select(String code, String libelle) throws Exception {
		RequeteSQL requeteCodeLibelle = new RequeteSQL();
		ClauseOrderBy orderBy = new ClauseOrderBy("LIBELLE", SensDeTri.ASC);
		ClauseWhere where = new ClauseWhere();
		if (StringUtils.isNotEmpty(code)) {
			where.setPremiereCondition(ConditionHelper.egalVarchar("CODE", code));
		}
		if (StringUtils.isNotEmpty(libelle)) {
			where.and(ConditionHelper.like("LIBELLE", libelle, "%", "%"));
		}
		requeteCodeLibelle.where(where).orderBy(orderBy);
		return select(requeteCodeLibelle.formaterRequete());
	}

	/**
	 * Sets the vecteur groupes.
	 * 
	 * @param v
	 *            the new vecteur groupes
	 *
	 */
	public void setVecteurGroupes(Vector<String> v){
		setGroupes(Chaine.convertirAccolades(v));
	}

	/**
	 * Gets the vecteur groupes.
	 * 
	 * @return the vecteur groupes
	 *
	 */
	public Vector<String> getVecteurGroupes() {
		return Chaine.getVecteurAccolades(getGroupes());
	}

	@Override
	public Profildsi clone() throws CloneNotSupportedException {
		return (Profildsi) super.clone();
	}
}
