/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.Vector;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.InfoBean;
import com.jsbsoft.jtf.database.OMContext;
import com.univ.utils.ContexteDao;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;

/**
 * Classe absraite definissant la gestion de groupe dynamique et notamment la prise en charge de cache.
 * 
 * Pour le cache, on utilise la source d'import de la table Groupeutilisateur pour identifier ce qui est genere par la requete.
 * 
 * PCO 02-02-2009 : Patch pour rendre plus générique et évolutif la gestion des groupes dynamiques.
 * 
 * @author jean-seb, jbiard
 */
public abstract class RequeteGroupeDynamique {

	private static final Logger LOG = LoggerFactory.getLogger(RequeteGroupeDynamique.class);

	/** The Constant SYNCHRO_ACTION_AJOUT_MODIFICATION. */
	public final static int SYNCHRO_ACTION_AJOUT_MODIFICATION = 0;

	/** The Constant SYNCHRO_ACTION_SUPPRESSION. */
	public final static int SYNCHRO_ACTION_SUPPRESSION = 1;

	/** The ctx. */
	@Deprecated
	protected OMContext ctx;

	/** The delai rechgt resolution groupes ut. */
	private long delaiRechgtResolutionGroupesUt;

	private String nomRequete;

	/**
	 * Positionne le contexte.
	 * 
	 * @param ctx
	 *            contexte
	 * @deprecated Méthode plus nécessaire
	 */
	@Deprecated
	public void setCtx(final OMContext ctx) {}

	public void setNomRequete(final String nomRequete) {
		this.nomRequete = nomRequete;
	}

	public String getNomRequete() {
		return nomRequete;
	}

	/**
	 * Positionne le delai d'expiration du cache des groupes de l'utilisateur.
	 * 
	 * @param delaiRechgtResolutionGroupesUt
	 *            delai en minutes
	 */
	public void setDelaiRechargementResolutionGroupesUtilisateur(final long delaiRechgtResolutionGroupesUt) {
		this.delaiRechgtResolutionGroupesUt = delaiRechgtResolutionGroupesUt;
	}

	/**
	 * Prend en charge la resolution code utilisateur -> groupes. Gere un cache si le delai de rechargement est different de 0.
	 * 
	 * @param codeUtilisateur
	 *            code de l'utilisateur dont on souhaite recuperer les groupes
	 * @param ts
	 *            the ts
	 * 
	 * @return codes des groupes rattaches
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public Vector<String> getGroupesUtilisateurCache(final String codeUtilisateur, final long ts) throws Exception {
		final String sourceImport = "[" + getNomRequete() + "]";
		if (delaiRechgtResolutionGroupesUt == 0) {
			LOG.debug("*** Cache desactive pour " + sourceImport);
			// cache desactive
			return getGroupesUtilisateur(codeUtilisateur);
		}
		// cache active		
		final Utilisateur utilisateur = new Utilisateur();
		utilisateur.init();
		try (ContexteDao ctx = new ContexteDao()) {
			utilisateur.setCtx(ctx);
			if (StringUtils.isEmpty(codeUtilisateur) || utilisateur.selectParCode(codeUtilisateur) == 0) {
				return new Vector<String>(0);
			}
			utilisateur.nextItem();
		}
		// lecture des relations relatives a la requete
		// on etablit le timestamp
		final long tsMin = ts / 60000;
		final Set<String> setGroupesCache = new HashSet<>();
		final Groupeutilisateur groupeUtilisateur = new Groupeutilisateur();
		groupeUtilisateur.init();
		try (ContexteDao ctx = new ContexteDao()) {
			groupeUtilisateur.setCtx(ctx);
			// associations User - Groupe cachees par les chargements de groupes
			final ClauseWhere whereCodeUserAndSource = new ClauseWhere(ConditionHelper.egalVarchar("CODE_UTILISATEUR", codeUtilisateur));
			whereCodeUserAndSource.and(ConditionHelper.egalVarchar("SOURCE_IMPORT", sourceImport));
			if (groupeUtilisateur.select(whereCodeUserAndSource.formaterSQL()) != 0) {
				// on recupere tous les groupes en cache rattaches a l'utilisateur
				while (groupeUtilisateur.nextItem()) {
					setGroupesCache.add(groupeUtilisateur.getCodeGroupe());
				}
			}
		}
		// recuperation du timestamp du dernier chargement des groupes rattaches
		final long timestamp = utilisateur.getTsCacheGroupes();
		// verification si on doit recharger
		if ((timestamp != tsMin) && (timestamp + delaiRechgtResolutionGroupesUt >= tsMin)) {
			LOG.debug("*** Cache encore valable pour " + codeUtilisateur + sourceImport);
			// rechargement : lecture du cache en BD
			// on remonte les groupes de l'utilisateur (il peut n'y en avoir aucun)
			return new Vector<>(setGroupesCache);
		}
		LOG.debug("*** Cache expire pour " + codeUtilisateur + sourceImport);
		LOG.debug("*** Création d'un cache utilisateur pour " + codeUtilisateur);
		// on doit effectuer le chargement
		final Vector<String> listeGroupes = getGroupesUtilisateur(codeUtilisateur);
		// comparaison entre ce qui a ete lu en cache et ce qui vient d etre charge		
		if (!setGroupesCache.isEmpty()) {
			String codeGroupe;
			// invalidation des groupes en cache qui n'ont pas de lien de parente avec ceux calcules			
			for (final Iterator<String> itCodeGroupeCharge = listeGroupes.iterator(); itCodeGroupeCharge.hasNext() && !setGroupesCache.isEmpty();) {
				codeGroupe = itCodeGroupeCharge.next();
				// l'utilisateur fait il maintenant partie d'un nouveau groupe qui a ete mis en cache ?
				if (!setGroupesCache.remove(codeGroupe)) {
					if (!supprimeGroupesHierarchie(codeGroupe, setGroupesCache)) {
						// le groupe n'a pas pu etre supprime des groupes en cache et ne fait pas partie de la hierarchie d'un groupe en cache
						// => on l'invalide
						invalideCacheGroupe(codeGroupe, sourceImport);
					}
				} else {
					supprimeGroupesHierarchie(codeGroupe, setGroupesCache);
				}
			}
			// reste-t-il des groupes en cache 			
			if (!setGroupesCache.isEmpty()) {
				// s'il reste des groupes en cache auquel apppartient l'utilisateur, cela signifie que l'utilisateur
				// n'en fait plus partie => on invalide le groupe
				for (final String string : setGroupesCache) {
					invalideCacheGroupe(string, sourceImport);
				}
			}
		}
		miseEnCacheGroupesUtilisateurCache(utilisateur, listeGroupes, tsMin, sourceImport);
		return listeGroupes;
	}

	/**
	 * Invalide un groupe et ses parents.
	 * 
	 * @param codeGroupeCache
	 *            code du groupe en cache
	 * @param sourceImport
	 *            the source import
	 * 
	 * @throws Exception
	 *             the exception
	 */
	private void invalideCacheGroupe(final String codeGroupeCache, final String sourceImport) throws Exception {
		final Long tsInit = new Long(0);
		try (ContexteDao ctx = new ContexteDao()) {
			final Groupedsi groupeDsi = new Groupedsi();
			groupeDsi.init();
			groupeDsi.setCtx(ctx);
			if (groupeDsi.select(codeGroupeCache, "", "", "", "1") != 0) {
				groupeDsi.nextItem();
				groupeDsi.setDerniereMajCache(tsInit);
				groupeDsi.update();
			}
		}
		Groupeutilisateur.deleteParGroupeSourceImport(codeGroupeCache, sourceImport);
		LOG.debug("*** on invalide le cache du groupe " + codeGroupeCache);
		InfosGroupeDsi infosGroupe = Groupedsi.renvoyerItemGroupeDsi(codeGroupeCache);
		String codeGroupePere;
		// on invalide les groupes parents
		try (ContexteDao ctx = new ContexteDao()) {
			for (int niveau = infosGroupe.getNiveau(); niveau > 0; --niveau) {
				codeGroupePere = infosGroupe.getCodeGroupePere();
				Groupeutilisateur.deleteParGroupeSourceImport(codeGroupePere, sourceImport);
				infosGroupe = Groupedsi.renvoyerItemGroupeDsi(codeGroupePere);
				final Groupedsi groupeDsi = new Groupedsi();
				groupeDsi.init();
				groupeDsi.setCtx(ctx);
				if (groupeDsi.select(codeGroupePere, "", "", "", "1") != 0) {
					groupeDsi.nextItem();
					groupeDsi.setDerniereMajCache(tsInit);
 					groupeDsi.update();
				}
			}
		}
	}

	/**
	 * Supprime de l'ensemble des groupes en cache ceux qui sont dans la hierarchie.
	 * 
	 * @param codeGroupe
	 *            the code groupe
	 * @param setGroupesCache
	 *            the set groupes cache
	 * 
	 * @return true, if supprime groupes hierarchie
	 * 
	 * @throws Exception
	 *             the exception
	 */
	private boolean supprimeGroupesHierarchie(final String codeGroupe, final Set<String> setGroupesCache) throws Exception {
		boolean groupeSupprime = false;
		// recherche dans les sous-groupes
		InfosGroupeDsi infosGroupe = Groupedsi.renvoyerItemGroupeDsi(codeGroupe);
		/*		List listeSousGroupes = infosGroupe.getSousGroupesTousNiveaux( ) ;
				
				// on supprime de la hierarchie les sous-groupes
				for( Iterator itSousGroupe = listeSousGroupes.iterator() ; itSousGroupe.hasNext( ) ; ) {
					groupeSupprime = setGroupesCache.remove( itSousGroupe.next( ) ) ;
				}
		*/
		// recherche dans les groupes parents
		String codeGroupePere = null;
		// on supprime de la hierarchie les groupes peres
		for (int niveau = infosGroupe.getNiveau(); niveau > 0; --niveau) {
			codeGroupePere = infosGroupe.getCodeGroupePere();
			infosGroupe = Groupedsi.renvoyerItemGroupeDsi(codeGroupePere);
			groupeSupprime = setGroupesCache.remove(codeGroupePere);
			if (groupeSupprime) {
				LOG.debug("*** Suppression du groupe pere " + codeGroupePere);
			}
		}
		return groupeSupprime;
	}

	/**
	 * Met en cache les groupes de l'utilisateur.
	 * 
	 * @param listeGroupes
	 *            liste des groupes rattaches
	 * @param tsMin
	 *            timestamp en min pour le cache
	 * @param sourceImport
	 *            source import pour la requete
	 * @param utilisateur
	 *            the utilisateur
	 * 
	 * @throws Exception
	 *             the exception
	 */
	private void miseEnCacheGroupesUtilisateurCache(final Utilisateur utilisateur, final Vector<String> listeGroupes, final Long tsMin, final String sourceImport) throws Exception {
		if (!listeGroupes.isEmpty()) {
			// ajout des nouvelles relations avec timestamp
			try (ContexteDao ctx = new ContexteDao()) {
				final String codeUtilisateur = utilisateur.getCode();
				final Groupeutilisateur groupeutilisateur = new Groupeutilisateur();
				groupeutilisateur.setCtx(ctx);
				for (String codeGroupe : listeGroupes) {
					final ClauseWhere whereCodeUserSourceEtGroupe = new ClauseWhere(ConditionHelper.egalVarchar("CODE_UTILISATEUR", codeUtilisateur));
					whereCodeUserSourceEtGroupe.and(ConditionHelper.egalVarchar("SOURCE_IMPORT", sourceImport));
					whereCodeUserSourceEtGroupe.and(ConditionHelper.egalVarchar("CODE_GROUPE", codeGroupe));
					// on ajoute les couples que s'ils n'existent pas deja
					if (groupeutilisateur.select(whereCodeUserSourceEtGroupe.formaterSQL()) == 0) {
						groupeutilisateur.init();
						groupeutilisateur.setCodeUtilisateur(codeUtilisateur);
						groupeutilisateur.setCodeGroupe(codeGroupe);
						groupeutilisateur.setSourceImport(sourceImport);
						groupeutilisateur.add();
					}
				}
				// maj timestamp au niveau de l'utilisateur
				utilisateur.setCtx(ctx);
				utilisateur.setTsCacheGroupes(tsMin);
				utilisateur.update();
			}
		}
	}

	/**
	 * Renvoie les groupes d'un utilisateur.
	 * 
	 * @param codeUtilisateur
	 *            code de l'utilisateur
	 * 
	 * @return codes des groupes rattaches
	 * 
	 * @throws Exception
	 *             the exception
	 */
	protected abstract Vector<String> getGroupesUtilisateur(String codeUtilisateur) throws Exception;

	/**
	 * Renvoie les utilisateurs correspondant a une requete de groupe avec prise en charge du cache.
	 * 
	 * @param codeGroupe
	 *            the requete groupe
	 * @param setCodesGroupesCache
	 *            the set codes groupes cache
	 * 
	 * @return the vecteur utilisateurs cache
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public Vector<String> getVecteurUtilisateursCache(final String codeGroupe, final Set<String> setCodesGroupesCache) throws Exception {
		return getVecteurUtilisateursCache(codeGroupe, setCodesGroupesCache, System.currentTimeMillis());
	}

	/**
	 * Renvoie les utilisateurs correspondant a une requete de groupe avec prise en charge du cache.
	 * 
	 * @param codeGroupe
	 *            the requete groupe
	 * @param setCodesGroupesCache
	 *            the set codes groupes cache
	 * @param ts
	 *            the ts
	 * 
	 * @return the vecteur utilisateurs cache
	 * 
	 * @throws Exception
	 *             the exception
	 */
	private Vector<String> getVecteurUtilisateursCache(final String codeGroupe, final Set<String> setCodesGroupesCache, final long ts) throws Exception {
		Vector<String> listeCodesUtilisateurs = null;
		if (! estGereParReqDyn(codeGroupe)) {
			listeCodesUtilisateurs = new Vector<>(0);
		}
		else {
			final Groupedsi groupeDsi = new Groupedsi();
			try (ContexteDao ctx = new ContexteDao()) {
				groupeDsi.init();
				groupeDsi.setCtx(ctx);
				if (groupeDsi.select(codeGroupe, "", "", "", "1") == 0) {
					return getVecteurUtilisateurs(codeGroupe);
				} else {
					groupeDsi.nextItem();
				}
			}
			final long tsMin = ts / 60000; // timestamp stocke en minutes
			final String sourceImport = "[" + getNomRequete() + "]";
			if (groupeDsi.getDelaiExpirationCache().longValue() + groupeDsi.getDerniereMajCache().longValue() >= tsMin) {
				LOG.debug("*** Cache encore valable pour " + codeGroupe);
				if (setCodesGroupesCache == null) {
					return Groupeutilisateur.getVecteurUtilisateursSourceImport(codeGroupe, sourceImport);
				}
			} else {
				LOG.debug("*** Cache expire pour " + codeGroupe);
				Set<String> setUtilisateursCache = null;
				// lecture du cache relation utilisateur => groupes
				try (ContexteDao ctx = new ContexteDao()) {
					final Groupeutilisateur groupeUtilisateur = new Groupeutilisateur();
					groupeUtilisateur.init();
					groupeUtilisateur.setCtx(ctx);
					setUtilisateursCache = new HashSet<>();
					// on prend dans le lot le groupe et les sous-groupes
					final ClauseWhere whereGroupeSource = new ClauseWhere(ConditionHelper.like("CODE_GROUPE", codeGroupe, "", "%"));
					whereGroupeSource.and(ConditionHelper.egalVarchar("SOURCE_IMPORT", sourceImport));
					if (groupeUtilisateur.select(whereGroupeSource.formaterSQL()) != 0) {
						while (groupeUtilisateur.nextItem()) {
							setUtilisateursCache.add(groupeUtilisateur.getCodeUtilisateur());
						}
					}
				}
				// rechargement necessaire
				listeCodesUtilisateurs = getVecteurUtilisateurs(codeGroupe);
				if ((setUtilisateursCache != null) && !setUtilisateursCache.isEmpty()) {
					String codeUtilisateur;
					for (final String string : listeCodesUtilisateurs) {
						codeUtilisateur = string;
						if (!setUtilisateursCache.remove(codeUtilisateur)) {
							// l'utilisateur apparait maintenant dans le groupe => on invalide son cache
							invalideCacheUtilisateur(codeUtilisateur);
						}
					}
					if (setUtilisateursCache.size() > 0) {
						for (final String string : setUtilisateursCache) {
							// l'utilisateur a ete supprime du groupe => on invalide son cache
							invalideCacheUtilisateur(string);
						}
					}
				}
				miseEnCacheUtilisateursGroupes(groupeDsi, listeCodesUtilisateurs, sourceImport, new Long(tsMin));
			}
			if (setCodesGroupesCache != null) {
				setCodesGroupesCache.add(groupeDsi.getCode());
			}
		}
		return listeCodesUtilisateurs;
	}

	/**
	 * Invalide le cache d'un utilisateur.
	 * 
	 * @param codeUtilisateur
	 *            code de l'utilisateur
	 * 
	 * @throws Exception
	 *             the exception
	 */
	private void invalideCacheUtilisateur(final String codeUtilisateur) throws Exception {
		try (ContexteDao ctx = new ContexteDao()) {
			final Utilisateur utilisateur = new Utilisateur();
			utilisateur.init();
			utilisateur.setCtx(ctx);
			utilisateur.selectParCode(codeUtilisateur);
			utilisateur.nextItem();
			utilisateur.setTsCacheGroupes(new Long(0));
			utilisateur.update();
		}
		LOG.debug("*** Cache de l'utilisateur " + codeUtilisateur + " invalide.");
	}

	/**
	 * Met en cache les groupes de l'utilisateur.
	 * 
	 * @param groupeDsi
	 *            groupe a mettre en cache
	 * @param listeCodesUtilisateurs
	 *            utilisateurs rattaches
	 * @param sourceImport
	 *            source de l'import de la requete
	 * @param tsMin
	 *            timestamp
	 * 
	 * @throws Exception
	 *             the exception
	 */
	private void miseEnCacheUtilisateursGroupes(final Groupedsi groupeDsi, final Vector<String> listeCodesUtilisateurs, final String sourceImport, final Long tsMin)
		throws Exception {
		// suppression de l'ancien cache
		Groupeutilisateur.deleteParGroupeSourceImport(groupeDsi.getCode(), sourceImport);
		try (ContexteDao ctx = new ContexteDao()) {
			if (!listeCodesUtilisateurs.isEmpty()) {
				// maj du cache : ajout des nouvelles relations avec timestamp
				final Groupeutilisateur groupeutilisateur = new Groupeutilisateur();
				groupeutilisateur.setCtx(ctx);
				for (String codeUtilisateur : listeCodesUtilisateurs) {
					groupeutilisateur.init();
   					groupeutilisateur.setCodeUtilisateur(codeUtilisateur);
					groupeutilisateur.setCodeGroupe(groupeDsi.getCode());
 					groupeutilisateur.setSourceImport(sourceImport);
					groupeutilisateur.add();
				}
			}
			// maj du timestamp en minutes, et ce meme si il n'y a personne dans le groupe
			groupeDsi.setCtx(ctx);
			groupeDsi.setDerniereMajCache(tsMin);
			groupeDsi.update();
		}
	}

	/**
	 * Retourne les utilisateurs d'un groupe a partir de la requete d'un groupe.
	 * 
	 * @param requeteGroupe
	 *            requete du groupe
	 * 
	 * @return utilisateurs rattaches
	 * 
	 * @throws Exception
	 *             the exception
	 */
	protected abstract Vector<String> getVecteurUtilisateurs(String requeteGroupe) throws Exception;

	/**
	 * Indique si la requete est prise en charge par l'implementation. Permet d'optimiser la gestion de cache.
	 * 
	 * @param requeteGroupe
	 *            requete du groupe dynamique
	 * 
	 * @return vrai si la requete est geree
	 */
	protected abstract boolean estGereParReqDyn(String requeteGroupe);

	/**
	 * Préparer l'interface de saisie d'un groupe dynamique .Cette fonction est appelée lors de l'appel au processus de saisie de groupe DSI.<br/>
	 * Il faut absolument qu'il y ait une propriété dans {@link InfoBean} nommée : "REQUETE_LDAP_" + NOM_REQUETE ainsi que dans l'interface pour être retournée au processsus.
	 * 
	 * @param infoBean
	 *            the info bean
	 * @param groupe
	 *            the groupe
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public abstract void preparerPRINCIPAL(InfoBean infoBean, Groupedsi groupe) throws Exception;

	/**
	 * Ce traitement qui permet de sauvegarder la requete de groupe dynamique, il s'agit d'un traitement de base : ajout de la requete de groupe dynamique dnas le groupe et du type
	 * de requete.
	 * 
	 * @param infoBean
	 *            the info bean
	 * @param groupe
	 *            the groupe
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public void traiterPRINCIPAL(final InfoBean infoBean, final Groupedsi groupe) throws Exception {
		groupe.setRequeteGroupe(this.getNomRequete());
		groupe.setRequeteLdap(infoBean.getString("REQUETE_LDAP_" + this.getNomRequete()));
	}
}
