/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.net.MalformedURLException;
import java.net.URL;
import java.sql.Date;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

import org.apache.commons.lang3.StringUtils;
import org.codehaus.jackson.annotate.JsonIgnore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.InfoBean;
import com.jsbsoft.jtf.core.LangueUtil;
import com.jsbsoft.jtf.database.OMContext;
import com.univ.objetspartages.sgbd.RessourceDB;
import com.univ.utils.ContexteDao;
import com.univ.utils.ContexteUniv;
import com.univ.utils.ContexteUtil;
import com.univ.utils.EscapeString;
import com.univ.utils.FicheUnivMgr;
import com.univ.utils.sql.Operateur;
import com.univ.utils.sql.RequeteSQL;
import com.univ.utils.sql.clause.ClauseOrderBy;
import com.univ.utils.sql.clause.ClauseOrderBy.SensDeTri;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.utils.sql.criterespecifique.OrderByHelper;
import com.univ.utils.sql.operande.TypeOperande;

/**
 * The Class Ressource.
 */
public class Ressource extends RessourceDB implements Cloneable {

    /**
     *
     */
    private static final long serialVersionUID = -2582769772291064992L;

    private static final String COLONNE_CODE_PARENT = "CODE_PARENT";

    private static final Logger LOGGER = LoggerFactory.getLogger(Ressource.class);

    /** The Constant ETAT_SAUVEGARDE_STRUCTURE_COURANTE. */
    public static final String ETAT_SAUVEGARDE_STRUCTURE_COURANTE = "destruction_sauvegarde_courante";

    /** The media. */
    private Media media = null;

    /**
     * Initialisation de l'objet.
     */
    public void init() {
        setIdRessource((long) 0);
        setIdMedia((long) 0);
        setCodeParent("");
        setEtat("");
        setOrdre(0);
    }

    /**
     * Suppression des fichiers d'une fiche.
     *
     * @param _codeparent
     *            the _codeparent
     *
     * @throws Exception
     *             the exception
     */
    public static void supprimerFichier(final String _codeparent) throws Exception {
        try (ContexteDao ctx = new ContexteDao()) {
            final Vector<Ressource> listeFichier = getListeFichier(_codeparent);
            Ressource temp = new Ressource();
            temp.init();
            temp.setCtx(ctx);
            if (listeFichier.size() > 0) {
                for (Ressource ficdel : listeFichier) {
                    ficdel.setCtx(ctx);
                    final ClauseWhere whereIdMedia = whereIdMediaEgal(ficdel.getIdMedia());
                    if (temp.select(whereIdMedia.formaterSQL()) == 1) {
                        Media.deleteById(ficdel.getIdMedia());
                    }
                    ficdel.delete();
                }
            }
        }
    }

    private static ClauseWhere whereIdMediaEgal(final Long idMedia) {
        return new ClauseWhere(ConditionHelper.egalVarchar("ID_MEDIA", String.valueOf(idMedia)));
    }

    private static ClauseWhere whereCodeParentEgal(final String codeParent) {
        return new ClauseWhere(ConditionHelper.egalVarchar(COLONNE_CODE_PARENT, codeParent));
    }

    /**
     *
     * @return le contenu à indexer
     */
    public String getFullTextString() {
        return StringUtils.join(getLibelle(), " ", getLegende(), " ", getDescription(), " ", getAuteur(), " ", getSource());
    }

    /**
     * suppression des fichiers d'une fiche.
     *
     * @param _ctx
     *            the _ctx
     * @param _fiche
     *            the _fiche
     *
     * @throws Exception
     *             the exception
     */
    public static void supprimerFichiers(final OMContext _ctx, final FicheUniv _fiche) throws Exception {
        final Ressource temp = new Ressource();
        temp.init();
        temp.setCtx(_ctx);
        boolean modifFichier = false;
        // parcours de tous les fichiers (sauf les vignettes)
        final Vector<Ressource> listeFichier = getListeTotale(_fiche);
        if (listeFichier.size() > 0) {
            for (final Ressource ficdel : listeFichier) {
                // on supprime le fichier physique seulement si il n'a pas d'autres références en base
                final ClauseWhere whereIdMedia = whereIdMediaEgal(ficdel.getIdMedia());
                if (temp.select(whereIdMedia.formaterSQL()) == 1) {
                    Media.deleteById(ficdel.getIdMedia());
                    // on note une supression de fichier
                    if (FicheUnivMgr.isFicheCollaborative(_fiche)) {
                        modifFichier = true;
                    }
                }
                // en apercu on ne supprime pas le fichier mais on passe son etat à 2 (sera supprimé par le scansite)
                if ("0005".equals(_fiche.getEtatObjet())) {
                    ficdel.setEtat("2");
                    ficdel.update();
                } else if (ETAT_SAUVEGARDE_STRUCTURE_COURANTE.equals(_fiche.getMessageAlerte())) {
                    ficdel.setEtat("0");
                    ficdel.update();
                } else {
                    ficdel.delete();
                }
            }
        }
    }

    /**
     * suppression des fichiers d'une fiche.
     *
     * @param fiche
     *            the fiche
     *
     * @throws Exception
     *             the exception
     */
    public static void supprimerFichiers(final FicheUniv fiche) throws Exception {
        final Ressource temp = new Ressource();
        boolean modifFichier = false;
        temp.init();
        try (ContexteDao ctx = new ContexteDao()) {
            temp.setCtx(ctx);
            // parcours de tous les fichiers (sauf les vignettes)
            final Vector<Ressource> listeFichier = getListeTotale(fiche);
            if (listeFichier.size() > 0) {
                for (final Ressource ficdel : listeFichier) {
                    // on supprime le fichier physique seulement si il n'a pas d'autres références en base
                    final ClauseWhere whereIdMedia = whereIdMediaEgal(ficdel.getIdMedia());
                    if (temp.select(whereIdMedia.formaterSQL()) == 1) {
                        Media.deleteById(ficdel.getIdMedia());
                        // on note une supression de fichier
                        if (FicheUnivMgr.isFicheCollaborative(fiche)) {
                            modifFichier = true;
                        }
                    }
                    try (ContexteDao ctx2 = new ContexteDao()) {
                        ficdel.setCtx(ctx2);
                        // en apercu on ne supprime pas le fichier mais on passe son etat à 2 (sera supprimé par le scansite)
                        if ("0005".equals(fiche.getEtatObjet())) {
                            ficdel.setEtat("2");
                            ficdel.update();
                        } else if (ETAT_SAUVEGARDE_STRUCTURE_COURANTE.equals(fiche.getMessageAlerte())) {
                            ficdel.setEtat("0");
                            ficdel.update();
                        } else {
                            ficdel.delete();
                        }
                    }
                }
            }
        }
    }

    /**
     * Synchroniser liste fichier.
     *
     * @param _ctx
     *            the _ctx
     * @param _fiche
     *            the _fiche
     * @param _chaineFichier
     *            the _chaine fichier
     * @param modeFichier
     *            the mode fichier
     *
     * @throws Exception
     *             the exception
     */
    public static void synchroniserListeFichier(final OMContext _ctx, final FicheUniv _fiche, final String _chaineFichier, final String modeFichier, final int indice) throws Exception {
        boolean modifFichier = false;
        final boolean apercu = "0005".equals(_fiche.getEtatObjet()) || "0006".equals(_fiche.getEtatObjet());
        final String codeParent = _fiche.getIdFiche() + ",TYPE=" + ReferentielObjets.getCodeObjet(_fiche) + ",NO=" + indice;
        final Hashtable<String, String> hListeFichier = new Hashtable<>();
        /* etape 1 : lecture et enregistrement des fichiers de la fiche courante */
        if (_chaineFichier.length() > 0) {
            final StringTokenizer st = new StringTokenizer(_chaineFichier, "|");
            while (st.hasMoreTokens()) {
                final String enregistrement = st.nextToken();
                final String[] ligne = enregistrement.split(";", -2);
                Ressource ressource = new Ressource();
                ressource.init();
                ressource.setCtx(_ctx);
                ressource.setIdRessource(new Long(ligne[0]));
                ressource.retrieve();
                if (ressource.getCodeParent().length() == 0) {
                    // upload generique si etat=0;
                    if ("0".equals(ressource.getEtat())) {
                        // on note un ajout de fichier
                        if (FicheUnivMgr.isFicheCollaborative(_fiche)) {
                            modifFichier = true;
                        }
                        ressource.setEtat("1");
                    }
                    // duplication du tuple fichier non rattaché dans le cas d'un apercu ou une sauveagarde automatique
                    // enregistrement generique du fichier
                    ressource.setCodeParent(codeParent);
                    // si mode apercu ou sauvegarde automatique on duplique compl�tement le tuple
                    if (apercu) {
                        // on duplique le fichier
                        ressource = dupliquerFichier(ressource, codeParent, "1");
                    } else {
                        ressource.update();
                    }
                }
                // dans le cas d'un apercu ou d'une sauvegarde automatique : le fichier pointe vers une autre fiche
                else if (!ressource.getCodeParent().equals(codeParent)) {
                    // on duplique le fichier
                    ressource = dupliquerFichier(ressource, codeParent, "1");
                }
                hListeFichier.put(ressource.getIdRessource().toString(), "");
            }
        }
        /* etape 2 : suppression des fichiers non liés à cette fiche */
        final Ressource ficdel = new Ressource();
        ficdel.init();
        ficdel.setCtx(_ctx);
        final ClauseWhere whereCodeParent = whereCodeParentEgal(codeParent);
        if (ficdel.select(whereCodeParent.formaterSQL()) > 0) {
            while (ficdel.nextItem()) {
                final Ressource temp = new Ressource();
                temp.init();
                temp.setCtx(_ctx);
                // le fichier n'est plus lié, on le supprime
                if (hListeFichier.get(ficdel.getIdRessource().toString()) == null) {
                    // si c'est l'unique media associé on le supprime aussi
                    final ClauseWhere whereIdMedia = whereIdMediaEgal(ficdel.getIdMedia());
                    if (temp.select(whereIdMedia.formaterSQL()) == 1) {
                        Media.deleteById(ficdel.getIdMedia());
                        // on note une suppression de fichier
                        if (FicheUnivMgr.isFicheCollaborative(_fiche)) {
                            modifFichier = true;
                        }
                    }
                    ficdel.delete();
                }
                // suppression des apercus à l'enregistrement
                else if (!apercu) {
                    final ClauseWhere whereIdMedia = whereIdMediaEgal(ficdel.getIdMedia());
                    whereIdMedia.and(ConditionHelper.egalVarchar("ETAT", "2"));
                    temp.select(whereIdMedia.formaterSQL());
                    while (temp.nextItem()) {
                        temp.delete();
                    }
                }
            }
        }
    }

    /**
     * duplication des fichiers sur une newsletter pour faire l'apercu methode utilisee dans SaisieModeleMail pour laquelle il n'y a pas de ficheUniv.
     *
     * @param _ctx
     *            the _ctx
     * @param _codeParent
     *            the _code parent
     * @param _chaineFichier
     *            the _chaine fichier
     * @param modeFichier
     *            the mode fichier
     * @param _apercu
     *            the _apercu
     *
     * @throws Exception
     *             the exception
     */
    public static void synchroniserListeFichier(final OMContext _ctx, final String _codeParent, final String _chaineFichier, final String modeFichier, final boolean _apercu) throws Exception {
        final Hashtable<String, String> hListeFichier = new Hashtable<>();
        /* etape 1 : lecture et enregistrement des fichiers de la fiche courante */
        if (_chaineFichier.length() > 0) {
            final StringTokenizer st = new StringTokenizer(_chaineFichier, "|");
            while (st.hasMoreTokens()) {
                final String enregistrement = st.nextToken();
                final String[] ligne = enregistrement.split(";", -2);
                Ressource fichiergw = new Ressource();
                fichiergw.init();
                fichiergw.setCtx(_ctx);
                fichiergw.setIdRessource(new Long(ligne[0]));
                fichiergw.retrieve();
                if (fichiergw.getCodeParent().length() == 0) {
                    // upload generique si etat=0;
                    if ("0".equals(fichiergw.getEtat())) {
                        fichiergw.setEtat("1");
                    }
                    // duplication du tuple fichier non rattaché dans le cas d'un apercu ou une sauvegarde automatique
                    if (_apercu) {
                        fichiergw.update();
                    }
                    // enregistrement generique du fichier
                    fichiergw.setCodeParent(_codeParent);
                    // si mode apercu ou sauvegarde automatique on duplique complètement le tuple
                    if (_apercu) {
                        // on duplique le fichier
                        fichiergw = dupliquerFichier(fichiergw, _codeParent, "1");
                    } else {
                        fichiergw.update();
                    }
                }
                // dans le cas d'un apercu ou d'une sauvegarde automatique : le fichier pointe vers une autre fiche
                else if (!fichiergw.getCodeParent().equals(_codeParent)) {
                    // on duplique le fichier en valorisant l'id de la vignette dupliqu�e
                    fichiergw = dupliquerFichier(fichiergw, _codeParent, "1");
                }
                hListeFichier.put(fichiergw.getIdRessource().toString(), "");
            }
        }
        /* etape 2 : suppression des fichiers non liés à cette fiche ou update des fichiers dupliqués*/
        final Ressource ficdel = new Ressource();
        ficdel.init();
        ficdel.setCtx(_ctx);
        final ClauseWhere whereCodeParent = whereCodeParentEgal(_codeParent);
        if (ficdel.select(whereCodeParent.formaterSQL()) > 0) {
            while (ficdel.nextItem()) {
                if (hListeFichier.get(ficdel.getIdRessource().toString()) == null) {
                    final Ressource temp = new Ressource();
                    temp.init();
                    temp.setCtx(_ctx);
                    final ClauseWhere whereIdMedia = whereIdMediaEgal(ficdel.getIdMedia());
                    if (temp.select(whereIdMedia.formaterSQL()) == 1) {
                        Media.deleteById(ficdel.getIdMedia());
                    }
                    ficdel.delete();
                }
            }
        }
    }

    /**
     * duplication du fichier unique sur la nouvelle fiche.
     *
     * @param _ctx
     *            the _ctx
     * @param _fiche
     *            the _fiche
     * @param _chaineFichier
     *            the _chaine fichier
     * @param modeFichier
     *            the mode fichier
     * @param indice
     *            the indice
     *
     * @throws Exception
     *             the exception
     */
    public static void synchroniserFichier(final OMContext _ctx, final FicheUniv _fiche, final String _chaineFichier, final String modeFichier, final int indice) throws Exception {
        boolean modifFichier = false;
        final String codeParent = _fiche.getIdFiche() + ",TYPE=FICHIER_" + ReferentielObjets.getCodeObjet(_fiche) + ",NO=" + Integer.toString(indice);
        String idFichier = "";
        /* etape 1 : lecture et enregistrement du fichier de la fiche courante */
        if (_chaineFichier.length() > 0 && !"0".equals(_chaineFichier)) {
            final String[] ligne = _chaineFichier.split(";", -2);
            Ressource fichiergw = new Ressource();
            fichiergw.init();
            fichiergw.setCtx(_ctx);
            fichiergw.setIdRessource(new Long(ligne[0]));
            fichiergw.retrieve();
            if (fichiergw.getCodeParent().length() == 0) {
                // upload generique si etat=0;
                if ("0".equals(fichiergw.getEtat())) {
                    fichiergw.setEtat("1");
                    if (FicheUnivMgr.isFicheCollaborative(_fiche)) {
                        modifFichier = true;
                    }
                }
                // duplication du tuple fichier non rattaché dans le cas d'un apercu
                if ("0005".equals(_fiche.getEtatObjet()) || "0006".equals(_fiche.getEtatObjet())) {
                    fichiergw.update();
                }
                // enregistrement generique du fichier
                fichiergw.setCodeParent(codeParent);
                // si mode apercu ou sauvegarde automatique on duplique complètement le tuple
                if ("0005".equals(_fiche.getEtatObjet()) || "0006".equals(_fiche.getEtatObjet())) {
                    // on duplique le fichier
                    fichiergw = dupliquerFichier(fichiergw, codeParent, "1");
                } else {
                    fichiergw.update();
                }
            } else if (!fichiergw.getCodeParent().equals(codeParent)) {
                // on duplique le fichier
                fichiergw = dupliquerFichier(fichiergw, codeParent, "1");
            }
            idFichier = fichiergw.getIdRessource().toString();
        }
        /* etape 2 : suppression du fichier non lié à cette fiche*/
        final Ressource ficdel = new Ressource();
        ficdel.init();
        ficdel.setCtx(_ctx);
        final ClauseWhere whereCodeParent = whereCodeParentEgal(codeParent);
        if (ficdel.select(whereCodeParent.formaterSQL()) > 0) {
            while (ficdel.nextItem()) {
                if (!ficdel.getIdRessource().toString().equals(idFichier)) {
                    final Ressource temp = new Ressource();
                    temp.init();
                    temp.setCtx(_ctx);
                    // supprime le fichier physique seulement s'il ne fait référence qu'au tuple courant et hors photothèque
                    final ClauseWhere whereIdMedia = whereIdMediaEgal(ficdel.getIdMedia());
                    if (temp.select(whereIdMedia.formaterSQL()) == 1) {
                        Media.deleteById(ficdel.getIdMedia());
                        // on note une suppression de fichier
                        if (FicheUnivMgr.isFicheCollaborative(_fiche)) {
                            modifFichier = true;
                        }
                    }
                    ficdel.delete();
                }
            }
        }
    }

    /**
     * Save content ressource.
     *
     * @param _ctx
     *            the _ctx
     * @param _infoBean
     *            the _info bean
     * @param _ficheUniv
     *            the _fiche univ
     *
     * @throws Exception
     *             the exception
     */
    public static void saveContentRessource(final OMContext _ctx, final Map<String, Object> _infoBean, final FicheUniv _ficheUniv) throws Exception {
        final Ressource ressource = new Ressource();
        ressource.init();
        ressource.setCtx(_ctx);
        // preparation du code parent
        String type = "";
        String codeParent = "";
        String _codeParent = "_" + ReferentielObjets.getCodeObjet(_ficheUniv) + ",CODE=";
        String codeRecherche = "";
        if (((String) _infoBean.get("TS_CODE")).startsWith("TS")) {
            codeRecherche = _codeParent + _infoBean.get("TS_CODE");
        } else {
            codeRecherche = _codeParent + _ficheUniv.getCode();
        }
        _codeParent = _codeParent + _ficheUniv.getCode();
        boolean duplique = true;
        @SuppressWarnings("unchecked")
        final HashSet<String> hSetNew = (HashSet<String>) _infoBean.get("contentNewRessources");
        @SuppressWarnings("unchecked")
        final HashSet<String> hSetOld = (HashSet<String>) _infoBean.get("contentOldRessources");
        // Parcours des fichiers courants
        for (String codeFichier : hSetNew) {
            try {
                duplique = true;
                if (codeFichier.contains("IMG")) {
                    type = "IMG";
                } else {
                    type = "LIEN";
                }
                codeFichier = codeFichier.substring(0, codeFichier.indexOf("#"));
                codeParent = "TYPE=" + type + _codeParent;
                // on recupere le fichier parmi grace au code parent code
                final ClauseWhere whereCodeParentEtIdMedia = new ClauseWhere(ConditionHelper.like(COLONNE_CODE_PARENT, codeRecherche, "%", ""));
                whereCodeParentEtIdMedia.and(ConditionHelper.egalVarchar("ID_MEDIA", codeFichier));
                if (ressource.select(whereCodeParentEtIdMedia.formaterSQL()) > 0) {
                    while (ressource.nextItem()) {
                        // nouveau fichier
                        if ("0".equals(ressource.getEtat())) {
                            // en creation sur un apercu on duplique le fichier car pas de sauvegarde automatique
                            if (_infoBean.get(InfoBean.ETAT_OBJET).equals(InfoBean.ETAT_OBJET_CREATION) && "0005".equals(_ficheUniv.getEtatObjet())) {
                                ressource.update();
                                break;
                            }
                            ressource.setCodeParent(_ficheUniv.getIdFiche().toString() + "," + codeParent);
                            ressource.setEtat("1");
                            ressource.update();
                            duplique = false;
                            break;
                            // le fichier existe déja pour la fiche
                        } else if (ressource.getCodeParent().indexOf(_ficheUniv.getIdFiche().toString() + ",") == 0) {
                            duplique = false;
                            break;
                        }
                    }
                    // il existe un fichier du meme code sur une autre fiche, ou sur un aperçu on duplique car pas de sauvegarde automatique
                    if (duplique) {
                        ressource.setIdRessource((long) 0);
                        ressource.setCodeParent(_ficheUniv.getIdFiche().toString() + "," + codeParent);
                        ressource.setEtat("1");
                        ressource.add();
                    }
                }
                // on ne retrouve pas la ressource (cela peut venir d'un copié collé ou un pb)
                else {
                    // on teste que le média existe bien
                    try {
                        final Media media = new Media();
                        media.init();
                        media.setCtx(_ctx);
                        media.setIdMedia(new Long(codeFichier));
                        media.retrieve();
                        ressource.setIdMedia(media.getIdMedia());
                        ressource.setIdRessource((long) 0);
                        ressource.setCodeParent(_ficheUniv.getIdFiche().toString() + "," + codeParent);
                        ressource.setEtat("1");
                        ressource.add();
                    } catch (final Exception e) {
                        // on ignore le fichier
                    }
                }
            } catch (final Exception e) {
                // TODO: handle exception
            }
        }
        final Ressource temp = new Ressource();
        temp.init();
        temp.setCtx(_ctx);
        // Suppression des fichiers obsoletes du contenu de la fiche (insertion toolbox)
        final String codeParentEchapper = EscapeString.escapeSql(_codeParent);
        final String valeurLike = "'" + _ficheUniv.getIdFiche() + ",TYPE=%" + codeParentEchapper + "'";
        final ClauseWhere whereCodeParentEtIdMedia = new ClauseWhere(ConditionHelper.genericConditionSurColonne(COLONNE_CODE_PARENT, valeurLike, TypeOperande.NON_ECHAPABLE, Operateur.LIKE));
        if (ressource.select(whereCodeParentEtIdMedia.formaterSQL()) > 0) {
            while (ressource.nextItem()) {
                // si le fichier est obsolete on le supprime
                if (hSetOld.contains(ressource.getIdMedia().toString() + "#LIEN") || hSetOld.contains(ressource.getIdMedia().toString() + "#IMG")) {
                    // supprime le fichier physique seulement s'il ne fait référence qu'au tuple courant et hors photothèque
                    final ClauseWhere whereIdMedia = whereIdMediaEgal(ressource.getIdMedia());
                    if (temp.select(whereIdMedia.formaterSQL()) == 1) {
                        Media.deleteById(ressource.getIdMedia());
                    }
                    // suppression de l'enregistrement en base
                    ressource.delete();
                }
            }
        }
    }

    /**
     * Synchroniser liste fichier.
     *
     * @param _ctx
     *            the _ctx
     * @param _codeParent
     *            the _code parent
     * @param _chaineFichier
     *            the _chaine fichier
     *
     * @throws Exception
     *             the exception
     */
    public static void synchroniserListeFichier(final OMContext _ctx, final String _codeParent, final String _chaineFichier) throws Exception {
        final Hashtable<String, String> hListeFichier = new Hashtable<>();
        /* etape 1 : lecture et enregistrement des fichiers de la fiche courante */
        if (_chaineFichier.length() > 0) {
            final StringTokenizer st = new StringTokenizer(_chaineFichier, "|");
            while (st.hasMoreTokens()) {
                final String enregistrement = st.nextToken();
                final String[] ligne = enregistrement.split(";", -2);
                final Ressource fichiergw = new Ressource();
                fichiergw.init();
                fichiergw.setCtx(_ctx);
                fichiergw.setIdRessource(new Long(ligne[0]));
                fichiergw.retrieve();
                if (fichiergw.getCodeParent().length() == 0) {
                    // upload generique si etat=0;
                    if ("0".equals(fichiergw.getEtat())) {
                        fichiergw.setEtat("1");
                    }
                    // enregistrement generique du fichier
                    fichiergw.setCodeParent(_codeParent);
                    fichiergw.update();
                }
                hListeFichier.put(fichiergw.getIdRessource().toString(), "");
            }
        }
        /* etape 2 : suppression des fichiers non liés à cette fiche ou update des fichiers dupliqués*/
        final Ressource ficdel = new Ressource();
        ficdel.init();
        ficdel.setCtx(_ctx);
        final ClauseWhere whereCodeParent = whereCodeParentEgal(_codeParent);
        if (ficdel.select(whereCodeParent.formaterSQL()) > 0) {
            while (ficdel.nextItem()) {
                if (hListeFichier.get(ficdel.getIdRessource().toString()) == null) {
                    final Ressource temp = new Ressource();
                    temp.init();
                    temp.setCtx(_ctx);
                    final ClauseWhere whereIdMedia = whereIdMediaEgal(ficdel.getIdMedia());
                    if (temp.select(whereIdMedia.formaterSQL()) == 1) {
                        Media.deleteById(ficdel.getIdMedia());
                    }
                    ficdel.delete();
                }
            }
        }
        // En cas de modification de fichiers on recharge la liste des espaces
    }

    /**
     * Synchroniser fichier : duplication du fichier unique sur un objet non fiche.
     *
     * @param _ctx
     *            the _ctx
     * @param _chaineFichier
     *            the _chaine fichier
     * @param codeParent
     *            the code parent
     * @param modeFichier
     *            the mode fichier
     *
     * @throws Exception
     *             the exception
     */
    public static void synchroniserFichier(final OMContext _ctx, final String _chaineFichier, final String codeParent, final String modeFichier) throws Exception {
        String idFichier = "";
        /* etape 1 : lecture et enregistrement du fichier de la fiche courante */
        if (_chaineFichier.length() > 0 && !"0".equals(_chaineFichier)) {
            final String[] ligne = _chaineFichier.split(";", -2);
            Ressource fichiergw = new Ressource();
            fichiergw.init();
            fichiergw.setCtx(_ctx);
            fichiergw.setIdRessource(new Long(ligne[0]));
            fichiergw.retrieve();
            if ("0".equals(fichiergw.getEtat())) {
                fichiergw.setCodeParent(codeParent);
                fichiergw.setEtat("1");
                fichiergw.update();
            } else if (!fichiergw.getCodeParent().equals(codeParent)) {
                fichiergw = dupliquerFichier(fichiergw, codeParent, "1");
            }
            idFichier = fichiergw.getIdRessource().toString();
        }
        /*suppression du fichier non lié à cette fiche*/
        final Ressource ficdel = new Ressource();
        ficdel.init();
        ficdel.setCtx(_ctx);
        final ClauseWhere whereCodeParent = whereCodeParentEgal(codeParent);
        if (ficdel.select(whereCodeParent.formaterSQL()) > 0) {
            while (ficdel.nextItem()) {
                if (!ficdel.getIdRessource().toString().equals(idFichier)) {
                    Media.deleteById(ficdel.getIdMedia());
                    ficdel.delete();
                }
            }
        }
    }

    /**
     * Duplication de tous les fichiers d'une fiche
     *
     * @param _ctx
     *            the _ctx
     * @param _ficheOrigine
     *            the _fiche origine
     * @param _ficheNouvelle
     *            the _fiche nouvelle
     *
     * @throws Exception
     *             the exception
     * @deprecated le ctx n'est pas nécessaire. Il faut utiliser {@link Ressource#dupliquerListeFichier(FicheUniv, FicheUniv)}
     */
    @Deprecated
    public static void dupliquerListeFichier(final OMContext _ctx, final FicheUniv _ficheOrigine, final FicheUniv _ficheNouvelle) throws Exception {
        String codeParent = "";
        // liste des fichiers + vignettes
        final Vector<Ressource> listeFichier = getListeTotale(_ficheOrigine);
        if (listeFichier.size() > 0) {
            for (Ressource fic : listeFichier) {
                // duplication du code parent
                codeParent = fic.getCodeParent().replaceFirst(_ficheOrigine.getIdFiche().toString(), _ficheNouvelle.getIdFiche().toString());
                // duplication du code du fichier
                if (codeParent.contains(_ficheOrigine.getCode())) {
                    codeParent = StringUtils.replace(codeParent, _ficheOrigine.getCode(), _ficheNouvelle.getCode());
                }
                // duplication du fichier avec le reste
                Ressource.dupliquerFichier(fic, codeParent, "1");
            }
        }
    }

    /**
     * Duplication de tous les fichiers d'une fiche
     *
     * @param ficheOrigine
     *            La fiche à laquelle appartiennent les fichiers d'origines
     * @param ficheNouvelle
     *            la nouvelle fiche
     *
     * @throws Exception
     *             Lors des requêtes en bdd
     */
    public static void dupliquerListeFichier(final FicheUniv ficheOrigine, final FicheUniv ficheNouvelle) throws Exception {
        String codeParent = "";
        // liste des fichiers + vignettes
        final Vector<Ressource> listeFichier = getListeTotale(ficheOrigine);
        if (listeFichier.size() > 0) {
            for (Ressource fic : listeFichier) {
                // duplication du code parent
                codeParent = fic.getCodeParent().replaceFirst(ficheOrigine.getIdFiche().toString(), ficheNouvelle.getIdFiche().toString());
                // duplication du code du fichier
                if (codeParent.contains(ficheOrigine.getCode())) {
                    codeParent = StringUtils.replace(codeParent, ficheOrigine.getCode(), ficheNouvelle.getCode());
                }
                // duplication du fichier avec le reste
                Ressource.dupliquerFichier(fic, codeParent, "1");
            }
        }
    }

    /**
     * Duplication d'un fichier sur un autre
     *
     * @param _ctx
     *            the _ctx
     * @param _fichiergw
     *            the _fichiergw
     * @param _codeParent
     *            the _code parent
     * @param _etat
     *            the _etat
     *
     * @return the ressource
     *
     * @throws Exception
     *             the exceptio
     * @deprecated le contexte n'ext pas nécessaire. il faut utiliser {@link Ressource#dupliquerFichier(Ressource, String, String)}
     */
    @Deprecated
    public static Ressource dupliquerFichier(final OMContext _ctx, final Ressource _fichiergw, final String _codeParent, final String _etat) throws Exception {
        return dupliquerFichier(_fichiergw, _codeParent, _etat);
    }

    /**
     * Duplication d'un fichier sur un autre
     *
     * @param fichiergw
     *            le fichier à dupliquer
     * @param codeParent
     *            le code parent de la ressource dupliquée
     * @param etat
     *            l'état de la ressource dupliquée
     *
     * @return la ressource dupliquée
     *
     * @throws Exception
     *             Lors de la requête en base
     */
    public static Ressource dupliquerFichier(final Ressource fichiergw, final String codeParent, String etat) throws Exception {
        final Ressource fichierACreer = new Ressource();
        final ContexteDao ctx = new ContexteDao();
        fichierACreer.setCtx(ctx);
        fichierACreer.init();
        fichierACreer.setIdRessource((long) 0);
        fichierACreer.setIdMedia(fichiergw.getIdMedia());
        fichierACreer.setCodeParent(codeParent);
        fichierACreer.setOrdre(fichiergw.getOrdre());
        if (StringUtils.isEmpty(etat)) {
            etat = fichiergw.getEtat();
        }
        fichierACreer.setEtat(etat);
        try {
            fichierACreer.add();
        } finally {
            ctx.close();
        }
        return fichierACreer;
    }

    /**
     * Renvoie la liste des fichiers globale d'une fiche tout type confondu sauf vignette.
     *
     * @param _ctx
     *            the _ctx
     * @param _fiche
     *            the _fiche
     *
     * @return the liste totale
     *
     * @throws Exception
     *             the exception
     * @deprecated le contexte n'est pas necessaire, utiliser {@link Ressource#getListeTotale(FicheUniv)}
     *
     */
    @Deprecated
    public static Vector<Ressource> getListeTotale(final OMContext _ctx, final FicheUniv _fiche) throws Exception {
        return getListeTotale(_fiche);
    }

    /**
     * Renvoie la liste des fichiers globale d'une fiche tout type confondu sauf vignette.
     *
     * @param fiche
     *            la fiche dont on veut récupérer les fichiers
     *
     * @return la liste totale des ressources liés à cette fiches
     *
     * @throws Exception
     *             the exception lors de la requête en base
     */
    public static Vector<Ressource> getListeTotale(final FicheUniv fiche) throws Exception {
        final String codeParent = fiche.getIdFiche() + ",TYPE=%" + ReferentielObjets.getCodeObjet(fiche) + "%";
        return getListeFichier(codeParent, StringUtils.EMPTY);
    }

    /**
     * Renvoie la liste des fichiers reliees directement a une fiche (liste de n fichier).
     *
     * @param _ctx
     *            the _ctx
     * @param _fiche
     *            the _fiche
     *
     * @return the liste fichier
     *
     * @throws Exception
     *             the exception
     * @deprecated le contexte n'est pas necessaire dans ce cas présent il ne sert qu'a avoir la connexion à la bdd. Il faut utiliser {@link Ressource#getListeFichier(FicheUniv)}
     */
    @Deprecated
    public static Vector<Ressource> getListeFichier(final OMContext _ctx, final FicheUniv _fiche) throws Exception {
        return getListeFichier(_fiche);
    }

    /**
     * Renvoie la liste des fichiers reliees directement a une fiche (liste de n fichier).
     *
     * @param fiche
     *            la fiche dont on veut avoir les ressources
     *
     * @return La liste des fichiers liés à la fiche
     *
     * @throws Exception
     *             En cas d'exception sur la requête en base
     */
    public static Vector<Ressource> getListeFichier(final FicheUniv fiche) throws Exception {
        final String codeParent = fiche.getIdFiche() + ",TYPE=" + ReferentielObjets.getCodeObjet(fiche) + "%";
        return getListeFichier(codeParent, COLONNE_CODE_PARENT);
    }

    /**
     *
     * @param _ctx
     * @param codeParent
     * @return
     * @throws Exception
     * @deprecated le contexte n'est pas necessaire dans ce cas présent il ne sert qu'a avoir la connexion à la bdd. Il faut utiliser
     *             {@link Ressource#getListeFichierByCodeParent(String)}
     */
    @Deprecated
    public static Vector<Ressource> getListeFichierByCodeParent(final OMContext _ctx, final String codeParent) throws Exception {
        return getListeFichierByCodeParent(codeParent);
    }

    /**
     * Renvoie la liste des fichiers reliees directement au code parent fourni en paramètre
     *
     * @param codeParent
     *            le code parent dont on veut avoir les fichiers liés
     * @return La liste des fichiers liés à la fiche
     * @throws Exception
     *             En cas d'exception sur la requête en base
     */
    public static Vector<Ressource> getListeFichierByCodeParent(final String codeParent) throws Exception {
        return getListeFichier(codeParent, COLONNE_CODE_PARENT);
    }

    /**
     *
     * @param _ctx
     * @param _fiche
     * @param indice
     * @return
     * @throws Exception
     * @deprecated le contexte n'est pas necessaire dans ce cas présent il ne sert qu'a avoir la connexion à la bdd. Il faut utiliser
     *             {@link Ressource#getListeFichier(FicheUniv, String)}
     */
    @Deprecated
    public static Vector<Ressource> getListeFichier(final OMContext _ctx, final FicheUniv _fiche, final String indice) throws Exception {
        return getListeFichier(_fiche, indice);
    }

    /**
     * Renvoie la liste des fichiers reliees directement au champs d'indice fourni pour la fiche passé en paramètre
     *
     * @param fiche
     *            la fiche dont on veut avoir les ressources
     * @param indice
     *            l'indice du champ dont on veut avoir les ressources
     * @return La liste des fichiers liés à la fiche
     * @throws Exception
     *             En cas d'exception sur la requête en base
     */
    public static Vector<Ressource> getListeFichier(final FicheUniv fiche, final String indice) throws Exception {
        final String codeParent = fiche.getIdFiche() + ",TYPE=" + ReferentielObjets.getCodeObjet(fiche) + ",NO=" + indice;
        return getListeFichier(codeParent, COLONNE_CODE_PARENT);
    }

    /**
     * Renvoie la liste des fichiers indicès relièes directement à une fiche
     *
     * @param _ctx
     *            the _ctx
     * @param _fiche
     *            the _fiche
     *
     * @return the liste fichier indice
     *
     * @throws Exception
     *             the exception
     * @deprecated le contexte n'est pas necessaire dans ce cas présent il ne sert qu'a avoir la connexion à la bdd. Il faut utiliser
     *             {@link Ressource#getListeFichierIndice(FicheUniv)}
     */
    @Deprecated
    public static Vector<Ressource> getListeFichierIndice(final OMContext _ctx, final FicheUniv _fiche) throws Exception {
        return getListeFichierIndice(_fiche);
    }

    /**
     * Renvoie la liste des fichiers indicès relièes directement à une fiche
     *
     * @param fiche
     *            la fiche dont on veut avoir les ressources
     *
     * @return La liste des fichiers indicès liés à la fiche
     *
     * @throws Exception
     *             En cas d'exception sur la requête en base
     */
    public static Vector<Ressource> getListeFichierIndice(final FicheUniv fiche) throws Exception {
        final String codeParent = fiche.getIdFiche() + ",TYPE=FICHIER_" + ReferentielObjets.getCodeObjet(fiche) + "%";
        return getListeFichier(codeParent, COLONNE_CODE_PARENT);
    }

    /**
     *
     * @param _ctx
     * @param _fiche
     * @param indice
     * @return
     * @throws Exception
     * @deprecated le contexte n'est pas necessaire dans ce cas présent il ne sert qu'a avoir la connexion à la bdd. Il faut utiliser
     *             {@link Ressource#getListeFichierIndice(FicheUniv, String)}
     */
    @Deprecated
    public static Vector<Ressource> getListeFichierIndice(final OMContext _ctx, final FicheUniv _fiche, final String indice) throws Exception {
        return getListeFichierIndice(_fiche, indice);
    }

    public static Vector<Ressource> getListeFichierIndice(final FicheUniv fiche, final String indice) throws Exception {
        final String codeParent = fiche.getIdFiche() + ",TYPE=FICHIER_" + ReferentielObjets.getCodeObjet(fiche) + ",NO=" + indice;
        return getListeFichier(codeParent, COLONNE_CODE_PARENT);
    }

    /**
     * Gets the liste fichier.
     *
     * @param _ctx
     *            the _ctx
     * @param _codeParent
     *            the _code parent
     *
     * @return the liste fichier
     *
     * @throws Exception
     *             the exception
     * @deprecated le contexte n'est pas nécessaire. Il faut utiliser {@link Ressource#getListeFichier(String)}
     */
    @Deprecated
    public static Vector<Ressource> getListeFichier(final OMContext _ctx, final String _codeParent) throws Exception {
        return getListeFichier(_codeParent);
    }

    /**
     * Récupère la liste des fichiers liés au code parent fourni en paramètre
     *
     * @param codeParent
     *            Le code parent de la fiche dont on souhaite aboir les fichiers joints
     *
     * @return la liste fichier lié au code parent
     *
     * @throws Exception
     *             Lors de la requête en bdd
     */
    public static Vector<Ressource> getListeFichier(final String codeParent) throws Exception {
        return getListeFichier(codeParent, StringUtils.EMPTY);
    }

    /**
     * Renvoie la liste des fichiers correspondant aux criteres voulus.
     *
     * @param _ctx
     *            the _ctx
     * @param _codeParent
     *            the _code parent
     * @param _orderBy
     *            the _order by
     *
     * @return the liste fichier
     *
     * @throws Exception
     *             the exception
     * @deprecated le contexte n'est pas nécessaire. Il faut utiliser {@link Ressource#getListeFichier(String, String)}
     */
    @Deprecated
    public static Vector<Ressource> getListeFichier(final OMContext _ctx, final String _codeParent, final String _orderBy) throws Exception {
        final Vector<Ressource> liste = new Vector<Ressource>();
        final Ressource fichiergw = new Ressource();
        fichiergw.init();
        fichiergw.setCtx(_ctx);
        // like sur le code parent qui peut contenir des %
        final ClauseWhere whereCodeParentlike = new ClauseWhere(ConditionHelper.like(COLONNE_CODE_PARENT, _codeParent, "", ""));
        ClauseOrderBy orderBy = new ClauseOrderBy();
        if (StringUtils.isNotEmpty(_orderBy)) {
            orderBy = OrderByHelper.reconstruireClauseOrderBy(_orderBy);
        } else {
            orderBy.orderBy("ID_RESSOURCE", SensDeTri.DESC);
        }
        final RequeteSQL requeteSQL = new RequeteSQL();
        requeteSQL.where(whereCodeParentlike).orderBy(orderBy);
        if (fichiergw.select(requeteSQL.formaterRequete()) > 0) {
            while (fichiergw.nextItem()) {
                Ressource fic = new Ressource();
                fic = (Ressource) fichiergw.clone();
                liste.add(fic);
            }
        }
        return liste;
    }

    /**
     * Renvoie la liste des fichiers correspondant aux criteres voulus.
     *
     * @param codeParent
     *            Le code parent dont on souhaite récupérer les fichiers
     * @param orderBy
     *            Le tri à faire
     *
     * @return la liste des fichiers liés au code parent
     *
     * @throws Exception
     *             Lors de la requête en bdd
     */
    public static Vector<Ressource> getListeFichier(final String codeParent, final String orderBy) throws Exception {
        final Vector<Ressource> liste = new Vector<>();
        final Ressource fichiergw = new Ressource();
        // like sur le code parent qui peut contenir des %
        final ClauseWhere whereCodeParentlike = new ClauseWhere(ConditionHelper.like(COLONNE_CODE_PARENT, codeParent, "", ""));
        ClauseOrderBy clauseOrderBy = new ClauseOrderBy();
        if (StringUtils.isNotEmpty(orderBy)) {
            clauseOrderBy = OrderByHelper.reconstruireClauseOrderBy(orderBy);
        } else {
            clauseOrderBy.orderBy("ID_RESSOURCE", SensDeTri.DESC);
        }
        final RequeteSQL requeteSQL = new RequeteSQL();
        requeteSQL.where(whereCodeParentlike).orderBy(clauseOrderBy);
        try (final ContexteDao ctx = new ContexteDao()) {
            fichiergw.init();
            fichiergw.setCtx(ctx);
            if (fichiergw.select(requeteSQL.formaterRequete()) > 0) {
                while (fichiergw.nextItem()) {
                    Ressource fic = (Ressource) fichiergw.clone();
                    liste.add(fic);
                }
            }
        }
        return liste;
    }

    /**
     * Renvoie le fichier lié a un objet parent.
     *
     * @param _ctx
     *            the _ctx
     * @param _fiche
     *            the _fiche
     *
     * @return the fichier
     *
     * @deprecated le contexte n'est pas nécessaire. Il faut utiliser {@link Ressource#getFichier(FicheUniv)}
     */
    @Deprecated
    public static Ressource getFichier(final OMContext _ctx, final FicheUniv _fiche) {
        return getFichier(_fiche);
    }

    /**
     * Renvoie le fichier lié a un objet parent.
     *
     * @param fiche
     *            la fiche dont on souhaite récupérer le fichier
     *
     * @return le fichier lié
     *
     */
    public static Ressource getFichier(final FicheUniv fiche) {
        return getFichier(fiche, "1");
    }

    /**
     * Renvoie le fichier lié a un objet parent.
     *
     * @param _ctx
     *            the _ctx
     * @param _fiche
     *            the _fiche
     * @param indice
     *            the indice
     *
     * @return the fichier
     *
     * @deprecated le contexte n'est pas nécessaire. Il faut utiliser {@link Ressource#getFichier(FicheUniv, String)}
     */
    @Deprecated
    public static Ressource getFichier(final OMContext _ctx, final FicheUniv _fiche, final String indice) {
        final String code = _fiche.getIdFiche() + ",TYPE=FICHIER_" + ReferentielObjets.getCodeObjet(_fiche) + ",NO=" + indice;
        final Ressource fichiergw = new Ressource();
        fichiergw.init();
        fichiergw.setCtx(_ctx);
        try {
            final ClauseWhere whereCodeParent = whereCodeParentEgal(code);
            final ClauseOrderBy orderByIdRessource = new ClauseOrderBy("ID_RESSOURCE", SensDeTri.DESC);
            final RequeteSQL requeteCodeParentTrie = new RequeteSQL();
            requeteCodeParentTrie.where(whereCodeParent).orderBy(orderByIdRessource);
            // soit on a un code parent
            fichiergw.select(requeteCodeParentTrie.formaterRequete());
            if (fichiergw.nextItem()) {
                return fichiergw;
            }
        } catch (final Exception e) {
            LOGGER.error("Fichier code=" + code + " introuvable.", e);
        }
        return null;
    }

    /**
     * Renvoie le fichier lié a un objet parent.
     *
     * @param _fiche
     *            La fiche surlaquelle on souhaite requeter
     * @param indice
     *            l'indice du champ du fichier
     *
     * @return le fichier lié ou null si rien n'a été trouvé
     *
     */
    public static Ressource getFichier(final FicheUniv _fiche, final String indice) {
        final String code = _fiche.getIdFiche() + ",TYPE=FICHIER_" + ReferentielObjets.getCodeObjet(_fiche) + ",NO=" + indice;
        final Ressource fichiergw = new Ressource();
        fichiergw.init();
        try (ContexteDao ctx = new ContexteDao()) {
            fichiergw.setCtx(ctx);
            final ClauseWhere whereCodeParent = whereCodeParentEgal(code);
            final ClauseOrderBy orderByIdRessource = new ClauseOrderBy("ID_RESSOURCE", SensDeTri.DESC);
            final RequeteSQL requeteCodeParentTrie = new RequeteSQL();
            requeteCodeParentTrie.where(whereCodeParent).orderBy(orderByIdRessource);
            // soit on a un code parent
            fichiergw.select(requeteCodeParentTrie.formaterRequete());
            if (fichiergw.nextItem()) {
                return fichiergw;
            }
        } catch (final Exception e) {
            LOGGER.error("Fichier code=" + code + " introuvable.", e);
        }
        return null;
    }

    /**
     * Renvoie le fichier lie a un objet parent.
     *
     * @param _ctx
     *            the _ctx
     * @param _code
     *            the _code
     *
     * @return the fichier
     *
     * @deprecated le contexte n'est pas nécessaire. Il faut utiliser {@link Ressource#getFichier(String)}
     */
    @Deprecated
    public static Ressource getFichier(final OMContext _ctx, final String _code) {
        final Ressource fichiergw = new Ressource();
        fichiergw.init();
        fichiergw.setCtx(_ctx);
        try {
            // soit on a un code parent
            if (_code.contains("TYPE=")) {
                final ClauseWhere whereCodeParent = whereCodeParentEgal(_code);
                final ClauseOrderBy orderByIdRessource = new ClauseOrderBy("ID_RESSOURCE", SensDeTri.DESC);
                final RequeteSQL requeteCodeParentTrie = new RequeteSQL();
                requeteCodeParentTrie.where(whereCodeParent).orderBy(orderByIdRessource);
                fichiergw.select(requeteCodeParentTrie.formaterRequete());
                if (fichiergw.nextItem()) {
                    return fichiergw;
                }
            }
            // soit un id
            else {
                fichiergw.setIdRessource(new Long(_code));
                fichiergw.retrieve();
                return fichiergw;
            }
        } catch (final Exception e) {
            LOGGER.error("Fichier code/id=" + _code + " introuvable.", e);
        }
        return null;
    }

    /**
     * Renvoie le fichier lié au code fourni
     *
     * @param codeParentOuId
     *            Le code parent ou l'id de la ressources...
     *
     * @return le fichier lié ou null si rien n'a été trouvé
     *
     */
    public static Ressource getFichier(final String codeParentOuId) {
        final Ressource fichiergw = new Ressource();
        fichiergw.init();
        final ContexteDao ctx = new ContexteDao();
        fichiergw.setCtx(ctx);
        try {
            // soit on a un code parent
            if (codeParentOuId.contains("TYPE=")) {
                final ClauseWhere whereCodeParent = whereCodeParentEgal(codeParentOuId);
                final ClauseOrderBy orderByIdRessource = new ClauseOrderBy("ID_RESSOURCE", SensDeTri.DESC);
                final RequeteSQL requeteCodeParentTrie = new RequeteSQL();
                requeteCodeParentTrie.where(whereCodeParent).orderBy(orderByIdRessource);
                fichiergw.select(requeteCodeParentTrie.formaterRequete());
                if (fichiergw.nextItem()) {
                    return fichiergw;
                }
            }
            // soit un id
            else {
                fichiergw.setIdRessource(new Long(codeParentOuId));
                fichiergw.retrieve();
                return fichiergw;
            }
        } catch (final Exception e) {
            LOGGER.error("Fichier code/id=" + codeParentOuId + " introuvable.", e);
        } finally {
            ctx.close();
        }
        return null;
    }

    /**
     * Gets the path absolu.
     *
     * @return le path physique absolu du fichier
     */
    public String getPathAbsolu() {
        return getMedia().getPathAbsolu();
    }

    /**
     * Checks if is photo.
     *
     * @return vrai si le fichier est du type photo
     */
    @JsonIgnore
    public boolean isPhoto() {
        return "photo".equalsIgnoreCase(getMedia().getTypeRessource());
    }

    /**
     * Checks if is photo.
     *
     * @return vrai si le fichier est du type photo
     */
    public Ressource getPhoto() {
        if (isPhoto()) {
            return this;
        } else {
            return new Ressource();
        }
    }

    /**
     * Gets the url photo.
     *
     * @return l'url de la photo
     *
     */
    public String getUrlPhoto() {
        String url = StringUtils.EMPTY;
        if (isPhoto()) {
            url = getMedia().getUrlAbsolue();
            if (StringUtils.contains(getCodeParent(), ",TYPE=") && "2".equals(getEtat())) {
                url += "?ID_FICHE=" + StringUtils.substringBefore(getCodeParent(), ",");
            }
        }
        return url;
    }

    /**
     * Gets the largeur photo.
     *
     * @return la largeur de la photo
     *
     */
    public int getLargeurPhoto() {
        int largeur = 0;
        if (isPhoto()) {
            try {
                largeur = Integer.parseInt(getMedia().getSpecificData(MediaPhoto.ATTRIBUT_LARGEUR));
            } catch (final Exception e) {
                largeur = MediaPhoto.getCritereLimite().getLargeur();
            }
        }
        return largeur;
    }

    /**
     * Gets the largeur vignette.
     *
     * @return la largeur de la vignette
     */
    public int getLargeurVignette() {
        return MediaPhoto.getCritereVignette().getLargeur();
    }

    /**
     * Gets the hauteur photo.
     *
     * @return la hauteur de la photo
     *
     */
    public int getHauteurPhoto() {
        int hauteur = 0;
        if (isPhoto()) {
            try {
                hauteur = Integer.parseInt(getMedia().getSpecificData(MediaPhoto.ATTRIBUT_HAUTEUR));
            } catch (final Exception e) {
                hauteur = MediaPhoto.getCritereLimite().getHauteur();
            }
        }
        return hauteur;
    }

    /**
     * Gets the hauteur vignette.
     *
     * @return la hauteur de la vignette
     *
     */
    public int getHauteurVignette() {
        return MediaPhoto.getCritereVignette().getHauteur();
    }

    /**
     * Checks if is local.
     *
     * @return true, if is local
     */
    public boolean isLocal() {
        boolean bLocal = false;
        try {
            new URL(getMedia().getUrl());
        } catch (final MalformedURLException e) {
            if (!getMedia().getUrl().contains("/")) {
                bLocal = true;
            }
        }
        return bLocal;
    }

    /**
     * Gets the media.
     *
     * @return the media
     */
    public Media getMedia() {
        Media res = new Media();
        res.init();
        if (media == null && getIdMedia() != null && getIdMedia() != 0) {
            try (ContexteDao ctxDao = new ContexteDao()) {
                final Media m = new Media();
                m.init();
                m.setCtx(ctxDao);
                m.setIdMedia(getIdMedia());
                m.retrieve();
                media = m;
                ContexteUniv ctxUniv = ContexteUtil.getContexteUniv();
                if (ctxUniv != null && ctxUniv.getLocale() != null && !ctxUniv.getLocale().equals(LangueUtil.getDefaultLocale())) {
                    media.setLangue(LangueUtil.getIndiceLocale(ctxUniv.getLocale()));
                }
                res = media;
            } catch (final Exception ignored) {
            }
        } else if (media != null) {
            res = media;
        }
        return res;
    }

    /**
     * Gets the auteur.
     *
     * @return the auteur
     */
    public String getAuteur() {
        return getMedia().getAuteur();
    }

    /**
     * Gets the code redacteur.
     *
     * @return the code redacteur
     */
    public String getCodeRedacteur() {
        return getMedia().getCodeRedacteur();
    }

    /**
     * Gets the copyright.
     *
     * @return the copyright
     */
    public String getCopyright() {
        return getMedia().getCopyright();
    }

    /**
     * Gets the date creation.
     *
     * @return the date creation
     */
    public Date getDateCreation() {
        return getMedia().getDateCreation();
    }

    /**
     * Gets the description.
     *
     * @return the description
     */
    public String getDescription() {
        return getMedia().getDescription();
    }

    /**
     * Gets the format.
     *
     * @return the format
     */
    public String getFormat() {
        return getMedia().getFormat();
    }

    /**
     * Gets the legende.
     *
     * @return the legende
     */
    public String getLegende() {
        return getMedia().getLegende();
    }

    /**
     * Gets the meta keywords.
     *
     * @return the meta keywords
     */
    public String getMetaKeywords() {
        return getMedia().getMetaKeywords();
    }

    /**
     * Gets the poids.
     *
     * @return the poids
     */
    public Integer getPoids() {
        return getMedia().getPoids();
    }

    /**
     * Gets the specific data.
     *
     * @return the specific data
     */
    public String getSpecificData() {
        return getMedia().getSpecificData();
    }

    /**
     * Gets the source.
     *
     * @return the source
     */
    public String getSource() {
        return getMedia().getSource();
    }

    /**
     * Gets the titre.
     *
     * @return the titre
     */
    public String getTitre() {
        return getMedia().getTitre();
    }

    /**
     * Checks if is securise.
     *
     * @return true, if is securise
     */
    public boolean isSecurise() {
        return !getMedia().isPublic();
    }

    /**
     * Gets the libelle.
     *
     * @return the libelle
     */
    public String getLibelle() {
        return getMedia().getLibelleAffichable();
    }

    /**
     * Gets the type media.
     *
     * @return the type media
     */
    public String getTypeMedia() {
        return getMedia().getTypeMedia();
    }

    /**
     * Gets the type ressource.
     *
     * @return the type ressource
     */
    public String getTypeRessource() {
        return getMedia().getTypeRessource();
    }

    /**
     * Gets the url.
     *
     * @return the url
     */
    public String getUrl() {
        String url = StringUtils.EMPTY;
        if (getMedia() != null) {
            url = media.getUrlAbsolue();
            if (StringUtils.contains(getCodeParent(), ",TYPE=") && "2".equals(getEtat())) {
                url += "?ID_FICHE=" + StringUtils.substringBefore(getCodeParent(), ",");
            }
        }
        return url;
    }

    /**
     * Gets the url vignette.
     *
     * @return the url vignette
     */
    public String getUrlVignette() {
        return getMedia().getUrlVignetteAbsolue();
    }

    /**
     * Retourne si possible l'extension de la ressource
     *
     * @return l'extension de la ressource, si le nom de la ressource se termine par un . ou un -
     */
    public String getExtension() {
        return getMedia().getExtension();
    }
}
