/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.Vector;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.kportal.cache.CacheUtil;
import com.kportal.core.config.MessageHelper;
import com.univ.mediatheque.UsingLibelleMedia;
import com.univ.multisites.InfosSite;
import com.univ.objetspartages.cache.CacheRubriqueManager;
import com.univ.objetspartages.sgbd.RubriqueDB;
import com.univ.url.CacheUrlManager;
import com.univ.url.CacheUrlRubrique;
import com.univ.utils.ContexteDao;
import com.univ.utils.ContexteUniv;
import com.univ.utils.UnivWebFmt;
import com.univ.utils.sql.RequeteSQL;
import com.univ.utils.sql.clause.ClauseOrderBy;
import com.univ.utils.sql.clause.ClauseOrderBy.SensDeTri;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.xhtml.JavascriptTreeRubrique;

/**
 * The Class Rubrique.
 */
public class Rubrique extends RubriqueDB implements UsingLibelleMedia {

	/**
	 *
	 */
	private static final long serialVersionUID = 7434176178304119510L;
	
	private static final Logger LOGGER = LoggerFactory.getLogger(Rubrique.class);

	/** The Constant CODE_RUBRIQUE_ROOT. */
	public final static String CODE_RUBRIQUE_ROOT = "00";

	/** The Constant CODE_RUBRIQUE_INEXISTANTE. */
	public final static String CODE_RUBRIQUE_INEXISTANTE = "ZYZYZYZYZYZYZYZYZYZYZY";

	/**
	 * Inits the.
	 */
	public void init() {
		setIdRubrique(0L);
		setCode(String.valueOf(System.currentTimeMillis()));
		setLangue("0");
		setIntitule("");
		setAccroche("");
		setIdBandeau(0L);
		setCouleurFond("");
		setCouleurTitre("");
		setNomOnglet("");
		setCodeRubriqueMere("");
		setTypeRubrique("");
		setPageAccueil("");
		setGestionEncadre("0");
		setEncadre("");
		setEncadreSousRubrique("0");
		setOrdre("");
		setContact("");
		setGroupesDsi("");
		setRequetesRubriquePublication("");
		setCategorie("");
		setIdPicto(0L);
	}

	/**
	 * Renvoie le contenu formate en HTML pour l'attribut accroche.
	 *
	 * @return the formated accroche
	 *
	 * @throws Exception
	 *             the exception
	 */
	public String getFormatedAccroche() throws Exception {
		return UnivWebFmt.formaterEnHTML((ContexteUniv) ctx, getAccroche());
	}

	/**
	 * Renvoie le contenu formate en HTML pour l'attribut encadre.
	 *
	 * @return the formated encadre
	 *
	 * @throws Exception
	 *             the exception
	 */
	public String getFormatedEncadre() throws Exception {
		return UnivWebFmt.formaterEnHTML((ContexteUniv) ctx, getEncadre());
	}

	/**
	 * Suppression d'une rubrique . On vérifie que la rubrique est vide, elle ne doit contenir : - ni page libres - ni sous-rubriques - ni articles
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	public void delete() throws Exception {
		final InfosRubriques infos = Rubrique.renvoyerItemRubrique(getCode());
		/* Sélection des objets API implémentant l'interface RubriqueExterne */
		FicheUniv maFicheUniv = null;
		String nomObjet = null;
		//boucle sur les objets qui peuvent être rattachés à une rubrique
		for (final String codeObjet : ReferentielObjets.getListeCodesObjet()) {
			nomObjet = ReferentielObjets.getNomObjet(codeObjet);
			maFicheUniv = ReferentielObjets.instancierFiche(nomObjet);
			if (maFicheUniv != null) {
				maFicheUniv.init();
				maFicheUniv.setCtx(ctx);
				if (maFicheUniv.select("WHERE CODE_RUBRIQUE='" + getCode() + "' AND ETAT_OBJET IN('0001','0002','0003')") > 0) {
					throw new ErreurApplicative(String.format("La rubrique \"%s\" ne peut pas être supprimé : des fiches y sont rattachées.", infos.getLibelleAffichable()));
				}
			}
		}
		/* Sélection des sous-rubriques */
		if (!infos.getListeSousRubriques().isEmpty()) {
			throw new ErreurApplicative(String.format("La rubrique \"%s\" ne peut pas être supprimé : des rubriques y sont rattachées", infos.getLibelleAffichable()));
		}
		super.delete();
		rechargement();
		// JSS 20050126 : suppression des rubriques référencées
		Rubriquepublication.supprimerRubriquePublication(ctx, getCode());
	}

	/**
	 * Forcage du rechargement des rubriques en mémoire.
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static void rechargement() {
		CacheRubriqueManager.getInstance().asyncRefresh();
		CacheUrlManager.getInstance().asyncRefresh();
	}

	/**
	 * Méthode qui permet de tester le périmètre de modification d'un objet sur une rubrique : si le droit en modification renvoit faux on va regarder le droit en création.
	 *
	 * @param autorisations
	 *            the autorisations
	 * @param sPermission
	 *            the s permission
	 * @param codeRubrique
	 *            the code rubrique
	 *
	 * @return true, if controler permission
	 *
	 */
	public static boolean controlerPermission(final AutorisationBean autorisations, final String sPermission, final String codeRubrique) {
		boolean res = false;
		PermissionBean permissionCourante = new PermissionBean(sPermission);
		final Perimetre perimetre = new Perimetre("", codeRubrique, "", "", "");
		res = autorisations.possedePermissionPartielleSurPerimetre(permissionCourante, perimetre);
		if (!res && permissionCourante.getType().equals("FICHE")) {
			if (permissionCourante.getAction().equals("M")) {
				permissionCourante = new PermissionBean(permissionCourante.getType(), permissionCourante.getObjet(), "C");
				if (autorisations.possedePermissionPartielleSurPerimetre(permissionCourante, perimetre)) {
					res = true;
				}
			}
		}
		return res;
	}

	/**
	 * Affichage dynamique de l'arbre javascript des rubriques.
	 *
	 * @param _code
	 *            the _code
	 * @param autorisations
	 *            the autorisations
	 * @param _sPermission
	 *            the _s permission
	 *
	 * @return the arbre java script
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String getArbreJavaScript(final String _code, final AutorisationBean autorisations, final String _sPermission) throws Exception {
		return getArbreJavaScript(_code, autorisations, _sPermission, CODE_RUBRIQUE_ROOT);
	}

	/**
	 * FG 20060712 Affichage dynamique de l'arbre javascript des rubriques pour le front office Le paramètre _codeRacine permet de définir la rubrique de plus haut niveau de
	 * l'arbre Permet de restreindre l'arbre à une sous-arborescence (ex: sous site).
	 *
	 * @param _code
	 *            the _code
	 * @param autorisations
	 *            the autorisations
	 * @param _sPermission
	 *            the _s permission
	 * @param _codeRacine
	 *            the _code racine
	 *
	 * @return the arbre java script
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String getArbreJavaScript(String _code, final AutorisationBean autorisations, final String _sPermission, String _codeRacine) throws Exception {
		if (_codeRacine == null || _codeRacine.length() == 0) {
			_codeRacine = CODE_RUBRIQUE_ROOT;
		}
		if (_code == null || _code.length() == 0) {
			_code = _codeRacine;
		}
		// teste si la rubrique de sélection est une sous-rubrique de la rubrique racine
		if (!_codeRacine.equals(CODE_RUBRIQUE_ROOT) && !_codeRacine.equals(_code) && renvoyerItemRubrique(_code).contains(renvoyerItemRubrique(_codeRacine))) {
			_codeRacine = CODE_RUBRIQUE_ROOT;
		}
		final JavascriptTreeRubrique tree = new JavascriptTreeRubrique(autorisations, _sPermission);
		tree.load(renvoyerItemRubrique(_codeRacine), renvoyerItemRubrique(_code));
		return tree.print();
	}

	/**
	 * Récupération de l'intitulé.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _code
	 *            the _code
	 * @param _inclureRubriqueCourante
	 *            the _inclure rubrique courante
	 *
	 * @return the intitule
	 *
	 * @deprecated le contexte et le booleen ne servent à rien...
	 */
	@Deprecated
	public static String getIntitule(final OMContext _ctx, final String _code, final boolean _inclureRubriqueCourante) {
		String res = "";
		if ((_code == null) || (_code.length() == 0)) {
			return res;
		}
		/* On lit les rubriques dans le sens ascendant */
		res = renvoyerItemRubrique(_code).getIntitule();
		return res;
	}

	/**
	 * Nouvelle méthode avec juste les paramètres dont on a besoin...
	 *
	 * @param codeRubrique
	 *            le code de la rubrique dont on veut l'intitulé
	 * @return l'intitulé de la rubrique ou vide si elle n'existe pas
	 * @throws Exception
	 *             possible sur la récupération de la rubrique
	 */
	public static String getIntitule(final String codeRubrique) {
		String intitule = MessageHelper.getCoreMessage("RUBRIQUE_INEXISTANTE");
		if (StringUtils.isNotBlank(codeRubrique)) {
			intitule = renvoyerItemRubrique(codeRubrique).getIntitule();
		}
		return intitule;
	}

	/**
	 * Récupération de l'intitulé complet (Concaténation mere + fille 1 niveau + ...)
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _code
	 *            the _code
	 * @param _inclureRubriqueCourante
	 *            the _inclure rubrique courante
	 *
	 * @return the intitule complet
	 * @deprecated utiliser {@link Rubrique#getIntituleComplet(String, boolean)}
	 */
	@Deprecated
	public static String getIntituleComplet(final OMContext _ctx, final String _code, final boolean _inclureRubriqueCourante) {
		String res = "";
		if ((_code == null) || (_code.length() == 0)) {
			return res;
		}
		// On lit les rubriques dans le sens ascendant
		InfosRubriques rubrique = renvoyerItemRubrique(_code);
		if (_inclureRubriqueCourante) {
			res = rubrique.getIntitule();
		}
		while (rubrique.getNiveau() > 1) {
			rubrique = rubrique.getRubriqueMere();
			if (res.length() > 0) {
				res = "&nbsp;&gt;&nbsp;" + res;
			}
			res = rubrique.getOnglet() + res;
		}
		return res;
	}

	/**
	 * Récupération de l'intitulé complet (Concaténation mere + fille 1 niveau + ...)
	 *
	 * @param code
	 *            le code de la rubrique à partir du quel on doit partir
	 * @param inclureRubriqueCourante
	 *            Doit on inclure la rubrique courante dans la chaine générée?
	 *
	 * @return L'intitulé de toute les rubriques jusqu'a la racine
	 */
	public static String getIntituleComplet(final String code, final boolean inclureRubriqueCourante) {
		String res = StringUtils.EMPTY;
		if (StringUtils.isEmpty(code)) {
			return res;
		}
		// On lit les rubriques dans le sens ascendant
		InfosRubriques rubrique = renvoyerItemRubrique(code);
		if (inclureRubriqueCourante) {
			res = rubrique.getIntitule();
		}
		while (rubrique.getNiveau() > 1) {
			rubrique = rubrique.getRubriqueMere();
			if (res.length() > 0) {
				res = "&nbsp;&gt;&nbsp;" + res;
			}
			res = rubrique.getOnglet() + res;
		}
		return res;
	}

	/**
	 * Récupération des rubriques.
	 *
	 * @param _ctx
	 *            the _ctx
	 *
	 * @return la liste rubriques par intitule complet
	 * @deprecated utiliser {@link Rubrique#getListeRubriquesParIntituleComplet()}
	 */
	@Deprecated
	public static Hashtable<String, String> getListeRubriquesParIntituleComplet(final OMContext _ctx) {
		final HashMap<String, InfosRubriques> mapRubrique = obtenirListeRubriques();
		final Iterator<InfosRubriques> it = mapRubrique.values().iterator();
		final Hashtable<String, String> res = new Hashtable<String, String>(mapRubrique.size());
		while (it.hasNext()) {
			final InfosRubriques info = it.next();
			res.put(info.getCode(), Rubrique.getIntituleComplet(info.getCode(), true));
		}
		return res;
	}

	/**
	 * Récupération des rubriques.
	 *
	 *
	 * @return la liste rubriques par intitule complet
	 *
	 *
	 */
	public static Hashtable<String, String> getListeRubriquesParIntituleComplet() {
		final HashMap<String, InfosRubriques> mapRubrique = obtenirListeRubriques();
		final Iterator<InfosRubriques> it = mapRubrique.values().iterator();
		final Hashtable<String, String> res = new Hashtable<String, String>(mapRubrique.size());
		while (it.hasNext()) {
			final InfosRubriques info = it.next();
			res.put(info.getCode(), Rubrique.getIntituleComplet(info.getCode(), true));
		}
		return res;
	}

	/**
	 * Récupération des rubriques.
	 *
	 * @param _ctx
	 *            the _ctx
	 *
	 * @return the liste codes rubriques
	 */
	public static Vector<String> getListeCodesRubriques(final OMContext _ctx) {
		final HashMap<String, InfosRubriques> mapRubriques = obtenirListeRubriques();
		final Vector<String> res = new Vector<>();
		for (InfosRubriques infosRubriques : mapRubriques.values()) {
			res.add(infosRubriques.getCode());
		}
		return res;
	}

	/**
	 * Récupère l'ensemble des codes de rubriques présent en base.
	 * @return la liste complète des codes de rubrique.
	 */
	public static List<String> getListeCodesRubriques() {
		final HashMap<String, InfosRubriques> mapRubriques = obtenirListeRubriques();
		final List<String> res = new ArrayList<>();
		for (InfosRubriques infosRubriques : mapRubriques.values()) {
			res.add(infosRubriques.getCode());
		}
		return res;
	}

	private static HashMap<String, InfosRubriques> obtenirListeRubriques() {
		return (HashMap<String, InfosRubriques>) CacheRubriqueManager.getInstance().call();
	}

	public static CacheUrlRubrique getCacheUrlRubrique() {
		final Object res = CacheUtil.getObjectValue(CacheUrlManager.KEY_CACHE, CacheUrlManager.KEY_CACHE);
		if (res != null) {
			return (CacheUrlRubrique) res;
		} else {
			ApplicationContextManager.getCoreContextBean(CacheUrlManager.ID_BEAN);
			return new CacheUrlRubrique();
		}
	}

	/**
	 * Renvoie la liste des rubriques principales (premier niveau) triée en fonction de l'ordre (méthode back, pas de dsi).
	 *
	 * @return Une liste d'InfosRubrique
	 *
	 */
	public static Collection<InfosRubriques> getListeRubriquesPrincipale() {
		return getListeRubriquesPrincipale(null, false);
	}

	/**
	 * Renvoie la liste des rubriques principales (premier niveau) triée en fonction de l'ordre.
	 *
	 * @param ctx
	 *            le contexte
	 * @param front
	 *            si front on applique la dsi
	 *
	 * @return Une liste d'InfosRubrique
	 *
	 */
	public static Collection<InfosRubriques> getListeRubriquesPrincipale(final ContexteUniv ctx, final boolean front) {
		//preload();
		final InfosRubriques topLevelRubrique = getTopLevelRubrique();
		Collection<InfosRubriques> list = null;
		if (front) {
			list = topLevelRubrique.getListeSousRubriquesFront(ctx);
		} else {
			list = topLevelRubrique.getListeSousRubriques();
		}
		return list;
	}

	/**
	 * Renvoie la rubrique de plus haut niveau (mère des rubriques de 1er niveau).
	 *
	 * @return Un InfosRubrique
	 *
	 */
	public static InfosRubriques getTopLevelRubrique() {
		return obtenirListeRubriques().get(CODE_RUBRIQUE_ROOT);
	}

	/**
	 * Récupération de l'objet rubrique.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _code
	 *            the _code
	 * @param _langue
	 *            the _langue
	 *
	 * @return the fiche rubrique
	 *
	 * @throws Exception
	 *             the exception
	 * @deprecated utiliser {@link Rubrique#getRubrique(String, String)} le contexte ne sert à rien ici...
	 */
	@Deprecated
	public static Rubrique getFicheRubrique(final OMContext _ctx, final String _code, final String _langue) throws Exception {
		Rubrique res = null, rubrique = null;
		if ((_code == null) || (_code.length() == 0)) {
			return rubrique;
		}
		String langue = _langue;
		if ((langue == null) || (langue.length() == 0)) {
			langue = "0";
		}
		rubrique = new Rubrique();
		rubrique.init();
		rubrique.setCtx(_ctx);
		// On cherche d'abord la version en ligne puis les autres versions
		int count = rubrique.selectCodeLangue(_code, langue);
		if (count > 0) {
			while (rubrique.nextItem()) {
				res = rubrique;
			}
		}
		return res;
	}

	/**
	 *
	 * @param code
	 * @param langue
	 * @return
	 * @throws Exception
	 * @deprecated cette méthode est déprécié étant donné que son nom peut engendrer un mal entendu avec la fiche en page de tête de rubrique.
	 * Utiliser {@link Rubrique#getRubrique(String, String)}
	 */
	@Deprecated
	public static Rubrique getFicheRubrique(final String code, final String langue) throws Exception {
		return getRubrique(code, langue);
	}
	/**
	 * Migration de getFicheRubrique.
	 * Récupére la rubrique suivant le code et la langue fourni en paramètre.
	 * @param code le code de la rubrique, si non renseigné la rubrique retourné sera null
	 * @param langue la langue de la rubrique
	 * @return Une rubrique ou null si non trouvé
	 * @throws Exception
	 */
	public static Rubrique getRubrique(final String code, final String langue) throws Exception {
		Rubrique res = null;
		if (StringUtils.isEmpty(code)) {
			return null;
		}
		Rubrique rubrique = new Rubrique();
		rubrique.init();
		try (ContexteDao ctx = new ContexteDao()) {
			rubrique.setCtx(ctx);
			// On cherche d'abord la version en ligne puis les autres versions
			int count = rubrique.selectCodeLangue(code, StringUtils.defaultIfEmpty(langue,"0"));
			if (count > 0) {
				while (rubrique.nextItem()) {
					res = rubrique;
				}
			}
		}
		return res;
	}

	/**
	 * Récupére la rubrique suivant le code fourni en paramètre.
	 * @param code le code de la rubrique, si non renseigné la rubrique retourné sera null
	 * @return Une rubrique ou null si non trouvé
	 * @throws Exception
	 */
	public static Rubrique getRubriqueByCode(final String code) throws Exception {
		Rubrique res = null;
		if (StringUtils.isEmpty(code)) {
			return null;
		}
		Rubrique rubrique = new Rubrique();
		rubrique.init();
		try (ContexteDao ctx = new ContexteDao()) {
			rubrique.setCtx(ctx);
			int count = rubrique.selectCodeLangue(code, StringUtils.EMPTY);
			if (count > 0) {
				while (rubrique.nextItem()) {
					res = rubrique;
				}
			}
		}
		return res;
	}

	/**
	 * Récupération d'une rubrique stockée en mémoire.
	 *
	 * @param code
	 *            the code
	 *
	 * @return the infos rubriques
	 *
	 */
	public static InfosRubriques renvoyerItemRubrique(final String code) {
		InfosRubriques res = obtenirListeRubriques().get(code);
		/* Structure vide pour éviter les plantages */
		if (res == null) {
			res = new InfosRubriques("");
		}
		return res;
	}

	/**
	 * Renvoie la rubrique à afficher dans l'encadré, qui peut être différente de la rubrique courante lorsqu'on affiche une liste d'articles.
	 *
	 * @param _codeRubrique
	 *            Le code de la rubrique dont on souhaite récupérer l'encadré.
	 *
	 * @return L'encadré de la rubrique ou une chaine vide si non retrouvé
	 *
	 */
	public static String renvoyerRubriqueEncadre(final String _codeRubrique) {
		final HashMap<String, InfosRubriques> mapRubriques = obtenirListeRubriques();
		String rubriqueNavigation = StringUtils.EMPTY;
		final InfosRubriques infos = mapRubriques.get(_codeRubrique);
		if (infos == null) {
			return rubriqueNavigation;
		}
		if (infos.getTypeRubrique().equals("0001")) {
			rubriqueNavigation = _codeRubrique;
		} else {
			rubriqueNavigation = Rubrique.renvoyerItemRubrique(_codeRubrique).getCodeRubriqueMere();
		}
		return rubriqueNavigation;
	}

	/**
	 * Sélection d'une rubrique à partir de l'ensemble des critères combinés.
	 *
	 * @param code
	 *            the _code
	 * @param langue
	 *            the langue
	 * @param intitule
	 *            the _intitule
	 * @param codeSaisi
	 *            the _code saisi
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 */
	public int select(final String code, final String langue, final String intitule, final String codeSaisi) throws Exception {
		String codeAVerifier = "";
		final RequeteSQL requeteSelect = new RequeteSQL();
		final ClauseOrderBy orderBy = new ClauseOrderBy("INTITULE", SensDeTri.ASC);
		final ClauseWhere where = new ClauseWhere();
		if (StringUtils.isNotEmpty(code)) {
			codeAVerifier = code;
		} else if (StringUtils.isNotEmpty(codeSaisi)) {
			codeAVerifier = codeSaisi;
		}
		if (StringUtils.isNotEmpty(codeAVerifier)) {
			where.setPremiereCondition(ConditionHelper.egalVarchar("CODE", codeAVerifier));
		}
		if (StringUtils.isNotEmpty(langue) && !"0000".equals(langue)) {
			where.and(ConditionHelper.egalVarchar("LANGUE", langue));
		}
		if (StringUtils.isNotEmpty(intitule)) {
			where.and(ConditionHelper.rechercheMots("INTITULE", intitule));
		}
		requeteSelect.where(where).orderBy(orderBy);
		return select(requeteSelect.formaterRequete());
	}

	/**
	 * FIXME : La selection sur l'état n'est pas gérer????? Select code langue etat.
	 *
	 * @param code
	 *            the _code
	 * @param langue
	 *            the langue
	 * @param etatObjet
	 *            the _etat objet
	 *
	 * @return the int
	 *
	 * @throws Exception
	 *             the exception
	 * @deprecated il n'y a pas d'état sur les rubriques... {@link Rubrique#selectCodeLangue}
	 */
	@Deprecated
	public int selectCodeLangueEtat(final String code, final String langue, final String etatObjet) throws Exception {
		final RequeteSQL requeteSelect = new RequeteSQL();
		final ClauseOrderBy orderBy = new ClauseOrderBy("INTITULE", SensDeTri.ASC);
		final ClauseWhere where = new ClauseWhere();
		if (StringUtils.isNotEmpty(code)) {
			where.setPremiereCondition(ConditionHelper.egalVarchar("CODE", code));
		}
		if (StringUtils.isNotEmpty(langue)) {
			where.and(ConditionHelper.egalVarchar("LANGUE", langue));
		}
		requeteSelect.where(where).orderBy(orderBy);
		return select(requeteSelect.formaterRequete());
	}
	/**
	 * Calcule la rubrique en fonction du code et de la langue fourni en paramètre
	 *
	 * @param code
	 *            le code de la rubrique
	 * @param langue
	 *            la langue de la rubrique
	 *
	 * @return le nombre de résultat obtenu
	 *
	 * @throws Exception
	 *             lors de la requête SQL
	 */
	public int selectCodeLangue(final String code, final String langue) throws Exception {
		final RequeteSQL requeteSelect = new RequeteSQL();
		final ClauseOrderBy orderBy = new ClauseOrderBy("INTITULE", SensDeTri.ASC);
		final ClauseWhere where = new ClauseWhere();
		if (StringUtils.isNotEmpty(code)) {
			where.setPremiereCondition(ConditionHelper.egalVarchar("CODE", code));
		}
		if (StringUtils.isNotEmpty(langue)) {
			where.and(ConditionHelper.egalVarchar("LANGUE", langue));
		}
		requeteSelect.where(where).orderBy(orderBy);
		return select(requeteSelect.formaterRequete());
	}

	/**
	 * Renvoie les infos du bandeau a partir du code de la rubrique.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param code
	 *            the _code
	 * @param _langue
	 *            the langue
	 *
	 * @retun InfosBandeau
	 * @deprecated utiliser {@link Rubrique#getInfosBandeau(String)} car le contexte ne sert à rien idem pour la langue...
	 */
	@Deprecated
	public static InfosBandeau getInfosBandeau(final OMContext _ctx, final String code, final String _langue) {
		final InfosBandeau infosBandeau = new InfosBandeau();
		if (StringUtils.isBlank(code)) {
			return infosBandeau;
		}
		// Recherche du bandeau  et des couleurs
		// On recherche la première rubrique dans l'arborescence contenant un bandeau
		String codeRubrique = code;
		InfosRubriques infos = renvoyerItemRubrique(codeRubrique);
		int niveau = infos.getNiveau();
		String urlBandeau = Media.getMedia(infos.getIdMediaBandeau()).getUrlAbsolue();
		String couleurFond = infos.getCouleurFond();
		String couleurTitre =infos.getCouleurTitre();
		while ((niveau > 1) && (urlBandeau.length() == 0)) {
			codeRubrique = infos.getCodeRubriqueMere();
			infos = renvoyerItemRubrique(codeRubrique);
			urlBandeau = Media.getMedia(infos.getIdMediaBandeau()).getUrlAbsolue();
			couleurFond = infos.getCouleurFond();
			couleurTitre = infos.getCouleurTitre();
			niveau--;
		}
		infosBandeau.setUrlBandeau(urlBandeau);
		infosBandeau.setCouleurFond(couleurFond);
		infosBandeau.setCouleurTitre(couleurTitre);
		return infosBandeau;
	}

	/**
	 * Renvoie les infos du bandeau a partir du code de la rubrique.
	 *
	 * @param code
	 *            le code de la rubrique
	 *
	 * @return InfosBandeau
	 */
	public static InfosBandeau getInfosBandeau(final String code) {
		final InfosBandeau infosBandeau = new InfosBandeau();
		if (StringUtils.isBlank(code)) {
			return infosBandeau;
		}
		// Recherche du bandeau  et des couleurs
		// On recherche la première rubrique dans l'arborescence contenant un bandeau
		String codeRubrique = code;
		InfosRubriques infos = renvoyerItemRubrique(codeRubrique);
		int niveau = infos.getNiveau();
		String urlBandeau = Media.getMedia(infos.getIdMediaBandeau()).getUrlAbsolue();
		String couleurFond = infos.getCouleurFond();
		String couleurTitre =infos.getCouleurTitre();
		while ((niveau > 1) && (urlBandeau.length() == 0)) {
			codeRubrique = infos.getCodeRubriqueMere();
			infos = renvoyerItemRubrique(codeRubrique);
			urlBandeau = Media.getMedia(infos.getIdMediaBandeau()).getUrlAbsolue();
			couleurFond = infos.getCouleurFond();
			couleurTitre = infos.getCouleurTitre();
			niveau--;
		}
		infosBandeau.setUrlBandeau(urlBandeau);
		infosBandeau.setCouleurFond(couleurFond);
		infosBandeau.setCouleurTitre(couleurTitre);
		return infosBandeau;
	}

	/**
	 * Controle si l'utilisateur a les droits sur la rubrique de la fiche courante.
	 *
	 * @param ctx
	 *            contexte pour la base et les autorisations du user
	 * @param codeRubriqueFiche
	 *            the code rubrique fiche
	 *
	 * @return boolean accesOK
	 */
	public static boolean controlerRestrictionRubrique(final ContexteUniv ctx, final String codeRubriqueFiche) {
		boolean accesOK = true;
		if (StringUtils.isNotBlank(codeRubriqueFiche)) {
			boolean boucleSurLaRubriqueParente = true;
			boolean droitRubriqueCourante = true;
			// on recupere tous les groupes de l'utilisateur
			final Set<String> listeGroupes = ctx.getGroupesDsiAvecAscendants();
			// on initialise le test avec la rubrique de la fiche
			InfosRubriques infoRubrique = null;
			String codeRubriqueCourante = codeRubriqueFiche;
			while (boucleSurLaRubriqueParente) {
				droitRubriqueCourante = false;
				infoRubrique = Rubrique.renvoyerItemRubrique(codeRubriqueCourante);
				// si la rubrique a des groupes de restriction
				final Set<String> groupesDSI = infoRubrique.getGroupesDsi();
				if (groupesDSI != null && !groupesDSI.isEmpty()) {
					for (final String codeGroupe : groupesDSI) {
						if (listeGroupes.contains(codeGroupe)) {
							droitRubriqueCourante = true;
							break;
						}
					}
				} else {
					droitRubriqueCourante = true;
				}
				// si droit sur la rubrique courante on boucle sur la rubrique parente si elle existe
				if (droitRubriqueCourante) {
					if (infoRubrique.getCodeRubriqueMere().length() > 0) {
						codeRubriqueCourante = infoRubrique.getCodeRubriqueMere();
					} else {
						boucleSurLaRubriqueParente = false;
					}
				}
				// sinon on sort de la boucle et on renvoit false
				else {
					boucleSurLaRubriqueParente = false;
					accesOK = false;
				}
			}
		}
		return accesOK;
	}

	/*
	 * Cette méthode renvoit une rubrique de la fiche dans le site courant
	 * par defaut la rubrique principale ou la première de ces rubriques de publication
	 * si précisé, on cherchera en priorité la rubrique incluse dans la rubrique courante sinon on renverra celle par défaut
	 */
	/**
	 * Gets the rubrique publication.
	 *
	 * @param ctx
	 *            the ctx
	 * @param infosRubriqueCourante
	 *            the infos rubrique courante
	 * @param lstCodeRubriquePubliable
	 *            the lst code rubrique publiable
	 * @param _inRubriqueCourante
	 *            the _in rubrique courante
	 *
	 * @return the rubrique publication
	 *
	 */
	public static String getRubriquePublication(final OMContext ctx, final InfosRubriques infosRubriqueCourante, final ArrayList<String> lstCodeRubriquePubliable,
		final boolean _inRubriqueCourante) {
		final InfosSite siteCourant = ctx.getInfosSite();
		boolean controlDsi = false;
		boolean rubriqueAutorisee = false;
		boolean possedeRubrique = false;
		if (ctx instanceof ContexteUniv) {
			controlDsi = true;
		}
		final Iterator<String> it = lstCodeRubriquePubliable.iterator();
		String codeRubrique = null;
		while (it.hasNext()) {
			possedeRubrique = true;
			final InfosRubriques rubrique = Rubrique.renvoyerItemRubrique(it.next());
			// si la rubrique n'existe pas/plus on passe
			if (StringUtils.isEmpty(rubrique.getCode())) {
				continue;
			}
			if (!controlDsi || Rubrique.controlerRestrictionRubrique((ContexteUniv) ctx, rubrique.getCode())) {
				rubriqueAutorisee = true;
				if (siteCourant.isRubriqueVisibleInSite(rubrique)) {
					if (!_inRubriqueCourante) {
						return rubrique.getCode();
					}
					if (codeRubrique == null) {
						codeRubrique = rubrique.getCode();
					}
					if (rubrique.getCode().equals(infosRubriqueCourante.getCode()) || infosRubriqueCourante.getListeSousRubriquesTousNiveaux().contains(rubrique)) {
						return rubrique.getCode();
					}
				}
			}
		}
		if (controlDsi && possedeRubrique && !rubriqueAutorisee) {
			((ContexteUniv) ctx).setCodeRubriquePageCourante(Rubrique.CODE_RUBRIQUE_INEXISTANTE);
		}
		return codeRubrique;
	}

	/**
	 * Détermine de facon récursive la liste des rubriques autorisées pour un utilisateur donné.
	 *
	 * @param ctx
	 *            contexte pour la base et les autorisations du user
	 * @param rubrique
	 *            the rubrique
	 *
	 * @return the collection
	 *
	 */
	public static Collection<InfosRubriques> determinerListeSousRubriquesAutorisees(final ContexteUniv ctx, InfosRubriques rubrique) {
		final Collection<InfosRubriques> listeRubriquesAutorisees = new HashSet<InfosRubriques>();
		if (rubrique == null || rubrique.getCode().length() == 0) {
			rubrique = Rubrique.getTopLevelRubrique();
		}
		determinerListeSousRubriquesAutorisees(ctx.getGroupesDsiAvecAscendants(), rubrique, true, listeRubriquesAutorisees);
		return listeRubriquesAutorisees;
	}

	/**
	 * Détermine la liste des rubriques autorisées pour un utilisateur donné.
	 *
	 * @param listeGroupes
	 *            the liste groupes
	 * @param rubrique
	 *            the rubrique
	 * @param rechercheArborescente
	 *            the recherche arborescente
	 * @param listeRecursive
	 *            the liste recursive
	 *
	 */
	public static void determinerListeSousRubriquesAutorisees(final Set<String> listeGroupes, final InfosRubriques rubrique, final boolean rechercheArborescente,
		final Collection<InfosRubriques> listeRecursive) {
		final Collection<InfosRubriques> listeSousRubriques = rubrique.getListeSousRubriques();
		// on boucle sur les rubriques
		final Iterator<InfosRubriques> iter = listeSousRubriques.iterator();
		InfosRubriques rubriqueCourante = null;
		while (iter.hasNext()) {
			rubriqueCourante = iter.next();
			// JSS 20051013 : bug sur récursivité si pas de DSI
			boolean autorisationRubrique = true;
			// on boucle sur les groupes de restriction de la rubrique
			final Iterator<String> iter2 = rubriqueCourante.getGroupesDsi().iterator();
			if (iter2.hasNext()) {
				autorisationRubrique = false;
				while (iter2.hasNext()) {
					// si l'utilisateur appartient a au moins un groupe de restriction on boucle recursivement
					if (listeGroupes.contains(iter2.next())) {
						autorisationRubrique = true;
					}
				}
			}
			if (autorisationRubrique) {
				listeRecursive.add(rubriqueCourante);
				if (rechercheArborescente && !rubriqueCourante.getListeSousRubriques().isEmpty()) {
					determinerListeSousRubriquesAutorisees(listeGroupes, rubriqueCourante, rechercheArborescente, listeRecursive);
				}
			}
		}
	}

	/**
	 * Gets the fil ariane.
	 *
	 * @param codeRubrique
	 *            the code rubrique
	 * @param separateur
	 *            the separateur
	 *
	 * @return the fil ariane
	 */
	public static String getFilAriane(final String codeRubrique, String separateur) {
		final StringBuilder filAriane = new StringBuilder();
		try {
			if (separateur.length() == 0) {
				separateur = "&gt;";
			}
			// On lit les rubriques dans le sens ascendant
			InfosRubriques rubrique = Rubrique.renvoyerItemRubrique(codeRubrique);
			while (rubrique.getNiveau() > 0) {
				if (filAriane.length() > 0) {
					filAriane.insert(0, " ");
					filAriane.insert(1, separateur);
					filAriane.insert(separateur.length() + 1, " ");
				}
				filAriane.insert(0, rubrique.getIntitule());
				rubrique = rubrique.getRubriqueMere();
			}
		} catch (final Exception e) {}
		return filAriane.toString();
	}

	@Override
	public List<String> getCodesLibelleMedia() {
		// 0100 Bandeau rubrique  est un type photo de libelle utilise dans la mediatheque
		return Arrays.asList(new String[] { "0100" });
	}

	@Override
	public void checkCodesLibelleMedia(final OMContext ctx, final long idMedia) throws Exception {
		// controle bandeau rubrique
		final Rubrique rubrique = new Rubrique();
		rubrique.setCtx(ctx);
		rubrique.init();
		rubrique.select("WHERE ID_BANDEAU=" + idMedia);
		if (rubrique.nextItem()) {
			throw new ErreurApplicative(String.format(MessageHelper.getCoreMessage(ctx.getLocale(), "RUBRIQUE.BANDEAU.ERREUR.SUPPRESSION_RESSOURCE"), rubrique.getIntitule()));
		}
	}
	
    /**
     * Permer de flusher le cache rubrique si l'id média est rattaché à une rubrique
     * @param idMedia
     */
    public static void invaliderCacheRubriqueParMedia(final long idMedia) {
        if (hasRubriqueParIdPictoOuIdBandeau(idMedia)) {
            rechargement();
        }
    }

    /**
     * Récupère la liste des rubriques ou le média est soit un picto soit un bandeau
     * @param idMedia du bandeau ou du picto
     * @return
     */
    public static boolean hasRubriqueParIdPictoOuIdBandeau(final long idMedia) {
        final Rubrique rubrique = new Rubrique();
        rubrique.init();
        try (ContexteDao ctx = new ContexteDao()) {
            rubrique.setCtx(ctx);
            int count = rubrique.select("WHERE ID_BANDEAU=" + idMedia + " OR ID_PICTO=" + idMedia);
            return count > 0;
        } catch (final Exception e) {
            LOGGER.error("Erreur dans la récupération des médias avec l'id médias {}", idMedia, e);
        }
        return false;
    }	
}
