/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.TreeSet;
import java.util.Vector;

import com.kportal.core.config.MessageHelper;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.univ.multisites.InfosFicheComparator;
import com.univ.multisites.InfosFicheReferencee;
import com.univ.multisites.InfosSite;
import com.univ.multisites.Site;
import com.univ.objetspartages.sgbd.RubriquepublicationDB;
import com.univ.utils.ContexteDao;
import com.univ.utils.ContexteUniv;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.condition.ConditionList;
import com.univ.utils.sql.criterespecifique.ConditionHelper;

// TODO: Auto-generated Javadoc
/**
 * The Class Rubriquepublication.
 */
public class Rubriquepublication extends RubriquepublicationDB implements Cloneable {

	private static org.slf4j.Logger LOG = LoggerFactory.getLogger(Rubriquepublication.class);

	/**
	 *
	 */
	private static final long serialVersionUID = -3902618148975148091L;

	/**
	 * Instantiates a new rubriquepublication.
	 */
	public Rubriquepublication() {
		super();
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#clone()
	 */
	@Override
	public Rubriquepublication clone() throws CloneNotSupportedException {
		return (Rubriquepublication) super.clone();
	}

	/**
	 * Inits the.
	 */
	public void init() {
		setIdRubriquepublication(new Long(0));
		setTypeFicheOrig("");
		setCodeFicheOrig("");
		setLangueFicheOrig("");
		setRubriqueDest("");
		setSourceRequete("");
	}

	/**
	 * Gets the liste rubriques publication.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param fiche
	 *            the fiche
	 *
	 * @return the liste rubriques publication
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static Vector<String> getListeRubriquesPublication(final OMContext _ctx, final FicheUniv fiche) throws Exception {
		return getListeRubriquesPublication(_ctx, fiche, "");
	}

	/**
	 * Gets the liste rubriques publication.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param fiche
	 *            the fiche
	 * @param _rubriqueRecherche
	 *            the _rubrique recherche
	 *
	 * @return the liste rubriques publication
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static Vector<String> getListeRubriquesPublication(final OMContext _ctx, final FicheUniv fiche, final String _rubriqueRecherche) throws Exception {
		return getListeRubriquesPublication(_ctx, fiche, _rubriqueRecherche, false);
	}

	/**
	 * Renvoie pour une fiche la liste des rubriques de publication. Il est également possible de préciser un périmètre de recherche sur les rubriques (ex : ne chercher que les
	 * rubriques d'un site) le booleen permet d'identifier les rubriques automatiques #AUTO#+code (utilisé dans controleurUniv pour éviter la suppression de ces rubriques)
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param fiche
	 *            the fiche
	 * @param _rubriqueRecherche
	 *            the _rubrique recherche
	 * @param _controleurUniv
	 *            the _controleur univ
	 *
	 * @return @throws Exception
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static Vector<String> getListeRubriquesPublication(final OMContext _ctx, final FicheUniv fiche, final String _rubriqueRecherche, final boolean _controleurUniv)
		throws Exception {
		final Vector<String> res = new Vector<String>();
		/* 1ère partie de la requête : recherche de la fiche */
		final String typeFiche = ReferentielObjets.getCodeObjet(fiche);
		final String codeFiche = fiche.getCode();
		final String langueFiche = fiche.getLangue();
		/* 2ème partie de la requête : restriction à une rubrique */
		final ArrayList<String> lstCodeRubriqueRecherche = new ArrayList<String>();
		if (_rubriqueRecherche.length() > 0) {
			lstCodeRubriqueRecherche.add(_rubriqueRecherche);
			final InfosRubriques infosRub = Rubrique.renvoyerItemRubrique(_rubriqueRecherche);
			final Collection<InfosRubriques> listeSousRubriques = infosRub.getListeSousRubriquesTousNiveaux();
			if (!listeSousRubriques.isEmpty()) {
				for (final InfosRubriques rubrique : listeSousRubriques) {
					lstCodeRubriqueRecherche.add(rubrique.getCode());
				}
			}
		}
		final Rubriquepublication rubriquePublication = new Rubriquepublication();
		rubriquePublication.setCtx(_ctx);
		rubriquePublication.init();
		String code = "";
		if (rubriquePublication.select(typeFiche, codeFiche, langueFiche) > 0) {
			while (rubriquePublication.nextItem()) {
				code = rubriquePublication.getRubriqueDest();
				//si la rubrique appartient à la rubrique recherche
				if (_rubriqueRecherche.length() == 0 || lstCodeRubriqueRecherche.contains(code)) {
					if (_controleurUniv && rubriquePublication.getSourceRequete().length() > 0) {
						code = "#AUTO#" + code;
					}
					if (!(_controleurUniv && res.contains(code))) {
						res.add(code);
					}
				}
			}
		}
		return res;
	}

	/**
	 * Renvoie pour une fiche la liste des rubriques de publication. Il est également possible de préciser un périmètre de recherche sur les rubriques (ex : ne chercher que les
	 * rubriques d'un site) le booleen permet d'identifier les rubriques automatiques #AUTO#+code (utilisé dans controleurUniv pour éviter la suppression de ces rubriques) FIXME :
	 * Pas de controle des valeurs fournis ?
	 *
	 * @param typeFiche
	 *            the type fiche
	 * @param codeFiche
	 *            the code fiche
	 * @param langueFiche
	 *            the langue fiche
	 *
	 * @return @throws Exception
	 *
	 * @throws Exception
	 *             the exception
	 */
	public int select(final String typeFiche, final String codeFiche, final String langueFiche) throws Exception {
		/* 1ère partie de la requête : recherche de la fiche */
		final ClauseWhere whereTypeCodeLangue = new ClauseWhere(ConditionHelper.egalVarchar("TYPE_FICHE_ORIG", typeFiche));
		whereTypeCodeLangue.and(ConditionHelper.egalVarchar("CODE_FICHE_ORIG", codeFiche));
		whereTypeCodeLangue.and(ConditionHelper.egalVarchar("LANGUE_FICHE_ORIG", langueFiche));
		return select(whereTypeCodeLangue.formaterSQL());
	}

	/**
	 * Suppression des fiches référencées pour unr rubrique.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param codeRubriquePublication
	 *            the code rubrique publication
	 *
	 * @return @throws Exception
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static void supprimerRubriquePublication(final OMContext _ctx, final String codeRubriquePublication) throws Exception {
		if (codeRubriquePublication != null) {
			final Rubriquepublication rubP = new Rubriquepublication();
			rubP.setCtx(_ctx);
			rubP.init();
			final ClauseWhere whereRubDestSource = new ClauseWhere(ConditionHelper.egalVarchar("RUBRIQUE_DEST", codeRubriquePublication));
			whereRubDestSource.and(ConditionHelper.egalVarchar("SOURCE_REQUETE", ""));
			if (rubP.select(whereRubDestSource.formaterSQL()) > 0) {
				while (rubP.nextItem()) {
					rubP.delete();
				}
			}
		}
	}

	/**
	 * Suppression des rubriques de publication d'une fiche.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param fiche
	 *            the fiche
	 * @param _ajouterCritereRequete
	 *            the _ajouter critere requete
	 *
	 * @return @throws Exception
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static void supprimerRubriquePublicationFiche(final OMContext _ctx, final InfosFicheReferencee fiche, final boolean _ajouterCritereRequete) throws Exception {
		if (fiche != null && fiche.getCode() != null) {
			final Rubriquepublication rubP = new Rubriquepublication();
			rubP.setCtx(_ctx);
			rubP.init();
			String sourceRequete = "";
			if (_ajouterCritereRequete) {
				sourceRequete = fiche.getRequete();
			}
			/* Suppression des anciennes valeurs */
			final ClauseWhere where = new ClauseWhere(ConditionHelper.egalVarchar("TYPE_FICHE_ORIG", fiche.getType()));
			where.and(ConditionHelper.egalVarchar("CODE_FICHE_ORIG", fiche.getCode()));
			where.and(ConditionHelper.egalVarchar("LANGUE_FICHE_ORIG", fiche.getLangue()));
			where.and(ConditionHelper.egalVarchar("SOURCE_REQUETE", sourceRequete));
			if (rubP.select(where.formaterSQL()) > 0) {
				while (rubP.nextItem()) {
					rubP.delete();
				}
			}
		}
	}

	/*
	 *  Methode de suppression globale des rubriques de publication d'une fiche
	 *  appelé uniquement dans la méthode FicheUnivMgr.supprimerFiche
	 */
	/**
	 * Delete global.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _ficheUniv
	 *            the _fiche univ
	 *
	 * @throws Exception
	 *             the exception
	 * @deprecated utiliser {@link Rubriquepublication#deleteGlobal(FicheUniv)}
	 */
	@Deprecated
	public static void deleteGlobal(final OMContext _ctx, final FicheUniv _ficheUniv) throws Exception {
		PreparedStatement _stmt = null;
		try {
			final String query = "DELETE FROM RUBRIQUEPUBLICATION WHERE TYPE_FICHE_ORIG='" + ReferentielObjets.getCodeObjet(_ficheUniv) + "' AND CODE_FICHE_ORIG ='" + _ficheUniv.getCode() + "' AND LANGUE_FICHE_ORIG='" + _ficheUniv.getLangue() + "'";
			_stmt = _ctx.getConnection().prepareStatement(query);
			_stmt.executeUpdate();
			_stmt.close();
		} catch (final SQLException exc) {
			throw new Exception("ERROR_IN_METHOD Rubriquepublication.deleteGlobal() " + exc);
		} finally {
			_stmt = null;
		}
	}

	/**
	 * SUpprime l'ensemble des rubriques de publications de la fiche fourni en paramètre.
	 *
	 * @param ficheUniv
	 *            la fiche dont on souhaite supprimer les rubriques de publications
	 *
	 * @throws Exception
	 *             lors de l'éxécution de la requête en base de données
	 */
	public static void deleteGlobal(final FicheUniv ficheUniv) throws Exception {
		final String query = "DELETE FROM RUBRIQUEPUBLICATION WHERE TYPE_FICHE_ORIG='" + ReferentielObjets.getCodeObjet(ficheUniv) + "' AND CODE_FICHE_ORIG ='" + ficheUniv.getCode() + "' AND LANGUE_FICHE_ORIG='" + ficheUniv.getLangue() + "'";
		try (ContexteDao ctx = new ContexteDao(); PreparedStatement _stmt = ctx.getConnection().prepareStatement(query);) {
			_stmt.executeUpdate();
		} catch (final SQLException exc) {
			throw new Exception("ERROR_IN_METHOD Rubriquepublication.deleteGlobal() " + exc);
		}
	}

	/**
	 * Enregistre pour une rubrique la liste des fiches référencées.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param codeRubriquePublication
	 *            the code rubrique publication
	 * @param listeFiches
	 *            the liste fiches
	 *
	 * @return @throws Exception
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static void enregistrerRubriquePublication(final OMContext _ctx, final String codeRubriquePublication, final List<InfosFicheReferencee> listeFiches) throws Exception {
		final Rubriquepublication rubP = new Rubriquepublication();
		rubP.setCtx(_ctx);
		supprimerRubriquePublication(_ctx, codeRubriquePublication);
		/* Insertion des nouvelles fiches */
		final Iterator<InfosFicheReferencee> iter = listeFiches.iterator();
		while (iter.hasNext()) {
			final InfosFicheReferencee infosFiche = iter.next();
			rubP.init();
			rubP.setTypeFicheOrig(infosFiche.getType());
			rubP.setCodeFicheOrig(infosFiche.getCode());
			rubP.setLangueFicheOrig(infosFiche.getLangue());
			rubP.setRubriqueDest(codeRubriquePublication);
			rubP.add();
		}
	}

	/**
	 * Enregistre pour une fiche la liste des rubriques référencées.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param infosFiche
	 *            the infos fiche
	 * @param listeRubriques
	 *            the liste rubriques
	 *
	 * @return @throws Exception
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static void enregistrerRubriquePublicationFiche(final OMContext _ctx, final InfosFicheReferencee infosFiche, final Vector<String> listeRubriques) throws Exception {
		final Rubriquepublication rubP = new Rubriquepublication();
		rubP.setCtx(_ctx);
		supprimerRubriquePublicationFiche(_ctx, infosFiche, false);
		/* Insertion des nouvelles fiches */
		final Enumeration<String> e = listeRubriques.elements();
		while (e.hasMoreElements()) {
			final String codeRubriquePublication = e.nextElement();
			rubP.init();
			rubP.setTypeFicheOrig(infosFiche.getType());
			rubP.setCodeFicheOrig(infosFiche.getCode());
			rubP.setLangueFicheOrig(infosFiche.getLangue());
			rubP.setRubriqueDest(codeRubriquePublication);
			rubP.add();
		}
	}

	/**
	 * Renvoie pour rubrique la liste des fiches référéencés.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _codeRubriquePublication
	 *            the _code rubrique publication
	 * @param _avecInfosFiche
	 *            the _avec infos fiche
	 *
	 * @return @throws Exception
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static ArrayList<InfosFicheReferencee> getListeFichesReferences(final OMContext _ctx, final String _codeRubriquePublication, final boolean _avecInfosFiche)
		throws Exception {
		final ArrayList<InfosFicheReferencee> liste = new ArrayList<InfosFicheReferencee>();
		if (_codeRubriquePublication != null) {
			final Rubriquepublication rubP = new Rubriquepublication();
			rubP.setCtx(_ctx);
			rubP.init();
			/* Suppression des anciennes valeurs */
			final ClauseWhere whereRubDestEtSource = new ClauseWhere(ConditionHelper.egalVarchar("RUBRIQUE_DEST", _codeRubriquePublication));
			whereRubDestEtSource.and(ConditionHelper.egalVarchar("SOURCE_REQUETE", ""));
			if (rubP.select(whereRubDestEtSource.formaterSQL()) > 0) {
				while (rubP.nextItem()) {
					final InfosFicheReferencee infosFiche = new InfosFicheReferencee();
					infosFiche.setCode(rubP.getCodeFicheOrig());
					infosFiche.setLangue(rubP.getLangueFicheOrig());
					infosFiche.setType(rubP.getTypeFicheOrig());
					infosFiche.setRequete(rubP.getSourceRequete());
					// Et on rajoute pour chaque élément le libellé
					if (_avecInfosFiche) {
						final FicheUniv fiche = ReferentielObjets.instancierFiche(ReferentielObjets.getNomObjet(rubP.getTypeFicheOrig()));
						if (fiche != null) {
							fiche.setCtx(_ctx);
							if (fiche.selectCodeLangueEtat(rubP.getCodeFicheOrig(), rubP.getLangueFicheOrig(), "0003") > 0) {
								while (fiche.nextItem()) {
									infosFiche.setIntitule(fiche.getLibelleAffichable());
									infosFiche.setEtat(fiche.getEtatObjet());
									infosFiche.setId(fiche.getIdFiche().toString());
									liste.add(infosFiche);
								}
							}
						} else {
							LOG.warn(String.format("Object with code %s couldn't be instanciated.", rubP.getTypeFicheOrig()));
						}
					} else {
						liste.add(infosFiche);
					}
				}
			}
		}
		return liste;
	}

	/**
	 * Méthode qui teste si la fiche est rubriquée dans le site courant.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _fiche
	 *            the _fiche
	 *
	 * @return the rubrique dans site courant
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String getRubriqueDansSiteCourant(final ContexteUniv _ctx, final FicheUniv _fiche) throws Exception {
		String codeRubriqueTemp = null;
		String codeRubrique = null;
		String codeRubriquePrincipale = null;
		final InfosRubriques infosRubCourante = Rubrique.renvoyerItemRubrique(_ctx.getCodeRubriqueHistorique());
		boolean rubriquePrincipaleAutorisee = false;
		boolean rubriquePublicationAutorisee = false;
		boolean possedeRubrique = false;
		//RP20070727 : on cherche en priorité la rubrique du site courant SI POSSIBLE dans la rubrique courante
		// Test sur rubrique principale
		codeRubriquePrincipale = _fiche.getCodeRubrique();
		if (codeRubriquePrincipale.length() > 0) {
			possedeRubrique = true;
			final InfosRubriques rubriquePrincipale = Rubrique.renvoyerItemRubrique(codeRubriquePrincipale);
			if (Rubrique.controlerRestrictionRubrique(_ctx, rubriquePrincipale.getCode())) {
				if (_ctx.getInfosSite().isRubriqueVisibleInSite(rubriquePrincipale)) {
					codeRubriqueTemp = rubriquePrincipale.getCode();
				}
				rubriquePrincipaleAutorisee = true;
			}
		}
		codeRubriquePrincipale = null;
		// on sauvegarde la rubrique principale dans le site courant
		if (codeRubriqueTemp != null) {
			// si la rubrique principale est dans le site ET dans la rubrique courante on la renvoit
			codeRubriquePrincipale = codeRubriqueTemp;
			if (infosRubCourante.getCode().equals(codeRubriqueTemp) || infosRubCourante.getListeSousRubriquesTousNiveaux().contains(Rubrique.renvoyerItemRubrique(codeRubriqueTemp))) {
				codeRubrique = codeRubriqueTemp;
			}
		}
		// test sur rubriques de publication
		if (codeRubrique == null) {
			final String codeRubriqueSite = _ctx.getInfosSite().getCodeRubrique();
			final Vector<String> listeRubriquesPublication = Rubriquepublication.getListeRubriquesPublication(_ctx, _fiche, codeRubriqueSite);
			final Iterator<String> itListeRubrique = listeRubriquesPublication.iterator();
			InfosRubriques rubriquePub = null;
			while (itListeRubrique.hasNext()) {
				possedeRubrique = true;
				rubriquePub = Rubrique.renvoyerItemRubrique(itListeRubrique.next());
				if (Rubrique.controlerRestrictionRubrique(_ctx, rubriquePub.getCode())) {
					rubriquePublicationAutorisee = true;
					if (_ctx.getInfosSite().isRubriqueVisibleInSite(rubriquePub)) {
						codeRubriqueTemp = rubriquePub.getCode();
						if (infosRubCourante.getCode().equals(codeRubriqueTemp) || infosRubCourante.getListeSousRubriquesTousNiveaux().contains(rubriquePub)) {
							codeRubrique = codeRubriqueTemp;
							break;
						}
					}
				}
			}
		}
		// si pas de rubrique de publication dans la rubrique courante
		if (codeRubrique == null) {
			// on renvoit en priorité la rubrique principale du site
			if (codeRubriquePrincipale != null) {
				codeRubrique = codeRubriquePrincipale;
				// sinon une des rubriques de publication
			} else {
				codeRubrique = codeRubriqueTemp;
			}
		}
		if (possedeRubrique && !rubriquePrincipaleAutorisee && !rubriquePublicationAutorisee) {
			_ctx.setCodeRubriquePageCourante(Rubrique.CODE_RUBRIQUE_INEXISTANTE);
		}
		return codeRubrique;
	}

	/**
	 * Renvoie le site d'affichage de la fiche si elle ne s'affiche pas dans le site courant.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _fiche
	 *            the _fiche
	 *
	 * @return the site affichage
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static InfosSite getSiteAffichage(final ContexteUniv _ctx, final FicheUniv _fiche) throws Exception {
		// on regarde si la fiche est rattachée dans un autre site
		// par sa rubrique principale
		String rubriquePrincipale = "";
		if (_fiche != null) {
			rubriquePrincipale = _fiche.getCodeRubrique();
		}
		InfosSite infosSite = Site.determinerSiteRubrique(rubriquePrincipale);
		if (infosSite == null) {
			// par ses rubriques de publication si rattachement à aucun site
			final Vector<String> listeRubriquesPublications = Rubriquepublication.getListeRubriquesPublication(_ctx, _fiche);
			if (listeRubriquesPublications.size() > 0) {
				final Iterator<String> it = listeRubriquesPublications.iterator();
				while (it.hasNext() && infosSite == null) {
					infosSite = Site.determinerSiteRubrique(it.next());
				}
			}
		}
		return infosSite;
	}

	/**
	 * Supprime le rattachement des fiches à une rubrique provenant d'une requete de publication automatique car cette requete a ete supprimee AA 200805 : ajout d'un test car les
	 * requetes comportent desormais un critère de plus : la langue leur syntaxe diffère donc entre les anciennes requetes et les nouvelles.
	 *
	 * @param ctx
	 *            the _ctx
	 * @param codeRubrique
	 *            the _code rubrique
	 * @param requete
	 *            the _requete
	 * @param suppressionEffective
	 *            the _suppression effective
	 *
	 * @return type treeset( new infosFichecomparator() )
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static TreeSet<InfosFicheReferencee> supprimerRubriquePublicationAutomatique(final OMContext ctx, final String codeRubrique, final String requete,
		final boolean suppressionEffective) throws Exception {
		final TreeSet<InfosFicheReferencee> listeReferencement = new TreeSet<InfosFicheReferencee>(new InfosFicheComparator());
		if (codeRubrique != null && requete != null) {
			final Rubriquepublication rubP = new Rubriquepublication();
			rubP.setCtx(ctx);
			rubP.init();
			// sauvegarde des fiches pour indexation
			String requeteOldFormat = "";
			final String item[] = requete.split("/", -2);
			//si la requete est au nouveau format et que la langue de cette requete n'est pas renseignée, on recherche la requete aussi dans l'ancien format
			if (item.length > 4 && item[3].length() == 0) {
				requeteOldFormat = item[0] + "/" + item[1] + "/" + item[2] + "/" + item[4];
			}
			final ClauseWhere whereRupPub = new ClauseWhere(ConditionHelper.egalVarchar("RUBRIQUE_DEST", codeRubrique));
			final ConditionList condSourceRequete = new ConditionList(ConditionHelper.egalVarchar("SOURCE_REQUETE", requete));
			if (StringUtils.isNotEmpty(requeteOldFormat)) {
				condSourceRequete.or(ConditionHelper.egalVarchar("SOURCE_REQUETE", requeteOldFormat));
			}
			whereRupPub.and(condSourceRequete);
			final String requeteFormatee = whereRupPub.formaterSQL();
			if (rubP.select(requeteFormatee) > 0) {
				while (rubP.nextItem()) {
					final InfosFicheReferencee infosFiche = new InfosFicheReferencee();
					infosFiche.setCode(rubP.getCodeFicheOrig());
					infosFiche.setLangue(rubP.getLangueFicheOrig());
					infosFiche.setType(rubP.getTypeFicheOrig());
					infosFiche.setRequete(rubP.getSourceRequete());
					listeReferencement.add(infosFiche);
				}
			}
			// suppression
			if (suppressionEffective && listeReferencement.size() > 0) {
				java.sql.PreparedStatement _stmt = null;
				try {
					final String query = "DELETE FROM RUBRIQUEPUBLICATION " + requeteFormatee;
					_stmt = ctx.getConnection().prepareStatement(query);
					_stmt.executeUpdate();
					_stmt.close();
				} catch (final SQLException exc) {
					throw new ErreurApplicative(String.format(MessageHelper.getCoreMessage(ctx.getLocale(),"RUBRIQUE_PUBLICATION.ERREUR.SUPPRESSION_FICHE_IMPOSSIBLE"), requeteFormatee));
				} finally {
					_stmt = null;
				}
			}
		}
		return listeReferencement;
	}
}
