/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.StringTokenizer;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.univ.objetspartages.cache.CacheStructureManager;
import com.univ.utils.Chaine;
import com.univ.utils.ContexteDao;
import com.univ.utils.ContexteUtil;
import com.univ.utils.FicheUnivMgr;
import com.univ.xhtml.JavascriptTreeStructure;

/**
 * The Class Structure.
 */
public class Structure {

	/** The Constant CODE_STRUCTURE_ROOT. */
	public static final String CODE_STRUCTURE_ROOT = "00";
	
	public static final String NOM_OBJET_STRUCURE ="structure";
	
	private static final Logger LOG = LoggerFactory.getLogger(Structure.class);

	/**
	 * Contrôle le code des objets de type structure : - ne doit pas contenir des caractères interdits - ne doit pas déjà exister dans un autre objet de type Structure.
	 *
	 * @param _code
	 *            the _code
	 * @param _langue
	 *            the _langue
	 * @param _ctx
	 *            the _ctx
	 *
	 * @return the string
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String checkCode(final String _code, final String _langue, final OMContext _ctx) throws Exception {
		Chaine.controlerCodeMetier(_code);
		FicheUniv ficheUniv;
		StructureModele structure;
		// controle de l'unicite
		for (final String codeObjet : ReferentielObjets.getListeCodesObjet()) {
			ficheUniv = ReferentielObjets.instancierFiche(codeObjet);
			if (ficheUniv != null && ficheUniv instanceof StructureModele) {
				structure = (StructureModele) ficheUniv;
				structure.init();
				structure.setCtx(_ctx);
				final int count = ficheUniv.selectCodeLangueEtat(_code, _langue, "");
				if (count > 0) {
					ficheUniv.nextItem();
					throw new ErreurApplicative(
						"Ce code est déjà utilisé pour la fiche " + ReferentielObjets.getNomObjet(ReferentielObjets.getCodeObjet(ficheUniv)) + " : " + ficheUniv.getLibelleAffichable());
				}
			}
		}
		return _code;
	}

	/**
	 *
	 * Check les cycles dans l'arbre de structure.
	 *
	 * @param _code
	 * @param _langue
	 * @return
	 * @throws ErreurApplicative
	 */
	public static void checkCycle(final String _code, final String _langue, final String _codeRattachement) throws ErreurApplicative {
		final InfosStructure infosStructure = Structure.renvoyerItemStructure(_code, _langue);
		final InfosStructure infosStructureMere = Structure.renvoyerItemStructure(_codeRattachement, _langue);
		if (infosStructure.contains(infosStructureMere)) {
			throw new ErreurApplicative("La structure parente ne peut pas être fille de la structure courante.");
		}
	}

	/**
	 * méthode qui permet de tester le périmètre de modification d'un objet sur une structure : si le droit en modification renvoie faux on va regarder le droit en création.
	 *
	 * @param autorisations
	 *            the autorisations
	 * @param sPermission
	 *            the s permission
	 * @param codeStructure
	 *            the code structure
	 *
	 * @return true, if controler permission
	 *
	 */
	public static boolean controlerPermission(final AutorisationBean autorisations, final String sPermission, final String codeStructure) {
		PermissionBean permissionCourante = new PermissionBean(sPermission);
		final Perimetre perimetre = new Perimetre(codeStructure, "", "", "", "");
		boolean res = autorisations.possedePermissionPartielleSurPerimetre(permissionCourante, perimetre);
		if (!res && permissionCourante.getType().equals("FICHE")) {
			if (permissionCourante.getAction().equals("M")) {
				permissionCourante = new PermissionBean(permissionCourante.getType(), permissionCourante.getObjet(), "C");
				if (autorisations.possedePermissionPartielleSurPerimetre(permissionCourante, perimetre)) {
					res = true;
				}
			}
		}
		return res;
	}

	/**
	 * Récupération du libellé de la composante d'une structure.
	 *
	 * @param _code
	 *            the _code
	 * @param _langue
	 *            the _langue
	 * @param autorisations
	 *            the autorisations
	 * @param _sPermission
	 *            the _s permission
	 * @param _sFiltre
	 *            the _s filtre
	 * @param front
	 *            the front
	 * @param _codeRoot
	 *            the _code root
	 * @return the arbre java script
	 */
	public static String getArbreJavaScript(String _code, final String _langue, final AutorisationBean autorisations, final String _sPermission, final String _sFiltre,
		final boolean front, String _codeRoot) {
		if (_code == null || _code.length() == 0) {
			_code = CODE_STRUCTURE_ROOT;
		}
		if (_codeRoot == null || _codeRoot.length() == 0) {
			_codeRoot = CODE_STRUCTURE_ROOT;
		}
		final JavascriptTreeStructure tree = new JavascriptTreeStructure(autorisations, _sPermission, _sFiltre, front);
		tree.load(getTopLevelStructure(_codeRoot, _langue), renvoyerItemStructure(_code, _langue));
		return tree.print();
	}

	/**
	 * Récupération du code de la composante d'une structure.
	 *
	 * @param _code
	 *            the _code
	 * @param _langue
	 *            the _langue
	 *
	 * @return the structure premier niveau
	 *
	 */
	public static InfosStructure getStructurePremierNiveau(final String _code, final String _langue) {
		if ((_code == null) || (_code.length() == 0)) {
			return new InfosStructure();
		}
		// preload();
		// On recherche la composante
		InfosStructure structure = renvoyerItemStructure(_code, _langue);
		while (structure.getNiveau() > 1) {
			structure = structure.getStructureRattachement();
		}
		return structure;
	}

	@Deprecated
	public static StructureModele getFicheStructure(final OMContext _ctx, final String _code, final String _langue) throws Exception {
		return getFicheStructure(_code, _langue);
	}

	/**
	 * Récupération d'une fiche structure en base à partir de son code et de sa langue si le contexte est de type ContexteUniv, seules les fiches en état "en ligne" seront
	 * renvoyées sinon, si aucune fiche en ligne n'existe, une fiche dans un autre état sera renvoyée.
	 *
	 * @param _code
	 *            code de la fiche
	 * @param _langue
	 *            langue de la fiche
	 *
	 * @return the fiche structure
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static StructureModele getFicheStructure(final String _code, final String _langue) throws Exception {
		boolean etatEnLigne = false;
		// si le contexte est ContexteUniv, on est en front-office, donc on ne
		// renvoie pas la fiche si elle n'existe pas en ligne
		if (ContexteUtil.getContexteUniv() != null) {
			etatEnLigne = true;
		}
		return getFicheStructure(_code, _langue, etatEnLigne);
	}

	@Deprecated
	public static StructureModele getFicheStructure(final OMContext _ctx, final String _code, final String _langue, final boolean _etatEnLigne) throws Exception {
		return getFicheStructure(_code, _langue, _etatEnLigne);
	}

	/**
	 * Récupération d'une fiche structure en base à partir de son code et de sa langue.
	 *
	 * @param _code
	 *            code de la fiche
	 * @param _langue
	 *            langue de la fiche
	 * @param _etatEnLigne
	 *            : permet de spécifier si on souhaite uniquement les fiches en ligne (true), ou non (false)
	 *
	 * @return the fiche structure
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static StructureModele getFicheStructure(final String _code, final String _langue, final boolean _etatEnLigne) throws Exception {
		StructureModele res = null, structure = null;
		if (StringUtils.isEmpty(_code)) {
			return structure;
		}
		final String langue = StringUtils.defaultIfEmpty(_langue, "0");
		FicheUniv ficheUniv;
		for (final String codeObjet : ReferentielObjets.getListeCodesObjet()) {
			if (res == null) {
				ficheUniv = ReferentielObjets.instancierFiche(codeObjet);
				if (ficheUniv != null && ficheUniv instanceof StructureModele) {
					structure = (StructureModele) ficheUniv;
					structure.init();
					try (ContexteDao ctx = new ContexteDao()) {
						structure.setCtx(ctx);
						// On cherche d'abord la version en ligne puis les autres versions
						int count = ficheUniv.selectCodeLangueEtat(_code, langue, "0003");
						if (count == 0 && !_etatEnLigne) {
							count = ficheUniv.selectCodeLangueEtat(_code, langue, "");
						}
						if (count > 0) {
							while (structure.nextItem()) {
								res = structure;
							}
						}
					}
				}
			}
		}
		return res;
	}

	@Deprecated
	public static String getLibelleAffichable(final OMContext _ctx, final String _codes, final String _langue) throws Exception {
		return getLibelleAffichable(_codes, _langue);
	}

	/**
	 * Récupération du libellé de la structure.
	 *
	 * @param _codes
	 *            the _codes
	 * @param _langue
	 *            the _langue
	 *
	 * @return the libelle affichable
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String getLibelleAffichable(final String _codes, final String _langue) throws Exception {
		String res = "";
		// preload();
		if ((_codes == null) || (_codes.length() == 0)) {
			return res;
		}
		final StringTokenizer st = new StringTokenizer(_codes, ";");
		String code = "";
		while (st.hasMoreTokens()) {
			code = st.nextToken();
			final InfosStructure info = renvoyerItemStructure(code, _langue);
			String libelleAffichable = info.getLibelleLong();
			// Pas de structure chargée en mémoire (peut-être
			// dans un autre état (type brouillon) : on cherche dans la base
			if (libelleAffichable.length() == 0) {
				final StructureModele structure = getFicheStructure(code, _langue);
				if (structure != null) {
					libelleAffichable = structure.getLibelleAffichable();
				}
			}
			if (res.length() > 0) {
				res += ";";
			}
			if (libelleAffichable != null && libelleAffichable.length() > 0) {
				res += libelleAffichable;
			} else {
				res += '-';
			}
		}
		if (res.length() == 0) {
			res = "-";
		}
		return res;
	}

	/**
	 * Récupération du libellé de la composante d'une structure.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _code
	 *            the _code
	 * @param _langue
	 *            the _langue
	 *
	 * @return the libelle structure premier niveau
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String getLibelleStructurePremierNiveau(final OMContext _ctx, final String _code, final String _langue) throws Exception {
		String res = "-";
		if ((_code == null) || (_code.length() == 0)) {
			return res;
		}
		// preload();
		/* On recherche la composante */
		final InfosStructure structure = getStructurePremierNiveau(_code, _langue);
		res = getLibelleAffichable(structure.getCode(), _langue);
		return res;
	}

	/**
	 * Récupération du libellé de rattachement (Concaténation composante Département ...)
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _code
	 *            the _code
	 * @param _langue
	 *            the _langue
	 * @param inclureStructureDepart
	 *            the inclure structure depart
	 *
	 * @return the libelle rattachement
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String getLibelleRattachement(final OMContext _ctx, final String _code, final String _langue, final boolean inclureStructureDepart) throws Exception {
		String res = "";
		if ((_code == null) || (_code.length() == 0)) {
			return res;
		}
		// preload();
		// On lit les structures dans le sens ascendant
		InfosStructure structure = renvoyerItemStructure(_code);
		if (inclureStructureDepart) {
			res = getLibelleAffichable(structure.getCode(), _langue);
		}
		while (structure.getNiveau() > 1) {
			structure = structure.getStructureRattachement();
			if (res.length() > 0) {
				res = "<br />" + res;
			}
			res = getLibelleAffichable(structure.getCode(), _langue) + res;
		}
		return res;
	}

	public static HashMap<String, ElementArboStructure> getListeStructures() {
		final CacheStructureManager cacheStructureManager = (CacheStructureManager) ApplicationContextManager.getCoreContextBean(CacheStructureManager.ID_BEAN);
		return cacheStructureManager.getListeStructures();
	}

	/**
	 * Renvoie la structure de plus haut niveau (mère des structures de 1er niveau).
	 *
	 * @param codeRoot
	 *            the code root
	 * @param _langue
	 *            the _langue
	 * @return Un InfosStructure
	 */
	public static InfosStructure getTopLevelStructure(final String codeRoot, final String _langue) {
		return getListeStructures().get(codeRoot).getInfosStructure(_langue);
	}

	/**
	 * Renvoie la structure de plus haut niveau (mère des structures de 1er niveau).
	 *
	 * @param _langue
	 *            the _langue
	 *
	 * @return Un InfosStructure
	 *
	 */
	public static InfosStructure getTopLevelStructure(final String _langue) {
		return getTopLevelStructure(CODE_STRUCTURE_ROOT, _langue);
	}

	/**
	 * Récupération des composantes en front (si front, filtre les structures non visibles).
	 *
	 * @param _langue
	 *            the _langue
	 * @param front
	 *            the front
	 *
	 * @return the liste structures premier niveau
	 *
	 */
	public static Hashtable<String, String> getListeStructuresPremierNiveau(final String _langue, final boolean front) {
		final Hashtable<String, String> res = new Hashtable<String, String>();
		/* On recherche les composantes */
		final InfosStructure topLevelStructure = getTopLevelStructure(_langue);
		final Iterator<InfosStructure> it = topLevelStructure.getListeSousStructures().iterator();
		InfosStructure structure = null;
		while (it.hasNext()) {
			structure = it.next();
			if (structure.isVisibleInFront() || !front) {
				res.put(structure.getCode(), structure.getLibelleCourt().toLowerCase());
			}
		}
		return res;
	}

	/**
	 * Récupération d'une structure stockée en mémoire.
	 *
	 * @param code
	 *            the code
	 *
	 * @return the infos structure
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static InfosStructure renvoyerItemStructure(final String code) {
		return renvoyerItemStructure(code, "0");
	}

	/**
	 * Récupération de toutes les structures avec leur chemin complet.
	 *
	 * @param _ctx
	 *            the _ctx
	 *
	 * @return the liste structure par intitule complet
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static Hashtable<String, String> getListeStructureParIntituleComplet(final OMContext _ctx) throws Exception {
		final Iterator<ElementArboStructure> it = getListeStructures().values().iterator();
		final Hashtable<String, String> res = new Hashtable<String, String>();
		while (it.hasNext()) {
			final ElementArboStructure eltArbo = it.next();
			final InfosStructure info = renvoyerItemStructure(eltArbo.getCode());
			String libelle = Structure.getLibelleRattachement(_ctx, info.getCode(), info.getLangue(), true);
			libelle = StringUtils.replace(libelle, "<br />", " > ");
			res.put(info.getCode(), libelle);
		}
		return res;
	}

	/**
	 * Récupération du libellé de la structure.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _codes
	 *            the _codes
	 * @param _langue
	 *            the _langue
	 *
	 * @return the libelle affichable long
	 *
	 * @throws Exception
	 *             the exception
	 * @deprecated Méthode plus utilisée
	 */
	@Deprecated
	public static String getLibelleAffichableLong(final OMContext _ctx, final String _codes, final String _langue) throws Exception {
		String res = "";
		if ((_codes == null) || (_codes.length() == 0)) {
			return res;
		}
		final StringTokenizer st = new StringTokenizer(_codes, ";");
		while (st.hasMoreTokens()) {
			final String code = st.nextToken();
			// VIN 23092004 Refonte des structures
			final StructureModele structure = getFicheStructure(code, _langue);
			if (res.length() > 0) {
				res += ";";
			}
			if (structure != null) {
				res += structure.getLibelleLong();
			} else {
				res += '-';
			}
		}
		if (res.length() == 0) {
			res = "-";
		}
		return res;
	}

	/**
	 * Récupération d'une structure stockée en mémoire.
	 *
	 * @param code
	 *            the code
	 * @param langue
	 *            the langue
	 * @return the infos structure
	 */
	public static InfosStructure renvoyerItemStructure(final String code, final String langue) {
		final ElementArboStructure elementArbo = getListeStructures().get(code);
		InfosStructure structure = null;
		if (elementArbo != null) {
			structure = elementArbo.getInfosStructure(langue);
			if (structure == null) {
				// si la structure dans la langue demandée n'est pas disponible,
				// on prend la première structure existante ayant le même code
				if (!elementArbo.getListeInfosStructure().isEmpty()) {
					structure = elementArbo.getListeInfosStructure().values().iterator().next();
				}
			}
		}
		if (structure == null) {
			structure = new InfosStructure();
		}
		return structure;
	}

	/**
	 * Gets the fil ariane.
	 *
	 * @param codeStructure
	 *            the code structure
	 * @param langue
	 *            the langue
	 *
	 * @return the fil ariane
	 */
	public static String getFilAriane(final String codeStructure, final String langue) {
		String filAriane = "";
		try {
			final String separateur = "&gt;";
			// On lit les structures dans le sens ascendant
			InfosStructure structure = Structure.renvoyerItemStructure(codeStructure, langue);
			String libelleStructure = "";
			while (structure.getNiveau() > 0) {
				if (filAriane.length() > 0) {
					filAriane = " " + separateur + " " + filAriane;
				}
				libelleStructure = structure.getLibelleCourt();
				// si le libellé court n'est pas renseigné, on prend le libellé
				// long
				if ("".equals(libelleStructure)) {
					libelleStructure = structure.getLibelleLong();
				}
				filAriane = libelleStructure + filAriane;
				structure = structure.getStructureRattachement();
			}
		} catch (final Exception e) {}
		return filAriane;
	}

	public static void forcerRechargement() {
		final CacheStructureManager cacheStructureManager = (CacheStructureManager) ApplicationContextManager.getCoreContextBean(CacheStructureManager.ID_BEAN);
		cacheStructureManager.flush();
	}

	/**
	 * Permet de récuperer le nom de l'objet de la structure 
	 * @param codeObjet
	 * @param langue
	 * @return
	 */
	public static String getNomObjetStructureParCodeLangue(String codeObjet, String langue) {
		StructureModele structure = null;
		try {
			structure = getFicheStructure(codeObjet, langue);
		} catch (Exception e) {
			LOG.error("Erreur lors de la récupération de la structure", e);
		}
		if(structure != null){
			return ReferentielObjets.getNomObjet(structure);
		}
		return StringUtils.EMPTY;
	}
}
