/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.om;

import java.sql.Date;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.TreeSet;
import java.util.Vector;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.SynchroniseurUtilisateur;
import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.kportal.core.config.PropertyHelper;
import com.kportal.core.security.MySQLHelper;
import com.univ.collaboratif.om.Espacecollaboratif;
import com.univ.objetspartages.sgbd.UtilisateurDB;
import com.univ.utils.ContexteDao;
import com.univ.utils.RequeteGroupeUtil;
import com.univ.utils.sql.ConstanteSQL;
import com.univ.utils.sql.RequeteSQL;
import com.univ.utils.sql.clause.ClauseJoin;
import com.univ.utils.sql.clause.ClauseJoin.TypeJointure;
import com.univ.utils.sql.clause.ClauseOrderBy;
import com.univ.utils.sql.clause.ClauseOrderBy.SensDeTri;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.condition.Condition;
import com.univ.utils.sql.condition.ConditionList;
import com.univ.utils.sql.criterespecifique.ClauseJoinHelper;
import com.univ.utils.sql.criterespecifique.ConditionHelper;

/**
 * The Class Utilisateur.
 */
public class Utilisateur extends UtilisateurDB {

	private static final Logger LOG = LoggerFactory.getLogger(Utilisateur.class);

	// JSS 20040409 : délégation
	// Groupe temporaire pour mémoriser dans une fiche utilisateur
	// les groupes au cours d'un traitement impliquant plusieurs fiches utilisateurs
	// (n'est pas chargé ni sauvegardé automatiquement)
	// utilisé dans UpdateFromLdap
	/** The vecteur groupes dsi tmp. */
	private Vector<String> vecteurGroupesDsiTmp;

	/**
	 * La source d'authentification de l'application kbd,ldap, cas ou ctrust
	 */
	private static final String SOURCE_AUTHENTIFICATION = PropertyHelper.getCoreProperty("authentification.source");

	/**
	 * CODE DE l'utilisateur anonyme
	 */
	public static final String UTILISATEUR_ANONYME = "ANONYME";

	/**
	 * Source d'authentification cleartrust
	 */
	private static final String SOURCE_AUTHENTIFICATION_CLEARTRUST = "ctrust";

	/**
	 * Source d'authentification cas
	 */
	private static final String SOURCE_AUTHENTIFICATION_CAS = "cas";

	/**
	 * Source d'authentification ldap
	 */
	private static final String SOURCE_AUTHENTIFICATION_LDAP = "ldap";

	/**
	 * @see com.univ.objetspartages.sgbd.UtilisateurDB#add()
	 */
	@Override
	public void add() throws Exception {
		super.add();
		//	 JB, FBO 20050810: synchro user
		SynchroniseurUtilisateur.getInstance().synchroUtilisateur(SynchroniseurUtilisateur.SYNCHRO_USER_ADD, this, ctx);
	}

	/**
	 * @see com.univ.objetspartages.sgbd.UtilisateurDB#delete()
	 */
	@Override
	public void delete() throws Exception {
		super.delete();
		//	 JB, FBO 20050810: synchro user
		SynchroniseurUtilisateur.getInstance().synchroUtilisateur(SynchroniseurUtilisateur.SYNCHRO_USER_DELETE, this, ctx);
	}

	/**
	 * @see com.univ.objetspartages.sgbd.UtilisateurDB#update()
	 */
	@Override
	public void update() throws Exception {
		super.update();
		//	 JB, FBO 20050810: synchro user
		SynchroniseurUtilisateur.getInstance().synchroUtilisateur(SynchroniseurUtilisateur.SYNCHRO_USER_UPDATE, this, ctx);
	}

	/**
	 * Inits the.
	 */
	public void init() {
		setIdUtilisateur((long) 0);
		setCode(StringUtils.EMPTY);
		setMotDePasse(StringUtils.EMPTY);
		setDateNaissance(new java.sql.Date(0));
		setCivilite(StringUtils.EMPTY);
		setNom(StringUtils.EMPTY);
		setPrenom(StringUtils.EMPTY);
		setCodeRattachement(StringUtils.EMPTY);
		setGroupes(StringUtils.EMPTY);
		setAdresseMail(StringUtils.EMPTY);
		setExtensionModification(StringUtils.EMPTY);
		setRestrictionValidation(StringUtils.EMPTY);
		setCentresInteret(StringUtils.EMPTY);
		setProfilDsi(StringUtils.EMPTY);
		setGroupesDsiImport(StringUtils.EMPTY);
		setCodeLdap(StringUtils.EMPTY);
		setRoles(StringUtils.EMPTY);
		setVecteurGroupesDsiTmp(new Vector<String>());
		setProfilDefaut(StringUtils.EMPTY);
		setSourceImport(StringUtils.EMPTY);
		setDateDerniereSession(new Date(0));
		setFormatEnvoi(StringUtils.EMPTY);
		// RP20051118 : activation du mode de saisie expert
		setModeSaisieExpert("1");
		// JB20051124 : timestamp sur le chargement des groupes rattaches
		setTsCacheGroupes((long) 0);
	}

	/**
	 * Renvoie l'adresse mail d'un utilisateur.
	 *
	 * @param ctx
	 *            the _ctx
	 * @param codeUtilisateur
	 *            the _code utilisateur
	 *
	 * @return the adresse mail
	 *
	 * @throws Exception
	 *             the exception
	 * @deprecated ne pas utiliesr car cette méthode utilise le contexte pour rien. Utiliser la même méthode sans le contexte {@link Utilisateur#getAdresseMail(String)}
	 */
	@Deprecated
	public static String getAdresseMail(final OMContext ctx, final String codeUtilisateur) throws Exception {
		return getAdresseMail(codeUtilisateur);
	}

	/**
	 * Renvoie l'adresse mail d'un utilisateur.
	 *
	 * @param codeUtilisateur
	 *            le code de l'utilisateur dont on souhaite récupérer le mail
	 *
	 * @return l'adresse mail de l'utilisateur associé
	 *
	 * @throws Exception
	 *             Lorsque la requête ne peut aboutir
	 */
	public static String getAdresseMail(final String codeUtilisateur) throws Exception {
		return getAdresseMail(codeUtilisateur, null);
	}

	/**
	 * Renvoie l'adresse mail d'un utilisateur, et si l'utilisateur est anonyme, recup du mail dans le metatag.
	 *
	 * @param ctx
	 *            the _ctx
	 * @param codeUtilisateur
	 *            the _code utilisateur
	 * @param meta
	 *            the _meta
	 *
	 * @return the adresse mail
	 *
	 * @throws Exception
	 *             the exception
	 * @deprecated ne pas utiliser car cette méthode utilises le contexte pour rien. Utiliser {@link Utilisateur#getAdresseMail(String, Metatag)}
	 */
	@Deprecated
	public static String getAdresseMail(final OMContext ctx, final String codeUtilisateur, final Metatag meta) throws Exception {
		String mail = "";
		if (StringUtils.isNotEmpty(codeUtilisateur) && !Utilisateur.UTILISATEUR_ANONYME.equals(codeUtilisateur)) {
			final Utilisateur utilisateur = new Utilisateur();
			utilisateur.setCtx(ctx);
			utilisateur.init();
			final ClauseWhere whereCode = new ClauseWhere(ConditionHelper.egalVarchar("CODE", codeUtilisateur));
			final int count = utilisateur.select(whereCode.formaterSQL());
			if (count == 1) {
				while (utilisateur.nextItem()) {
					mail = utilisateur.getAdresseMail();
				}
			}
		} else {
			if (meta != null) {
				mail = meta.getMetaMailAnonyme();
			}
		}
		return mail;
	}

	/**
	 * Renvoie l'adresse mail d'un utilisateur, et si l'utilisateur est anonyme, recup du mail dans le metatag.
	 *
	 * @param codeUtilisateur le code de l'utilisateur dont on souhaite recupérer l'adresse mail
	 * @param meta en cas d'utilisateur anonyme, l'adresse mail est stocké dans le meta...
	 *
	 * @return l'adresse mail de l'utilisateur ou un chaine vide si non trouvé
	 *
	 * @throws Exception lors des accès en bdd
	 */
	public static String getAdresseMail(final String codeUtilisateur, final Metatag meta) throws Exception {
		String mail = "";
		if (StringUtils.isNotEmpty(codeUtilisateur) && !Utilisateur.UTILISATEUR_ANONYME.equals(codeUtilisateur)) {
			final Utilisateur utilisateur = new Utilisateur();
			try (ContexteDao ctx = new ContexteDao()) {
				utilisateur.setCtx(ctx);
				utilisateur.init();
				final ClauseWhere whereCode = new ClauseWhere(ConditionHelper.egalVarchar("CODE", codeUtilisateur));
				final int count = utilisateur.select(whereCode.formaterSQL());
				if (count == 1) {
					while (utilisateur.nextItem()) {
						mail = utilisateur.getAdresseMail();
					}
				}
			}
		} else {
			if (meta != null) {
				mail = meta.getMetaMailAnonyme();
			}
		}
		return mail;
	}

	/**
	 * Récupération du libellé à afficher.
	 * 
	 * @return the libelle
	 *
	 */
	public String getLibelle() {
		return getNom() + " " + getPrenom();
	}

	/**
	 * Récupération du libellé à afficher JSS 20040409 : gestion de plusieurs codes.
	 *
	 * @param ctx
	 *            the _ctx
	 * @param code
	 *            the _code
	 *
	 * @return the libelle
	 *
	 * @throws Exception
	 *             the exception
	 * @deprecated Utiliser {@link Utilisateur#getLibelle(String)}
	 */
	@Deprecated
	public static String getLibelle(final OMContext ctx, final String code) throws Exception {
		return getLibelle(code);
	}

	/**
	 * Récupération du libellé à afficher JSS 20040409 : gestion de plusieurs codes.
	 *
	 * @param codes le code de l'utilisateur ou une liste de code séparé par des ";" ...
	 *
	 * @return Une chaine vide si non trouvé ou le libellé du code utilisateur ou une liste de libéllé séparé par des ";"...
	 * 
	 * @throws Exception lors de la récupération de l'utilisateur en bdd
	 */
	public static String getLibelle(final String codes) throws Exception {
		String res = "";
		if ((codes == null) || (codes.length() == 0)) {
			return res;
		}
		final StringTokenizer st = new StringTokenizer(codes, ";");
		while (st.hasMoreTokens()) {
			final String code = st.nextToken();
			if (res.length() > 0) {
				res += ";";
			}
			final Utilisateur utilisateur = getUtilisateur(code);
			String intitule = "-";
			if (utilisateur != null) {
				intitule = utilisateur.getLibelle();
			}
			res += intitule;
		}
		return res;
	}

	/**
	 * Récupération de l'objet utilisateur.
	 * 
	 * @param ctx
	 *            the _ctx
	 * @param code
	 *            the _code
	 * 
	 * @return the utilisateur
	 * 
	 * @throws Exception
	 *             the exception
	 * @deprecated le contexte ne sert que pour la requête SQL. Utilisez {@link Utilisateur#getUtilisateur(String)}
	 */
	@Deprecated
	public static Utilisateur getUtilisateur(final OMContext ctx, final String code) throws Exception {
		return getUtilisateur(code);
	}

	/**
	 * Récupération de l'objet utilisateur.
	 *
	 * @param code
	 *            le code de l'utilisateur
	 *
	 * @return l'utilisateur correspondant au code ou un utilisateur vide
	 *
	 * @throws Exception
	 *             lours de la requête en bdd
	 */
	public static Utilisateur getUtilisateur(final String code) throws Exception {
		final Utilisateur utilisateur = new Utilisateur();
		try (ContexteDao ctx = new ContexteDao()) {
			utilisateur.setCtx(ctx);
			utilisateur.init();
			if (StringUtils.isNotBlank(code)) {
				final ClauseWhere whereCode = new ClauseWhere(ConditionHelper.egalVarchar("CODE", code));
				if (utilisateur.select(whereCode.formaterSQL()) > 0) {
					utilisateur.nextItem();
				}
			}
		}
		return utilisateur;
	}
	/**
	 * Despecialise chaine.
	 * 
	 * @param chaine
	 *            the chaine
	 * 
	 * @return the string
	 * @deprecated ne pas despécialiser les valeurs avec cette méthodes! Elle n'échape que certains caractères.
	 * Pour échaper des chaines utiliser  @see EscapeString
	 */
	@Deprecated
	public static String despecialiseChaine(String chaine) {
		if (chaine == null) {
			return chaine;
		}
		final char[] listeCars = { '\\', '\'', '"', '#', '(', ')', ';' };
		for (final char listeCar : listeCars) {
			int idxCar = chaine.indexOf(listeCar);
			while (idxCar != -1) {
				chaine = chaine.substring(0, idxCar) + "\\" + listeCar + chaine.substring(idxCar + 1);
				idxCar += 2;
				idxCar = chaine.indexOf(listeCar, idxCar);
			}
		}
		return chaine;
	}

	/**
	 * Modification de l'adresse mail/password.
	 *
	 * @param code
	 *            the _code
	 * @param mail
	 *            the _mail
	 * @param password
	 *            the _password
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public static void modifierMailPassword(final String code, final String mail, final String password) throws Exception {
		/* Mise à jour de la base utilisateur */
		final Utilisateur utilisateur = Utilisateur.getUtilisateur(code);
		if (utilisateur != null) {
			// JSS 20030305-001		
			if (StringUtils.isNotEmpty(password)) {
				final String message = verifieMotDePasse(password);
				if (message.length() == 0) {
					utilisateur.setMotDePasse(MySQLHelper.encodePassword(password));
				} else {
					throw new ErreurApplicative(message);
				}
			}
			if (mail != null) {
				utilisateur.setAdresseMail(mail);
			}
			try (ContexteDao ctx = new ContexteDao()) {
				utilisateur.setCtx(ctx);
				utilisateur.update();
			}
		} else {
			LOG.warn("Problème mise à jour base utilisateur");
		}
	}

	/**
	 * Renvoie la liste des centres d'intéret sous forme de vecteur.
	 * 
	 * @return the vecteur centres interet
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public Vector<String> getVecteurCentresInteret() throws Exception {
		final Vector<String> v = new Vector<>();
		final StringTokenizer st = new StringTokenizer(StringUtils.defaultString(getCentresInteret()), ";");
		while (st.hasMoreTokens()) {
			v.add(st.nextToken());
		}
		return v;
	}

	/**
	 * @deprecated utiliser {@link Espacecollaboratif#renvoyerCodesUtilisateursParEspaceParGroupe(String)}
	 * @param ctx
	 * @param codeEspace
	 * @return
	 * @throws Exception
	 */
	@Deprecated
	public static Vector<String> renvoyerCodesUtilisateursParEspaceParGroupe(final OMContext ctx, final String codeEspace) throws Exception {
		return new Vector<>(Espacecollaboratif.renvoyerCodesUtilisateursParEspaceParGroupe(codeEspace));
	}

	/**
	 * @deprecated utiliser {@link Espacecollaboratif#renvoyerCodesUtilisateursParEspace(String)}
	 * @param ctx
	 * @param codeEspace
	 * @return
	 * @throws Exception
	 */
	@Deprecated
	public static Vector<String> renvoyerCodesUtilisateursParEspace(final OMContext ctx, final String codeEspace, final boolean tousLesMembres,
																	final Vector<String> autresUtilisateurs) throws Exception {
		return new Vector<>(Espacecollaboratif.renvoyerCodesUtilisateursParEspace(codeEspace));
	}

	/**
	 * Détermine pour un groupe
	 * 
	 * <br>
	 * - la liste des groupes à inclure dans la requête <br>
	 * - la liste des users issus des groupes dynamiques.
	 * 
	 * @param codeGroupe
	 *            the code groupe
	 * @param listeUsers
	 *            the liste users
	 * @param listeGroupes
	 *            the liste groupes
	 * 
	 * @throws Exception
	 *             the exception
	 */
	private void determinerUsersEtGroupes(final String codeGroupe, final Set<String> listeUsers, final Set<String> listeGroupes) throws Exception {
		final InfosGroupeDsi infoGroupe = Groupedsi.renvoyerItemGroupeDsi(codeGroupe);
		if (infoGroupe.getRequete().length() == 0) {
			// Cas d'un groupe statique
			listeGroupes.add(codeGroupe);
			// Calcul des sous-groupes
			final Iterator<InfosGroupeDsi> itGroupes = infoGroupe.getListeSousGroupes().iterator();
			InfosGroupeDsi infoSousGroupe = null;
			while (itGroupes.hasNext()) {
				infoSousGroupe = itGroupes.next();
				determinerUsersEtGroupes(infoSousGroupe.getCode(), listeUsers, listeGroupes);
			}
		} else {
			// cas groupe dynamique
			final Set<String> setCodesGroupesCache = new HashSet<>();
			final Collection<String> listeUsersGroupe = RequeteGroupeUtil.getVecteurUtilisateurs(codeGroupe, setCodesGroupesCache);
			if (setCodesGroupesCache.isEmpty()) {
				listeUsers.addAll(listeUsersGroupe);
			} else {
				for (final String string : setCodesGroupesCache) {
					listeGroupes.add(string);
				}
			}
		}
	}

	/**
	 * Requete de selection des membres d'un espace collaboratif.
	 * 
	 * @param code
	 *            the _code
	 * @param nom
	 *            the _nom
	 * @param prenom
	 *            the _prenom
	 * @param chaineDiffusion
	 *            the _chaine diffusion
	 * @param profil
	 *            the _profil
	 * @param groupe
	 *            the _groupe
	 * @param structure
	 *            the _structure
	 * @param rechercheDansSousGroupes
	 *            the _recherche dans sous groupes
	 * @param adresseMail
	 *            the _adresse mail
	 * 
	 * @return the string
	 * 
	 * @throws Exception
	 *             the exception
	 */
	private String construireRequete(final String code, final String nom, final String prenom, final String chaineDiffusion, final String profil, final String groupe,
		final String structure, final String adresseMail, final boolean rechercheDansSousGroupes) throws Exception {
		final RequeteSQL requeteSelect = new RequeteSQL();
		ClauseWhere where = new ClauseWhere();
		final ClauseOrderBy orderBy = new ClauseOrderBy();
		orderBy.orderBy("T1.NOM", SensDeTri.ASC).orderBy("T1.PRENOM", SensDeTri.ASC);
		if (StringUtils.isNotEmpty(code)) {
			where.setPremiereCondition(ConditionHelper.egalVarchar("CODE", code));
		}
		if (StringUtils.isNotEmpty(nom)) {
			where.and(ConditionHelper.like("NOM", nom, "%", "%"));
		}
		if (StringUtils.isNotEmpty(prenom)) {
			where.and(ConditionHelper.like("PRENOM", prenom, "%", "%"));
		}
		if (StringUtils.isNotEmpty(adresseMail)) {
			where.and(ConditionHelper.like("ADRESSE_MAIL", adresseMail, "%", "%"));
		}
		if (StringUtils.isNotEmpty(chaineDiffusion)) {
			final Collection<String> codeDespecialise = despecialiseChaineDiffusion(chaineDiffusion);
			where.and(getCondtionCodeOuCodeGroupe(codeDespecialise));
		}
		if (StringUtils.isNotEmpty(structure)) {
			where.and(ConditionHelper.getConditionStructure("CODE_RATTACHEMENT", structure));
		}
		if (StringUtils.isNotEmpty(profil) && !"0000".equals(profil)) {
			final InfosProfilDsi infosProfilDsi = Profildsi.renvoyerItemProfilDsi(profil);
			final Vector<String> groupesProfilDsi = infosProfilDsi.getGroupes();
			if (groupesProfilDsi.isEmpty()) {
				where = new ClauseWhere(ConstanteSQL.CONDITION_IMPOSSIBLE);
			} else {
				where.and(getCondtionCodeOuCodeGroupe(groupesProfilDsi));
			}
		}
		if (StringUtils.isNotEmpty(groupe) && !"0000".equals(groupe)) {
			if (rechercheDansSousGroupes) {
				final Collection<String> codeDesGroupes = Arrays.asList(groupe.split(";"));
				where.and(getCondtionCodeOuCodeGroupe(codeDesGroupes));
			} else {
				where.and(ConditionHelper.egalVarchar("T2.CODE_GROUPE", groupe));
			}
		}
		ClauseJoin join = traiterJointureGroupeUtilisateur(where);
		requeteSelect.where(where).join(join).orderBy(orderBy);
		return requeteSelect.formaterRequete();
	}

	/**
	 * CQCB?
	 * 
	 * @param codeDesGroupes
	 * @return
	 * @throws Exception
	 */
	private Condition getCondtionCodeOuCodeGroupe(final Collection<String> codeDesGroupes) throws Exception {
		final ConditionList conditioncodeOuCodeGroupe = new ConditionList();
		if (!codeDesGroupes.isEmpty()) {
			// Le treeset permet d'optimiser la gestion des doublons
			final Set<String> usersDynamiques = new TreeSet<>();
			final Set<String> groupes = new TreeSet<>();
			for (final String groupeDiffusion : codeDesGroupes) {
				if (StringUtils.isNotEmpty(groupeDiffusion)) {
					determinerUsersEtGroupes(groupeDiffusion, usersDynamiques, groupes);
				}
			}
			if (!groupes.isEmpty()) {
				conditioncodeOuCodeGroupe.or(ConditionHelper.in("T2.CODE_GROUPE", groupes));
			}
			if (!usersDynamiques.isEmpty()) {
				conditioncodeOuCodeGroupe.or(ConditionHelper.in("T1.CODE", usersDynamiques));
			}
		}
		if (conditioncodeOuCodeGroupe.isEmpty()) {
			conditioncodeOuCodeGroupe.setPremiereCondtion(ConstanteSQL.CONDITION_IMPOSSIBLE);
		}
		return conditioncodeOuCodeGroupe;
	}

	/**
	 * Neccessaire du au mécanisme de construction de la requête, on ne peut pas savoir autrement à l'heure actuelle si il y a une jointure sur la table GROUPEUTILISATEUR ou non.
	 * 
	 * @param where
	 * @return
	 */
	private ClauseJoin traiterJointureGroupeUtilisateur(final ClauseWhere where) {
		ClauseJoin join = new ClauseJoin();
		final String requeteFormater = where.formaterSQL();
		if (StringUtils.isNotEmpty(requeteFormater) && requeteFormater.contains("T2.")) {
			join = ClauseJoinHelper.creerJointure(TypeJointure.LEFT_JOIN, "GROUPEUTILISATEUR T2", ConditionHelper.critereJointureSimple("T1.CODE", "T2.CODE_UTILISATEUR"));
		}
		return join;
	}

	/**
	 * Controle si l'utilisateur courant appartient au groupe ou a ses sous-groupes.
	 * 
	 * @param codeGroupe
	 *            the _code groupe
	 * @param groupesUtilisateurs
	 *            the groupes utilisateurs
	 * 
	 * @return true, if appartient a groupe ou sous groupes
	 *
	 */
	public boolean appartientAGroupeOuSousGroupes(final String codeGroupe, final Vector<String> groupesUtilisateurs) {
		boolean res = false;
		final InfosGroupeDsi groupe = Groupedsi.renvoyerItemGroupeDsi(codeGroupe);
		// Parcours du vecteur de groupes de l'utilisateurs
		final Iterator<String> itGroupesUtilisateurs = groupesUtilisateurs.iterator();
		InfosGroupeDsi groupeUtilisateur = null;
		while (itGroupesUtilisateurs.hasNext() && !res) {
			groupeUtilisateur = Groupedsi.renvoyerItemGroupeDsi(itGroupesUtilisateurs.next());
			res = groupe.contains(groupeUtilisateur);
		}
		return res;
	}

	/**
	 * Valorise la liste des centres d'interet sous forme de vecteur.
	 * 
	 * @param v
	 *            the new vecteur centres interet
	 *
	 */
	public void setVecteurCentresInteret(final Vector<String> v) {
		setCentresInteret(StringUtils.join(v,";"));
	}

	private static Collection<String> despecialiseChaineDiffusion(final String chaineDiffusion) {
		final Collection<String> listeChaineDiffusion = new ArrayList<>();
		if (StringUtils.isNotEmpty(chaineDiffusion)) {
			final String[] codesChaines = chaineDiffusion.split(";");
			for (String codeChaine : codesChaines) {
				// On enleve les crochets
				codeChaine = codeChaine.substring(1, codeChaine.length() - 1);
				//Extraction groupe
				final int iSlash = codeChaine.indexOf("/");
				final String groupe = codeChaine.substring(iSlash + 1);
				listeChaineDiffusion.add(groupe);
			}
		}
		return listeChaineDiffusion;
	}

	/**
	 * Renvoie la liste des mails des personnes webmaster Chaque element de la liste est un tableau de string [mail][libelle][code].
	 * 
	 * @param ctx
	 *            the _ctx
	 * 
	 * @return the array list
	 * 
	 * @throws Exception
	 *             the exception
	 * @deprecated Méthode plus utilisée, à supprimer
	 */
	@Deprecated
	public static ArrayList<String[]> renvoyerMailsWebmasters(final OMContext ctx) throws Exception {
		final Set<String> mailsWebmaster = Utilisateur.getListeUtilisateursPossedantPermission(new PermissionBean("TECH", "wmg", ""), new ArrayList<String>(), "", "",
			"", true);
		final ArrayList<String[]> res = new ArrayList<>();
		for (String aMailsWebmaster : mailsWebmaster) {
			res.add(aMailsWebmaster.split(";", -2));
		}
		return res;
	}

	/**
	 * Sélection d'un utilisateur
	 * 
	 * Peut donner lieu à des jointures notamment lors des requêtes de groupe qui donnent
	 * 
	 * SELECT DISTINCT T1.CODE FROM UTILISATEUR T1
	 * 
	 * LEFT JOIN GROUPEUTILISATEUR T2 ON T1.CODE = T2.CODE_UTILISATEUR WHERE T2.CODE_GROUPE IN ('TEST_A') AND( T1.CODE IN('TEST_BB_1','TEST_BB_2'))
	 * 
	 * @param code
	 *            the _code
	 * @param nom
	 *            the _nom
	 * @param prenom
	 *            the _prenom
	 * @param chaineDiffusion
	 *            the _chaine diffusion
	 * @param profil
	 *            the _profil
	 * @param groupe
	 *            the _groupe
	 * @param structure
	 *            the _structure
	 * @param adresseMail
	 *            the _adresseMail
	 * 
	 * @return the int
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public int select(final String code, final String nom, final String prenom, final String chaineDiffusion, final String profil, final String groupe,
		final String structure, final String adresseMail) throws Exception {
		return select(construireRequete(code, nom, prenom, chaineDiffusion, profil, groupe, structure, adresseMail, true));
	}

	/**
	 * Sélection d'un utilisateur par son code.
	 * 
	 * @param code
	 *            the code
	 * 
	 * @return the int
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public int selectParCode(final String code) throws Exception {
		return select(code, "", "", "", "", "", "", "");
	}

	/**
	 * select nocount pour les listes avec pagination.
	 * 
	 * @param code
	 *            the _code
	 * @param nom
	 *            the _nom
	 * @param prenom
	 *            the _prenom
	 * @param chaineDiffusion
	 *            the _chaine diffusion
	 * @param profil
	 *            the _profil
	 * @param groupe
	 *            the _groupe
	 * @param from
	 *            the from
	 * @param increment
	 *            the increment
	 * @param structure
	 *            the _structure
	 * @param adresseMail
	 *            the _adresse mail
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public void selectNoCount(final String code, final String nom, final String prenom, final String chaineDiffusion, final String profil, final String groupe,
		final String structure, final String adresseMail, final int from, final int increment) throws Exception {
		String requete = construireRequete(code, nom, prenom, chaineDiffusion, profil, groupe, structure, adresseMail, true);
		if (increment != 0) {
			requete += " LIMIT " + from + "," + increment;
		}
		selectNoCount(requete);
	}

	/**
	 * Verifie le mot de passe de l'utilisateur.
	 * 
	 * @param motdepasse
	 *            the _motdepasse
	 * 
	 * @return the string
	 * 
	 * @throws Exception
	 *             the exception
	 * @deprecated les messages sont en dur, les contrôles sont tout sauf sécurisés. A ne pas utiliser.
	 * Utilisez juste les properties utilisateur.password.longueur.max et utilisateur.password.longueur.min pour contrôler la taille du mdp.
	 */
	@Deprecated
	public static String verifieMotDePasse(final String motdepasse) throws Exception {
		String restriction = StringUtils.defaultString(PropertyHelper.getCoreProperty("utilisateur.password.restriction"), "0");
		//si le mot de passe est vide ou que la restriction n'est pas activée, on sort
		if (StringUtils.isEmpty(motdepasse) || "0".equals(restriction)) {
			return StringUtils.EMPTY;
		}
		/**** Longueur du mot de passe *****/
		int iMax =  PropertyHelper.getCorePropertyAsInt("utilisateur.password.longueur.max",-1);
		int iMin = PropertyHelper.getCorePropertyAsInt("utilisateur.password.longueur.min", -1);
		if (iMax != -1 && motdepasse.length() > iMax) {
			return "Votre mot de passe ne doit pas dépasser " + iMax + " caractères.";
		}
		if (iMin != -1 && motdepasse.length() < iMin) {
			return "Votre mot de passe doit contenir au minimum " + iMin + " caractères.";
		}
		/****** Récupération de la liste des caractères autorisés ******/
		String alphanum = StringUtils.defaultString(PropertyHelper.getCoreProperty("utilisateur.password.alphanum"));
		String accents = StringUtils.defaultString(PropertyHelper.getCoreProperty("utilisateur.password.accent"));
		String speciaux = StringUtils.defaultString(PropertyHelper.getCoreProperty("utilisateur.password.special"));
		final String liste = alphanum.toLowerCase() + alphanum.toUpperCase() + accents.toLowerCase() + accents.toUpperCase() + speciaux;
		boolean ok = true;
		/* parcours du mot de passe */
		for (int i = 0; i < motdepasse.length(); i++) {
			/* si le mot de passe contient un caractère qui n'est pas présent dans la liste, on sort*/
			if (liste.indexOf(motdepasse.charAt(i)) == -1) {
				ok = false;
				break;
			}
		}
		if (!ok) {
			return "Votre mot de passe comporte des caractères non autorisés";
		}
		return "";
	}

	/**
	 * Détermine la liste des utilisateurs ayant une permission donnée sur un périmètre donné
	 * 
	 * Par défaut, renvoie les codes
	 * 
	 * Si renvoyerMails vaut true, ce ne sont pas les codes mais les mails qui sont renvoyés.
	 * 
	 * @param ctx
	 *            the _ctx
	 * @param permission
	 *            the _permission
	 * @param codesStructures
	 *            the _codes structures
	 * @param codeRubrique
	 *            the _code rubrique
	 * @param publicsVises
	 *            the _publics vises
	 * @param codeEspaceCollaboratif
	 *            the _code espace collaboratif
	 * @param renvoyerMails
	 *            the renvoyer mails
	 * 
	 * @return the liste utilisateurs possedant permission
	 * 
	 * @throws Exception
	 *             the exception
	 *
	 * @deprecated utiliser {@link Utilisateur#getListeUtilisateursPossedantPermission(PermissionBean, List, String, String, String, boolean)}
	 */
	@Deprecated
	public static TreeSet<String> getListeUtilisateursPossedantPermission(final OMContext ctx, final PermissionBean permission, final List<String> codesStructures, final String codeRubrique,
																		  final String publicsVises, final String codeEspaceCollaboratif, final boolean renvoyerMails) throws Exception {
		return getListeUtilisateursPossedantPermission(permission, codesStructures, codeRubrique, publicsVises, codeEspaceCollaboratif, renvoyerMails);
	}

	/**
	 * Récupère la liste des utilisateurs possédant la permission fourni en paramètre.
	 * @param permission la permission dont on souhaite avoir la liste des utilisateurs.
	 * @return la liste des utilisateurs possédant la permission.
	 * @throws Exception les classiques, en gros BDD généralement...
	 */
	public static List<Utilisateur> getUtilisateursPossedantPermission(final PermissionBean permission) throws Exception {
		List<Utilisateur> result = new ArrayList<>();
		final Set<String> listeRoles = new TreeSet<>();
		final Set<String> listeGroupes = new TreeSet<>();
		try (ContexteDao ctx = new ContexteDao()) {
			final Role role = new Role();
			role.init();
			role.setCtx(ctx);
			if (role.select("", "", permission.getChaineSerialisee()) == 0) {
				return result;
			}
			while (role.nextItem()) {
				listeRoles.add(role.getCode());
				/* Stockage des groupes contenant le role */
				Map<String, Vector<Perimetre>> hGroupes = Groupedsi.renvoyerGroupesEtPerimetres(role.getCode(), Collections.<String>emptyList(), StringUtils.EMPTY, StringUtils.EMPTY, StringUtils.EMPTY);
				for (String codeGroupe : hGroupes.keySet()) {
					listeGroupes.add(codeGroupe);
				}
			}
		}
		final Vector<String> codeUtilisateursGroupe = Groupeutilisateur.getVecteurUtilisateurs(StringUtils.join(listeGroupes,";"));
		for (String codeUtilisateur : codeUtilisateursGroupe) {
			final Utilisateur utilisateur = Utilisateur.getUtilisateur(codeUtilisateur);
			if (utilisateur != null && StringUtils.isNotBlank(utilisateur.getCode())) {
				result.add(utilisateur);
			}
		}
		if (CollectionUtils.isNotEmpty(listeRoles)) {
			/* 1 - sélection des utilisateur par requete sur leurs roles */
			final ClauseWhere whereRoles = new ClauseWhere();
			for (String codeRole : listeRoles) {
				whereRoles.or(ConditionHelper.like("ROLES", codeRole, "%[", ";%"));
			}
			final Utilisateur utilisateur = new Utilisateur();
			utilisateur.init();
			try (ContexteDao ctx2 = new ContexteDao()) {
				utilisateur.setCtx(ctx2);
				/* 2 - pour chaque utilisateur, analyse du périmètre de chaque role */
				if (utilisateur.select(whereRoles.formaterSQL()) > 0) {
					while (utilisateur.nextItem()) {
						result.add((Utilisateur) BeanUtils.cloneBean(utilisateur));
					}
				}
			}
		}
		return result;
	}

	/**
	 * Détermine la liste des utilisateurs ayant une permission donnée sur un périmètre donné
	 *
	 * Par défaut, renvoie les codes
	 *
	 * Si renvoyerMails vaut true, ce ne sont pas les codes mais les mails qui sont renvoyés.
	 *
	 * @param permission
	 *            the _permission
	 * @param codesStructures
	 *            the _codes structures
	 * @param codeRubrique
	 *            the _code rubrique
	 * @param publicsVises
	 *            the _publics vises
	 * @param codeEspaceCollaboratif
	 *            the _code espace collaboratif
	 * @param renvoyerMails
	 *            the renvoyer mails
	 *
	 * @return the liste utilisateurs possedant permission
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static TreeSet<String> getListeUtilisateursPossedantPermission(final PermissionBean permission, final List<String> codesStructures,
																		  final String codeRubrique, final String publicsVises, final String codeEspaceCollaboratif, final boolean renvoyerMails) throws Exception {
		if (codeEspaceCollaboratif.length() > 0) {
			return Espacecollaboratif.getListeUtilisateursPossedantPermissionPourUnEspace(permission, codeEspaceCollaboratif, renvoyerMails);
		}
		final TreeSet<String> res = new TreeSet<>();
		final Set<String> listeRoles = new TreeSet<>();
		/* Selection des roles comprenant la permission */
		final Role role = new Role();
		role.init();
		final Set<String> listeGroupes = new TreeSet<>();
		try (ContexteDao ctx = new ContexteDao()) {
			role.setCtx(ctx);
			if (role.select("", "", permission.getChaineSerialisee()) == 0) {
				return res;
			}
			/** *************************************** */
			/* PHASE 1 - ANALYSE PAR GROUPE			 */
			/** *************************************** */
			/* Selection des groupes et des profils comprenant les roles */
			while (role.nextItem()) {
				LOG.debug("getListeUtilisateursPossedantPermission role " + role.getCode());
				listeRoles.add(role.getCode());
				/* Stockage des groupes contenant le role */
				Hashtable<String, Vector<Perimetre>> hGroupes = Groupedsi.renvoyerGroupesEtPerimetres(role.getCode(), codesStructures, codeRubrique, publicsVises, codeEspaceCollaboratif);
				final Enumeration<String> e = hGroupes.keys();
				while (e.hasMoreElements()) {
					listeGroupes.add(e.nextElement());
				}
			}
		}
		/* Recherche des utilisateurs associés aux groupes */
		String codesGroupes = "";
		for (String codeGroupe : listeGroupes) {
			if (codesGroupes.length() > 0) {
				codesGroupes += ";";
			}
			codesGroupes += codeGroupe;
		}
		final Vector<String> listeUtilisateursGroupe = Groupeutilisateur.getVecteurUtilisateurs(codesGroupes);
		final Enumeration<String> e = listeUtilisateursGroupe.elements();
		while (e.hasMoreElements()) {
			final String codeUtilisateur = e.nextElement();
			if (renvoyerMails) {
				final Utilisateur ut = Utilisateur.getUtilisateur(codeUtilisateur);
				if (ut.getCode().length() > 0) {
					res.add(ut.getAdresseMail() + ";" + ut.getLibelle() + ";" + ut.getCode());
				}
			} else {
				res.add(codeUtilisateur);
			}
		}
		/** *************************************** */
		/* PHASE 2 - ANALYSE PAR UTILISATEUR */
		/** *************************************** */
		/* Construction de la requete sur les utilisateurs */
		if (!listeRoles.isEmpty()) {
			/* 1 - sélection des utilisateur par requete sur leurs roles */
			final ClauseWhere whereRoles = new ClauseWhere();
			for (String codeRole : listeRoles) {
				whereRoles.or(ConditionHelper.like("ROLES", codeRole, "%[", ";%"));
			}
			final Utilisateur utilisateur = new Utilisateur();
			utilisateur.init();
			try  (ContexteDao ctx = new ContexteDao()) {
				utilisateur.setCtx(ctx);
				/* 2 - pour chaque utilisateur, analyse du périmètre de chaque role */
				if (utilisateur.select(whereRoles.formaterSQL()) > 0) {
					while (utilisateur.nextItem()) {
						for (String listeRole : listeRoles) {
						/* Controle périmetre sur chaque role */
							final Vector<Perimetre> v = Role.renvoyerPerimetresAffectation(utilisateur.getRoles(), listeRole, codesStructures, codeRubrique, publicsVises, codeEspaceCollaboratif);
							if (!v.isEmpty()) {
								if (renvoyerMails) {
									res.add(utilisateur.getAdresseMail() + ";" + utilisateur.getLibelle() + ";" + utilisateur.getCode());
								} else {
									res.add(utilisateur.getCode());
								}
							}
						}
					}
				}
			}
		}
		return res;
	}

	/**
	 * Gets the vecteur groupes dsi tmp.
	 * 
	 * @return the vecteur groupes dsi tmp
	 */
	public Vector<String> getVecteurGroupesDsiTmp() {
		return vecteurGroupesDsiTmp;
	}

	/**
	 * Sets the vecteur groupes dsi tmp.
	 * 
	 * @param vecteurGroupesDsiTmp
	 *            the new vecteur groupes dsi tmp
	 */
	public void setVecteurGroupesDsiTmp(final Vector<String> vecteurGroupesDsiTmp) {
		this.vecteurGroupesDsiTmp = vecteurGroupesDsiTmp;
	}

	/**
	 * Retourne vrai si l'utilisateur provient d'un annuaire ldap. Basé sur l'assertion : si le code_ldap est non vide l'utilisateur provient du ldap.
	 * 
	 * @return Vrai si l'utilsateur provient du ldap
	 */
	public boolean provientDuLdap() {
		return StringUtils.isNotBlank(this.getCodeLdap());
	}

	/**
	 * retourne vrai si la source d'authenfication interdit la modification du mot de passe et que l'utilisateur ne provient pas du ldap
	 * 
	 * @return vrai si la source d'authenfication interdit la modification du mot de passe et que l'utilisateur ne provient pas du ldap
	 */
	public boolean isModificationMotDePasseInterditeParSource() {
		final boolean ldapActif = StringUtils.equals(SOURCE_AUTHENTIFICATION, SOURCE_AUTHENTIFICATION_LDAP);
		final boolean casActif = StringUtils.equals(SOURCE_AUTHENTIFICATION, SOURCE_AUTHENTIFICATION_CAS);
		final boolean cleartrustActif = StringUtils.equals(SOURCE_AUTHENTIFICATION, SOURCE_AUTHENTIFICATION_CLEARTRUST);
		final boolean utilisateurProvientDuLdap = StringUtils.isNotBlank(this.getCodeLdap());
		return utilisateurProvientDuLdap && (ldapActif || casActif || cleartrustActif);
	}
}
